
package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ContainerStats}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableContainerStats.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableContainerStats implements ContainerStats {
  private final Date read;
  private final @Nullable NetworkStats network;
  private final @Nullable Map<String, NetworkStats> networks;
  private final MemoryStats memoryStats;
  private final BlockIoStats blockIoStats;
  private final CpuStats cpuStats;
  private final CpuStats precpuStats;

  private ImmutableContainerStats(
      Date read,
      @Nullable NetworkStats network,
      @Nullable Map<String, NetworkStats> networks,
      MemoryStats memoryStats,
      BlockIoStats blockIoStats,
      CpuStats cpuStats,
      CpuStats precpuStats) {
    this.read = read;
    this.network = network;
    this.networks = networks;
    this.memoryStats = memoryStats;
    this.blockIoStats = blockIoStats;
    this.cpuStats = cpuStats;
    this.precpuStats = precpuStats;
  }

  /**
   * @return The value of the {@code read} attribute
   */
  @JsonProperty("read")
  @Override
  public Date read() {
    return read;
  }

  /**
   * @return The value of the {@code network} attribute
   */
  @JsonProperty("network")
  @Override
  public @Nullable NetworkStats network() {
    return network;
  }

  /**
   * @return The value of the {@code networks} attribute
   */
  @JsonProperty("networks")
  @Override
  public @Nullable Map<String, NetworkStats> networks() {
    return networks;
  }

  /**
   * @return The value of the {@code memoryStats} attribute
   */
  @JsonProperty("memory_stats")
  @Override
  public MemoryStats memoryStats() {
    return memoryStats;
  }

  /**
   * @return The value of the {@code blockIoStats} attribute
   */
  @JsonProperty("blkio_stats")
  @Override
  public BlockIoStats blockIoStats() {
    return blockIoStats;
  }

  /**
   * @return The value of the {@code cpuStats} attribute
   */
  @JsonProperty("cpu_stats")
  @Override
  public CpuStats cpuStats() {
    return cpuStats;
  }

  /**
   * @return The value of the {@code precpuStats} attribute
   */
  @JsonProperty("precpu_stats")
  @Override
  public CpuStats precpuStats() {
    return precpuStats;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#read() read} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for read
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withRead(Date value) {
    if (this.read == value) return this;
    Date newValue = Objects.requireNonNull(value, "read");
    return new ImmutableContainerStats(
        newValue,
        this.network,
        this.networks,
        this.memoryStats,
        this.blockIoStats,
        this.cpuStats,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#network() network} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for network (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withNetwork(@Nullable NetworkStats value) {
    if (this.network == value) return this;
    return new ImmutableContainerStats(
        this.read,
        value,
        this.networks,
        this.memoryStats,
        this.blockIoStats,
        this.cpuStats,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by replacing the {@link ContainerStats#networks() networks} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the networks map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainerStats withNetworks(@Nullable Map<String, ? extends NetworkStats> entries) {
    if (this.networks == entries) return this;
    @Nullable Map<String, NetworkStats> newValue = entries == null ? null : createUnmodifiableMap(true, false, entries);
    return new ImmutableContainerStats(
        this.read,
        this.network,
        newValue,
        this.memoryStats,
        this.blockIoStats,
        this.cpuStats,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#memoryStats() memoryStats} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for memoryStats
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withMemoryStats(MemoryStats value) {
    if (this.memoryStats == value) return this;
    MemoryStats newValue = Objects.requireNonNull(value, "memoryStats");
    return new ImmutableContainerStats(
        this.read,
        this.network,
        this.networks,
        newValue,
        this.blockIoStats,
        this.cpuStats,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#blockIoStats() blockIoStats} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for blockIoStats
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withBlockIoStats(BlockIoStats value) {
    if (this.blockIoStats == value) return this;
    BlockIoStats newValue = Objects.requireNonNull(value, "blockIoStats");
    return new ImmutableContainerStats(
        this.read,
        this.network,
        this.networks,
        this.memoryStats,
        newValue,
        this.cpuStats,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#cpuStats() cpuStats} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for cpuStats
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withCpuStats(CpuStats value) {
    if (this.cpuStats == value) return this;
    CpuStats newValue = Objects.requireNonNull(value, "cpuStats");
    return new ImmutableContainerStats(
        this.read,
        this.network,
        this.networks,
        this.memoryStats,
        this.blockIoStats,
        newValue,
        this.precpuStats);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerStats#precpuStats() precpuStats} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for precpuStats
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerStats withPrecpuStats(CpuStats value) {
    if (this.precpuStats == value) return this;
    CpuStats newValue = Objects.requireNonNull(value, "precpuStats");
    return new ImmutableContainerStats(
        this.read,
        this.network,
        this.networks,
        this.memoryStats,
        this.blockIoStats,
        this.cpuStats,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableContainerStats} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableContainerStats
        && equalsByValue((ImmutableContainerStats) another);
  }

  private boolean equalsByValue(ImmutableContainerStats another) {
    return read.equals(another.read)
        && Objects.equals(network, another.network)
        && Objects.equals(networks, another.networks)
        && memoryStats.equals(another.memoryStats)
        && blockIoStats.equals(another.blockIoStats)
        && cpuStats.equals(another.cpuStats)
        && precpuStats.equals(another.precpuStats);
  }

  /**
   * Computes a hash code from attributes: {@code read}, {@code network}, {@code networks}, {@code memoryStats}, {@code blockIoStats}, {@code cpuStats}, {@code precpuStats}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + read.hashCode();
    h += (h << 5) + Objects.hashCode(network);
    h += (h << 5) + Objects.hashCode(networks);
    h += (h << 5) + memoryStats.hashCode();
    h += (h << 5) + blockIoStats.hashCode();
    h += (h << 5) + cpuStats.hashCode();
    h += (h << 5) + precpuStats.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ContainerStats} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ContainerStats{"
        + "read=" + read
        + ", network=" + network
        + ", networks=" + networks
        + ", memoryStats=" + memoryStats
        + ", blockIoStats=" + blockIoStats
        + ", cpuStats=" + cpuStats
        + ", precpuStats=" + precpuStats
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ContainerStats} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ContainerStats instance
   */
  public static ImmutableContainerStats copyOf(ContainerStats instance) {
    if (instance instanceof ImmutableContainerStats) {
      return (ImmutableContainerStats) instance;
    }
    return ImmutableContainerStats.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableContainerStats ImmutableContainerStats}.
   * <pre>
   * ImmutableContainerStats.builder()
   *    .read(Date) // required {@link ContainerStats#read() read}
   *    .network(org.mandas.docker.client.messages.NetworkStats | null) // nullable {@link ContainerStats#network() network}
   *    .networks(Map&lt;String, org.mandas.docker.client.messages.NetworkStats&gt; | null) // nullable {@link ContainerStats#networks() networks}
   *    .memoryStats(org.mandas.docker.client.messages.MemoryStats) // required {@link ContainerStats#memoryStats() memoryStats}
   *    .blockIoStats(org.mandas.docker.client.messages.BlockIoStats) // required {@link ContainerStats#blockIoStats() blockIoStats}
   *    .cpuStats(org.mandas.docker.client.messages.CpuStats) // required {@link ContainerStats#cpuStats() cpuStats}
   *    .precpuStats(org.mandas.docker.client.messages.CpuStats) // required {@link ContainerStats#precpuStats() precpuStats}
   *    .build();
   * </pre>
   * @return A new ImmutableContainerStats builder
   */
  public static ImmutableContainerStats.Builder builder() {
    return new ImmutableContainerStats.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableContainerStats ImmutableContainerStats}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_READ = 0x1L;
    private static final long INIT_BIT_MEMORY_STATS = 0x2L;
    private static final long INIT_BIT_BLOCK_IO_STATS = 0x4L;
    private static final long INIT_BIT_CPU_STATS = 0x8L;
    private static final long INIT_BIT_PRECPU_STATS = 0x10L;
    private long initBits = 0x1fL;

    private Date read;
    private NetworkStats network;
    private Map<String, NetworkStats> networks = null;
    private MemoryStats memoryStats;
    private BlockIoStats blockIoStats;
    private CpuStats cpuStats;
    private CpuStats precpuStats;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ContainerStats} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ContainerStats instance) {
      Objects.requireNonNull(instance, "instance");
      this.read(instance.read());
      @Nullable NetworkStats networkValue = instance.network();
      if (networkValue != null) {
        network(networkValue);
      }
      @Nullable Map<String, NetworkStats> networksValue = instance.networks();
      if (networksValue != null) {
        putAllNetworks(networksValue);
      }
      this.memoryStats(instance.memoryStats());
      this.blockIoStats(instance.blockIoStats());
      this.cpuStats(instance.cpuStats());
      this.precpuStats(instance.precpuStats());
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#read() read} attribute.
     * @param read The value for read 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("read")
    public final Builder read(Date read) {
      this.read = Objects.requireNonNull(read, "read");
      initBits &= ~INIT_BIT_READ;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#network() network} attribute.
     * @param network The value for network (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("network")
    public final Builder network(@Nullable NetworkStats network) {
      this.network = network;
      return this;
    }

    /**
     * Put one entry to the {@link ContainerStats#networks() networks} map.
     * @param key The key in the networks map
     * @param value The associated value in the networks map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addNetwork(String key, NetworkStats value) {
      if (this.networks == null) {
        this.networks = new LinkedHashMap<String, NetworkStats>();
      }
      this.networks.put(
          Objects.requireNonNull(key, "networks key"),
          Objects.requireNonNull(value, value == null ? "networks value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link ContainerStats#networks() networks} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addNetwork(Map.Entry<String, ? extends NetworkStats> entry) {
      if (this.networks == null) {
        this.networks = new LinkedHashMap<String, NetworkStats>();
      }
      String k = entry.getKey();
      NetworkStats v = entry.getValue();
      this.networks.put(
          Objects.requireNonNull(k, "networks key"),
          Objects.requireNonNull(v, v == null ? "networks value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ContainerStats#networks() networks} map. Nulls are not permitted as keys or values, but parameter itself can be null
     * @param entries The entries that will be added to the networks map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("networks")
    public final Builder networks(@Nullable Map<String, ? extends NetworkStats> entries) {
      if (entries == null) {
        this.networks = null;
        return this;
      }
      this.networks = new LinkedHashMap<String, NetworkStats>();
      return putAllNetworks(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ContainerStats#networks() networks} map. Nulls are not permitted
     * @param entries The entries that will be added to the networks map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllNetworks(Map<String, ? extends NetworkStats> entries) {
      if (this.networks == null) {
        this.networks = new LinkedHashMap<String, NetworkStats>();
      }
      for (Map.Entry<String, ? extends NetworkStats> e : entries.entrySet()) {
        String k = e.getKey();
        NetworkStats v = e.getValue();
        this.networks.put(
            Objects.requireNonNull(k, "networks key"),
            Objects.requireNonNull(v, v == null ? "networks value for key: " + k : null));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#memoryStats() memoryStats} attribute.
     * @param memoryStats The value for memoryStats 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("memory_stats")
    public final Builder memoryStats(MemoryStats memoryStats) {
      this.memoryStats = Objects.requireNonNull(memoryStats, "memoryStats");
      initBits &= ~INIT_BIT_MEMORY_STATS;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#blockIoStats() blockIoStats} attribute.
     * @param blockIoStats The value for blockIoStats 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("blkio_stats")
    public final Builder blockIoStats(BlockIoStats blockIoStats) {
      this.blockIoStats = Objects.requireNonNull(blockIoStats, "blockIoStats");
      initBits &= ~INIT_BIT_BLOCK_IO_STATS;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#cpuStats() cpuStats} attribute.
     * @param cpuStats The value for cpuStats 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("cpu_stats")
    public final Builder cpuStats(CpuStats cpuStats) {
      this.cpuStats = Objects.requireNonNull(cpuStats, "cpuStats");
      initBits &= ~INIT_BIT_CPU_STATS;
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerStats#precpuStats() precpuStats} attribute.
     * @param precpuStats The value for precpuStats 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("precpu_stats")
    public final Builder precpuStats(CpuStats precpuStats) {
      this.precpuStats = Objects.requireNonNull(precpuStats, "precpuStats");
      initBits &= ~INIT_BIT_PRECPU_STATS;
      return this;
    }

    /**
     * Builds a new {@link ImmutableContainerStats ImmutableContainerStats}.
     * @return An immutable instance of ContainerStats
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableContainerStats build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableContainerStats(
          read,
          network,
          networks == null ? null : createUnmodifiableMap(false, false, networks),
          memoryStats,
          blockIoStats,
          cpuStats,
          precpuStats);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_READ) != 0) attributes.add("read");
      if ((initBits & INIT_BIT_MEMORY_STATS) != 0) attributes.add("memoryStats");
      if ((initBits & INIT_BIT_BLOCK_IO_STATS) != 0) attributes.add("blockIoStats");
      if ((initBits & INIT_BIT_CPU_STATS) != 0) attributes.add("cpuStats");
      if ((initBits & INIT_BIT_PRECPU_STATS) != 0) attributes.add("precpuStats");
      return "Cannot build ContainerStats, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
