/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.smarthome.storage.json.internal;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonIOException;
import com.google.gson.JsonSyntaxException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.eclipse.smarthome.config.core.Configuration;
import org.eclipse.smarthome.config.core.ConfigurationDeserializer;
import org.eclipse.smarthome.core.storage.Storage;
import org.eclipse.smarthome.storage.json.internal.StorageEntry;
import org.eclipse.smarthome.storage.json.internal.StorageEntryMapDeserializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class JsonStorage<T>
implements Storage<T> {
    private final Logger logger = LoggerFactory.getLogger(JsonStorage.class);
    private final int maxBackupFiles;
    private final int writeDelay;
    private final int maxDeferredPeriod;
    static final String CLASS = "class";
    static final String VALUE = "value";
    private final String BACKUP_EXTENSION = "backup";
    private final String SEPARATOR = "--";
    private final Timer commitTimer;
    private @Nullable TimerTask commitTimerTask = null;
    private long deferredSince = 0L;
    private final File file;
    private final ClassLoader classLoader;
    private final Map<String, StorageEntry> map = new ConcurrentHashMap<String, StorageEntry>();
    private transient Gson internalMapper;
    private transient Gson entityMapper;
    private boolean dirty = false;

    public JsonStorage(File file, ClassLoader classLoader, int maxBackupFiles, int writeDelay, int maxDeferredPeriod) {
        this.file = file;
        this.classLoader = classLoader;
        this.maxBackupFiles = maxBackupFiles;
        this.writeDelay = writeDelay;
        this.maxDeferredPeriod = maxDeferredPeriod;
        this.internalMapper = new GsonBuilder().registerTypeHierarchyAdapter(Map.class, (Object)new StorageEntryMapDeserializer()).setPrettyPrinting().create();
        this.entityMapper = new GsonBuilder().registerTypeAdapter(Configuration.class, (Object)new ConfigurationDeserializer()).setPrettyPrinting().create();
        this.commitTimer = new Timer();
        Map<String, StorageEntry> inputMap = null;
        if (file.exists()) {
            inputMap = this.readDatabase(file);
        }
        if (inputMap == null) {
            if (file.exists()) {
                this.logger.info("Json storage file at '{}' seems to be corrupt - checking for a backup.", (Object)file.getAbsolutePath());
            } else {
                this.logger.debug("Json storage file at '{}' does not exist - checking for a backup.", (Object)file.getAbsolutePath());
            }
            int cnt = 1;
            while (cnt <= maxBackupFiles) {
                File backupFile = this.getBackupFile(cnt);
                if (backupFile == null) break;
                inputMap = this.readDatabase(backupFile);
                if (inputMap != null) {
                    this.logger.info("Json storage file at '{}' is used (backup {}).", (Object)backupFile.getAbsolutePath(), (Object)cnt);
                    break;
                }
                ++cnt;
            }
        }
        if (inputMap != null) {
            this.map.putAll(inputMap);
            this.logger.debug("Opened Json storage file at '{}'.", (Object)file.getAbsolutePath());
        }
    }

    public @Nullable T put(String key, @Nullable T value) {
        if (value == null) {
            return this.remove(key);
        }
        StorageEntry val = new StorageEntry(value.getClass().getName(), this.entityMapper.toJsonTree(value));
        StorageEntry previousValue = this.map.put(key, val);
        this.deferredCommit();
        if (previousValue == null) {
            return null;
        }
        return this.deserialize(previousValue);
    }

    public @Nullable T remove(String key) {
        StorageEntry removedElement = this.map.remove(key);
        this.deferredCommit();
        if (removedElement == null) {
            return null;
        }
        return this.deserialize(removedElement);
    }

    public boolean containsKey(String key) {
        return this.map.containsKey(key);
    }

    public @Nullable T get(String key) {
        StorageEntry value = this.map.get(key);
        if (value == null) {
            return null;
        }
        return this.deserialize(value);
    }

    public Collection<String> getKeys() {
        return this.map.keySet();
    }

    public Collection<@Nullable T> getValues() {
        ArrayList<@Nullable T> values = new ArrayList<T>();
        for (String key : this.getKeys()) {
            values.add(this.get(key));
        }
        return values;
    }

    private @Nullable T deserialize(@Nullable StorageEntry entry) {
        if (entry == null) {
            return null;
        }
        @Nullable Object value = null;
        try {
            Class<?> loadedValueType = null;
            loadedValueType = this.classLoader == null ? Class.forName(entry.getEntityClassName()) : this.classLoader.loadClass(entry.getEntityClassName());
            value = this.entityMapper.fromJson((JsonElement)entry.getValue(), loadedValueType);
            this.logger.trace("deserialized value '{}' from Json", value);
        }
        catch (Exception e) {
            this.logger.error("Couldn't deserialize value '{}'. Root cause is: {}", (Object)entry, (Object)e.getMessage());
        }
        return (T)value;
    }

    private @Nullable Map<String, StorageEntry> readDatabase(File inputFile) {
        try {
            ConcurrentHashMap<String, StorageEntry> inputMap = new ConcurrentHashMap<String, StorageEntry>();
            FileReader reader = new FileReader(inputFile);
            Map loadedMap = (Map)this.internalMapper.fromJson((Reader)reader, this.map.getClass());
            if (loadedMap != null && loadedMap.size() != 0) {
                inputMap.putAll(loadedMap);
            }
            return inputMap;
        }
        catch (JsonIOException | JsonSyntaxException | FileNotFoundException e) {
            this.logger.error("Error reading JsonDB from {}. Cause {}.", (Object)inputFile.getPath(), (Object)e.getMessage());
            return null;
        }
    }

    private @Nullable File getBackupFile(int age) {
        List<Long> fileTimes = this.calculateFileTimes();
        if (fileTimes.size() < age) {
            return null;
        }
        return new File(String.valueOf(this.file.getParent()) + File.separator + "backup", fileTimes.get(fileTimes.size() - age) + "--" + this.file.getName());
    }

    private List<Long> calculateFileTimes() {
        File folder = new File(String.valueOf(this.file.getParent()) + File.separator + "backup");
        if (!folder.isDirectory()) {
            return Collections.emptyList();
        }
        ArrayList<Long> fileTimes = new ArrayList<Long>();
        File[] files = folder.listFiles();
        if (files != null) {
            int count = files.length;
            int i = 0;
            while (i < count) {
                String[] parts;
                if (files[i].isFile() && (parts = files[i].getName().split("--")).length == 2 && parts[1].equals(this.file.getName())) {
                    long time = Long.parseLong(parts[0]);
                    fileTimes.add(time);
                }
                ++i;
            }
        }
        Collections.sort(fileTimes);
        return fileTimes;
    }

    private void writeDatabaseFile(File dataFile, String data) {
        try {
            Throwable throwable = null;
            Object var4_6 = null;
            try (FileOutputStream outputStream = new FileOutputStream(dataFile, false);){
                outputStream.write(data.getBytes());
                outputStream.flush();
            }
            catch (Throwable throwable2) {
                if (throwable == null) {
                    throwable = throwable2;
                } else if (throwable != throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        catch (Exception e) {
            this.logger.error("Error writing JsonDB to {}. Cause {}.", (Object)dataFile.getPath(), (Object)e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void flush() {
        if (this.commitTimerTask != null) {
            this.commitTimerTask.cancel();
            this.commitTimerTask = null;
        }
        if (this.dirty) {
            String json = this.internalMapper.toJson(this.map);
            Map<String, StorageEntry> map = this.map;
            synchronized (map) {
                this.writeDatabaseFile(this.file, json);
                this.writeDatabaseFile(new File(String.valueOf(this.file.getParent()) + File.separator + "backup", String.valueOf(System.currentTimeMillis()) + "--" + this.file.getName()), json);
                this.cleanupBackups();
                this.deferredSince = 0L;
                this.dirty = false;
            }
        }
    }

    private void cleanupBackups() {
        List<Long> fileTimes = this.calculateFileTimes();
        if (fileTimes.size() > this.maxBackupFiles) {
            int counter = 0;
            while (counter < fileTimes.size() - this.maxBackupFiles) {
                File deleter = new File(String.valueOf(this.file.getParent()) + File.separator + "backup", fileTimes.get(counter) + "--" + this.file.getName());
                deleter.delete();
                ++counter;
            }
        }
    }

    public synchronized void deferredCommit() {
        this.dirty = true;
        if (this.commitTimerTask != null) {
            this.commitTimerTask.cancel();
            this.commitTimerTask = null;
        }
        if (this.deferredSince != 0L && this.deferredSince < System.nanoTime() - (long)this.maxDeferredPeriod * 1000L) {
            this.flush();
            return;
        }
        if (this.deferredSince == 0L) {
            this.deferredSince = System.nanoTime();
        }
        this.commitTimerTask = new CommitTimerTask();
        this.commitTimer.schedule(this.commitTimerTask, this.writeDelay);
    }

    private class CommitTimerTask
    extends TimerTask {
        private CommitTimerTask() {
        }

        @Override
        public void run() {
            JsonStorage.this.flush();
        }
    }
}

