/* PSPP - a program for statistical analysis.
   Copyright (C) 2009, 2011 Free Software Foundation, Inc.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. */

#ifndef OUTPUT_CHART_ITEM_H
#define OUTPUT_CHART_ITEM_H 1

/* Chart items.

   A chart item is a subclass of an output item (see output/output-item.h).

   A chart item is abstract.  Every actual chart is a subclass of
   chart_item. */

#include <stdbool.h>
#include "output/output-item.h"

/* A chart item.

   The members of struct chart_item should not be accessed directly.  Use one
   of the accessor functions defined below. */
struct chart_item
  {
    struct output_item output_item; /* Superclass */
    const struct chart_item_class *class; /* Subclass. */
    char *title;                /* May be null if there is no title. */
  };

const char *chart_item_get_title (const struct chart_item *);
void chart_item_set_title (struct chart_item *, const char *);

/* This boilerplate for chart_item, a subclass of output_item, was
   autogenerated by mk-class-boilerplate. */

#include <assert.h>
#include "libpspp/cast.h"

extern const struct output_item_class chart_item_class;

/* Returns true if SUPER is a chart_item, otherwise false. */
static inline bool
is_chart_item (const struct output_item *super)
{
  return super->class == &chart_item_class;
}

/* Returns SUPER converted to chart_item.  SUPER must be a chart_item, as
   reported by is_chart_item. */
static inline struct chart_item *
to_chart_item (const struct output_item *super)
{
  assert (is_chart_item (super));
  return UP_CAST (super, struct chart_item, output_item);
}

/* Returns INSTANCE converted to output_item. */
static inline struct output_item *
chart_item_super (const struct chart_item *instance)
{
  return CONST_CAST (struct output_item *, &instance->output_item);
}

/* Increments INSTANCE's reference count and returns INSTANCE. */
static inline struct chart_item *
chart_item_ref (const struct chart_item *instance)
{
  return to_chart_item (output_item_ref (&instance->output_item));
}

/* Decrements INSTANCE's reference count, then destroys INSTANCE if
   the reference count is now zero. */
static inline void
chart_item_unref (struct chart_item *instance)
{
  output_item_unref (&instance->output_item);
}

/* Returns true if INSTANCE's reference count is greater than 1,
   false otherwise. */
static inline bool
chart_item_is_shared (const struct chart_item *instance)
{
  return output_item_is_shared (&instance->output_item);
}

void chart_item_submit (struct chart_item *);

#endif /* output/chart-item.h */
