/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System;
using System.Collections;
using System.Collections.Generic;
using ICSharpCode.SharpZipLib.Zip;

namespace LocaleInpector
{

internal sealed class XpiList
{

// input
// pathInput = scrapbook.xpi

// output
// pathInput =
// return arr[] "scrapbook.xpi|chrome/scrapbook.jar|locale/en-US/calculate.dtd"	string

private readonly string       m_addr;
private readonly List<string> m_fileList;

public XpiList(string res)
{
    m_addr = res; // foo.xpi:pt-BR
    m_fileList = this.GetFileList();
}

// scrapbook.xpi|chrome/scrapbook.jar|locale/en-US/captureURL.dtd
//                                ==> locale/en-US/captureURL.dtd

public bool IsDir
{
    get {
        return BPIO.IsDir(m_addr);
    }
}

public bool Exists
{
    get {
        if (BPIO.IsDir(m_addr)) {
            return true;
        }

        string[] d = this.FileLoc();
        if (BPIO.IsFile(d[0])) {
            return true;
        }

        return false;
    }
}

public string Path
{
    get {
        return FileLoc()[0];
    }
}

public string Locale
{
    get {
        return FileLoc()[1];
    }
}

public int Count
{
    get {
        return m_fileList.Count;
    }
}

public string GetFile(int idx)
{
    return m_fileList[idx];
}

public string GetFile(string path)
{
	// [0] "flashgot.xpi|chrome/flashgot.jar|locale/pt-BR/flashgot/contents.rdf"
    string[] path2 = PoEntry.FormatRef(path).Split('/');
    path2[1] = this.Locale.Length == 0 ? "en-US" : this.Locale;
    path = string.Join("/", path2);

    for (int idx = 0; idx < this.Count; idx++) {
        if (PoEntry.FormatRef(m_fileList[idx]) == path) {
            return m_fileList[idx];
        }
    }
    return null;
}

public string[] FileLoc()
{
    string[] arr = m_addr.Split('?'); // c:\file.xpi?ab-CD
    string pathArchive;
    string locale;
    if (arr.Length > 1) {
        pathArchive = arr[0];
        locale = arr[1];
    } else {
        pathArchive = m_addr;
        locale = string.Empty;
    }
    return new string[] { pathArchive, locale };
}

private List<string> GetFileList()
{
    List<string> fileList;
    if (BPIO.IsDir(this.Path)) {
        fileList = BPIO.GetFileList(this.Path, "*");

    } else if (BPIO.IsFile(this.Path)) {
        fileList = this.GetLocale();
        //pathInput = string.Empty;

    } else {
        fileList = new List<string>();
    }
    return fileList;
}

private List<string> GetLocale()
{
    string pathArchive = this.Path;
    string findLocale = this.Locale;

    ZipFile zip = null;
    try {
        System.IO.FileStream stream = System.IO.File.OpenRead(pathArchive);
        zip = new ZipFile(stream);
    } catch {
    }

    List<string> localeFiles;

    localeFiles = FindLocaleFiles(zip, findLocale);
    if (localeFiles.Count > 0) {
        zip.Close();
        for (int idx = 0; idx < localeFiles.Count; idx++) {
            localeFiles[idx] = pathArchive + "|" + localeFiles[idx];
        }
        return localeFiles;
    }

    string pathJar = FindChromeJar(zip);
    if (pathJar == null) {
        zip.Close();
        return localeFiles;
    }

    System.IO.Stream streamJar = zip.GetInputStream(zip.FindEntry(pathJar, false));
    ZipFile zipJar = new ZipFile(BPIO.ExtractZip(streamJar));
    zip.Close();

    localeFiles = FindLocaleFiles(zipJar, findLocale);
    if (localeFiles.Count > 0) {
        zipJar.Close();
        for (int idx = 0; idx < localeFiles.Count; idx++) {
            localeFiles[idx] = pathArchive + "|" + pathJar + "|" + localeFiles[idx];
        }
        return localeFiles;
    }

    zipJar.Close();
    return localeFiles;
}

private static string FindChromeJar(ZipFile zip)
{
    IEnumerator entries = zip.GetEnumerator();

    // find locale/en-US/foo.bar.dtd
    while (entries.MoveNext()) {
        ZipEntry entryZip = entries.Current as ZipEntry;
        string[] dirs = entryZip.Name.Split('/');
        if (dirs.Length != 2) {
            continue;
        }
        if (dirs[0] != "chrome") {
            continue;
        }
        if (dirs[1].EndsWith(".jar") == false) {
            continue;
        }
        return entryZip.Name;
    }
    return null;
}

private static List<string> FindLocaleFiles(ZipFile zip, string findLocale)
{
    List<string> allLocales = GetAllLocaleFiles(zip);
    if (allLocales.Count == 0) {
        return allLocales;
    }
    List<string> theLocale = new List<string>();

    if (findLocale.Length > 0) {
        FindLocaleFiles2(allLocales, findLocale, theLocale);
        return theLocale;
    }

    FindLocaleFiles2(allLocales, "en-US", theLocale);
    if (theLocale.Count > 0) {
        return theLocale;
    }
    FindLocaleFiles2(allLocales, "en", theLocale);
    if (theLocale.Count > 0) {
        return theLocale;
    }
    string[] arr = allLocales[0].Split('/');
    FindLocaleFiles2(allLocales, arr[1], theLocale);
    if (theLocale.Count > 0) {
        return theLocale;
    }
    return theLocale;
}

//               locale/en-US/foo.bar.dtd
// chrome/foobar/locale/en-US/foo.bar.dtd
private static void FindLocaleFiles2(List<string> allLocales, string locale, List<string> theLocale)
{
    theLocale.Clear();

    locale = "locale/" + locale + "/";
    for (int idx = 0; idx < allLocales.Count; idx++) {
        if (allLocales[idx].StartsWith(locale)) {
            theLocale.Add(allLocales[idx]);
        }
    }

    if (theLocale.Count > 0) {
        return;
    }

    locale = "/" + locale;
    for (int idx = 0; idx < allLocales.Count; idx++) {
        if (allLocales[idx].IndexOf(locale) > -1) {
            theLocale.Add(allLocales[idx]);
        }
    }
}

private static List<string> GetAllLocaleFiles(ZipFile zip)
{
    List<string> allLocales = new List<string>();
    IEnumerator entries = zip.GetEnumerator();

    // find locale/en-US/foo.bar.dtd
    while (entries.MoveNext()) {
        ZipEntry entryZip = entries.Current as ZipEntry;
        if (entryZip.IsDirectory) {
            continue;
        }
        if (entryZip.Name.StartsWith("locale/")) {
            allLocales.Add(entryZip.Name);
        }
    }

    if (allLocales.Count > 0) {
        return allLocales;
    }

    // find chrome/foobar/locale/en-US/foo.bar.dtd
    entries.Reset();
    while (entries.MoveNext()) {
        ZipEntry entryZip = entries.Current as ZipEntry;
        if (entryZip.IsDirectory) {
            continue;
        }
        string[] dirs = entryZip.Name.Split('/');
        if (dirs.Length < 4) {
            continue;
        }
        if (dirs[0] != "chrome") {
            continue;
        }

        if (dirs[2] != "locale") {
            continue;
        }

        allLocales.Add(entryZip.Name);
    }

    return allLocales;
}

public static string LoadTextFromArchive(string fullName, System.Text.Encoding enc)
{
    string[] arr = fullName.Split('|');

    // arr[0] = foo.jar
    // arr[1] = chrome/foo.jar
    // arr[2] = locale/en-US/bar.dtd
    if (BPIO.IsFile(arr[0]) == false) {
        return string.Empty;
    }

    if (arr.Length == 2) {

        ZipFile jar = null;
        jar = new ZipFile(arr[0]);

        string txt;
        int idx = jar.FindEntry(arr[1], false);
        if (idx == -1) {
            txt = string.Empty;
        } else {
            System.IO.StreamReader sreader = new System.IO.StreamReader(jar.GetInputStream(idx), enc);
            txt = sreader.ReadToEnd();
            sreader.Close();
        }
        jar.Close();
        return txt;

    } else if (arr.Length == 3) {

        ZipFile jar = null;
        jar = new ZipFile(arr[0]);

        string txt;
        int idx = jar.FindEntry(arr[1], false);
        if (idx == -1) {
            txt = string.Empty;
        } else {
            ZipFile jar2 = new ZipFile(BPIO.ExtractZip(jar.GetInputStream(idx)));

            int idx2 = jar2.FindEntry(arr[2], false);
            if (idx2 == -1) {
                txt = string.Empty;
            } else {
                System.IO.StreamReader sreader = new System.IO.StreamReader(jar2.GetInputStream(idx2), enc);
                txt = sreader.ReadToEnd();
                sreader.Close();
            }
            jar2.Close();
        }
        jar.Close();
        return txt;

    }

    return string.Empty;
}

}//class
}//ns
