/*\
||| Written By Fredrik Hbinette <hubbe@lysator.liu.se>
||| and Louis Bavoil <bavoil@enseirb.fr>
||| All rights reserved. No warrenties, use at your own risk.
\*/

#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <sysexits.h>
#include <signal.h>
#include <stdarg.h>
#include <npapi.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <errno.h>
#include <sys/time.h>
#include <X11/X.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <sys/socket.h>

#include "mozplugger.h"

#define WINDOW ((Window) windata.window)

Display *display=0;
char *displayname=0;
NPWindow windata;
int flags;
int repeats;
char *winname;
char *command;
char *file;
char *mimetype;

#define FEND(X,Y) (strchr(X,Y)?strchr(X,Y):(X)+strlen(X))
#define xalloc malloc
#define xfree free

char *my_strstr(char *s1,const char *s2)
{
  for (;*s1;s1++)
  {
    const char *p1,*p2;
    for (p1=s1,p2=s2;*p2;p1++,p2++)
      if (*p2!=*p1) break;
    if (!*p2) return s1;
  }
  return NULL;
}

/*
 * Functions below are called by the child process
 */

static int childpid=-1;

void kill_child(int tmp)
{
  D("GOT SIGTERM\n");
  if (childpid != -1)
    kill(childpid, SIGTERM);
  exit(0);
}

int error_handler(Display *dpy, XErrorEvent *err)
{
  D("!!!ERROR_HANDLER!!!\n");
  return 0;
}

#define MAX_IGNORED 1000
static int windows_found=0;
static Window ignored[MAX_IGNORED];

int inital_win_cb(Window w, int depth)
{
  if (windows_found < MAX_IGNORED)
    ignored[windows_found++]=w;
  return 0;
}

int win_cb(Window w, int depth)
{
  int q;
  for (q=0;q<windows_found;q++)
    if (ignored[q]==w)
      return 0;
  return 1;
}

static Window winrecur(Window from,
		       int depth,
		       char *name,
		       int (*cb)(Window,int))
{
  Window root, parent;
  Window *children=NULL;
  int e,num_children;
  Window found=-1;
  
  D("Winrecur, checking window %x (depth=%d)\n",from,depth);

  if (!XQueryTree(display, from, &root, &parent, &children, &num_children))
  {
    D("Querytree failed!!!\n");
    return (Window)0;
  }

  D("Num children = %d\n",num_children);

  for (e=0;e<num_children;e++)
  {
    char *windowname;
    unsigned long nitems=0, bytes=0; 
    unsigned char *property=0;
    int fmt;
    Atom type;
    XWindowAttributes a;

    D("XGetWindowAttributes\n");
    if (!XGetWindowAttributes(display, children[e], &a))
      continue;
    if (a.map_state != IsViewable) continue;

    D("XFetchName\n");
    if (0 != XFetchName(display, children[e], &windowname))
    {
      int yes=0;
      D("Winrecur, checking window NAME %x (%s != %s)\n",children[e],windowname,name);
   
      if (!name || !strcmp(windowname, name))
	yes=cb(children[e],depth);
      XFree(windowname);
      if (yes) found = children[e];
    }

    /* Return the last window found */
    if (found != -1) return found;

    D("XGetTextProperty\n");
    property=0;
    XGetWindowProperty(display, children[e], XA_WM_CLASS,
		       0, 10000,
		       0,
		       XA_STRING,
		       &type, &fmt, &nitems, &bytes,
		       &property);

    if (property)
    {
      if (type == XA_STRING && fmt==8)
      {
	int len=nitems;
	char *n=(char *)property;
	while(len)
	{
	  D("Winrecur, checking window CLASS %x (%s != %s)\n",children[e],n,name);
	  if (!name || !strcmp(n, name))
	    if (cb(children[e],depth))
	      found= children[e];
	  
	  len-=strlen(n)+1;
	  n+=strlen(n)+1;
	}
      }
      XFree(property);
    }
  }
  if (found != -1) return found;
  if (depth>0)
  {
    for (e=0;e<num_children;e++)
    {
      Window ret=winrecur(children[e], depth-1, name, cb);
      if (ret) found = ret;
    }
  }
  if (found != -1) return found;
  XFree(children);
  return (Window)0;
}


static int gcd(int a, int b)
{
  if (a < b) return gcd(b,a);
  if (b == 0) return a;
  return gcd(b, a % b);
}

static XWindowAttributes wattr;
static int xaspect;
static int yaspect;
static Window victim=0;
static Window old_parent=0;
static Atom lock_property;

static void place_window(void)
{
  int x;
  Window parent=WINDOW;
  int w=windata.width;
  int h=windata.height;

  if (old_parent != parent)
  {
    XWithdrawWindow(display, victim,
		    XScreenNumberOfScreen (wattr.screen));
    XSync (display, FALSE);

    if (!old_parent)
    {
      XWMHints *leader_change;
      D("Changing leader of window %x\n",victim);
      if ((leader_change = XGetWMHints(display, victim)))
      {
	leader_change->flags = (leader_change->flags | WindowGroupHint);
	leader_change->window_group = wattr.root;
	
	XSetWMHints(display,victim,leader_change);
	XFree(leader_change);
      }
    }

    for (x=0;x<25;x++)
    {
      D("Reparenting window %x into %x\n",victim,WINDOW);
      XReparentWindow (display, victim, WINDOW, 0, 0);
      XSync (display, FALSE);
    }
  }

  if (flags & H_FILL)
  {
    D("Resizing window %x with FILL\n",victim);
    D("New size: %dx%d+%d+%d (of %dx%d)\n",w,h,
      0,0,
      w,h);
    
    XMoveResizeWindow(display, victim,
		      0,0,
		      w,h);
  }
  else if (flags & H_MAXASPECT)
  {
    int tmpw, tmph;
    D("Resizing window %x with MAXASPECT\n",victim);
    if (xaspect && yaspect)
    {
      tmph=h/yaspect;
      tmpw=w/xaspect;
      if (tmpw<tmph) tmph=tmpw;
      tmpw=tmph*xaspect;
      tmph=tmph*yaspect;
      
      D("New size: %dx%d+%d+%d (of %dx%d)\n",tmpw,tmph,
	(w - tmpw)/2,
	(h - tmph)/2,
	w,h);
      XMoveResizeWindow(display, victim,
			(w - tmpw)/2,
			(h - tmph)/2,
			tmpw, 
			tmph);
    }
  }

  XMapWindow (display, victim);
  
  XSync (display, FALSE);
  D("placewindow done\n");
}

/* X11 locking mechanism */
static void my_xlockwindow(Display *dpy,
			   Window win,
			   Atom prop)
{
  char buffer[1024];
  struct timeval tv, tv2;

  gettimeofday(&tv, 0);
  gethostname(buffer,sizeof(buffer)-1);
  sprintf(buffer+strlen(buffer),"-%d-%ld:%ld!",
	  getpid(),(long)tv.tv_sec, (long)tv.tv_usec);

  tv.tv_sec+=8; /* Max lock wait time, 8 seconds */
  D("Attempting to lock property = %d!\n",prop);
  D("My LOCKID='%s'\n",buffer);
  while(1)
  {
    int mode=PropModeAppend;
    int fmt;
    unsigned long nitems=0, bytes=0;
    unsigned char *property=0;
    Atom type;

    while(1)
    {
      property=0;
      XGetWindowProperty(dpy, win, prop,
			  0, 1,
			  0, /* Delete */
			  XA_STRING,
			  &type, &fmt, &nitems, &bytes,
			  &property);
      if (!property) break;
      D("Property already set, bytes_left=%d, waiting..\n",bytes);
      XFree(property);
      gettimeofday(&tv2, 0);
      if (tv2.tv_sec > tv.tv_sec)
      {
	D("Stealing lock....\n");
	/* Time to steal the lock! */
	mode=PropModeReplace;
	gettimeofday(&tv, 0);
	INC_TIME(&tv, 0, 10000 + (rand() & 16383));
	break;
      }
      usleep((rand() & 16383) + 2000); /* 0.18383 seconds */
    }
    
    D("XChangeProperty, mode=%d\n",mode);
    XChangeProperty(dpy, win, prop, XA_STRING, 8,
		    mode, buffer, strlen(buffer));

    D("Getting property, did we get the lock?\n");
    property=0;
    XGetWindowProperty(dpy, win, prop,
		       0, sizeof(buffer),
		       0, /* Delete */
		       XA_STRING,
		       &type, &fmt, &nitems, &bytes,
		       &property);
    if (property)
    {
      int done=!strncmp(buffer,property,strlen(buffer));
      D("LOCK: done=%d, bytes_left=%d\n",done,bytes);
      XFree(property);
      if (done) return;
    }
  }
}

static void my_xunlockwindow(Display *dpy,
			     Window win,
			     Atom prop)
{
  D("!!!UNLOCKING!!!\n");
  XDeleteProperty(dpy, win, prop);
}


static int got_sigwinch=0;

static void sigwinch(int tmp)
{
//  D("---SIGWINCH---\n");
  got_sigwinch=1;
}

static void setup_swallow()
{
  D("Setting up X data (%s)\n",displayname);

  if (!display)
    {
      display = XOpenDisplay(displayname);
      XSetErrorHandler(error_handler);
    }
  D("display=%x\n",display);

  if (display && XGetWindowAttributes(display, WINDOW, &wattr))
    {
      victim=0;
      old_parent=0;
      xaspect=yaspect=0;
      D("rootwin=%x\n",wattr.root);
      lock_property=XInternAtom(display, "PLUGGER_SWALLOW_LOCK", 0);
      D("lock_property=%x\n",lock_property);
      
      my_xlockwindow(display, wattr.root, lock_property);
      
      windows_found=0;
      winrecur(wattr.root, 3, winname, inital_win_cb);
      
      D("Done Setting up X data\n");
    }
  else
    {
      D("XGetWIndowAttributes failed!!!!\n");
      flags &=~ H_SWALLOW;
    }
}

static void run_app(char **argv)
{
#ifdef H_NOISY
  int nl;
  if (flags & H_NOISY)
  {
    D("Redirecting stdout and stderr\n");
    if ((nl = open("/dev/null", O_RDONLY)) == -1)
      exit(EX_UNAVAILABLE);
    dup2(nl,1);
    dup2(nl,2);
    close(nl);
  }
#endif
  
  execvp(argv[0],argv);
  D("Execvp failed..%d\n",errno);
  exit(EX_UNAVAILABLE);
}

static void handle_app(int pid)
{
  int waitpid_done=0;
  int unlock_done=0;
  int status;

  if (!(flags & H_DAEMON))
    childpid=pid;
  
  if (flags & H_SWALLOW)
    {
      int tmp;
      for (tmp=0; tmp<500; tmp++) /* approx 10 seconds */
	{
	  if (!(flags & H_EXITS) && waitpid(pid, &status, WNOHANG) > 0)
	    {
	      waitpid_done=1;
	      break;
	    }
	  
	  D("Looking for victim... (%s)\n",winname);
	  
	  if ((victim=winrecur(wattr.root, 3, winname, win_cb)))
	    {
	      my_xunlockwindow(display, wattr.root, lock_property);
	      unlock_done=1;
	      
	      D("Found it!! Victim=%x\n",victim);
	      if (flags & H_MAXASPECT)
		{
		  int tmp;
		  XWindowAttributes ca;
		  XGetWindowAttributes(display, victim, &ca);
		  while (1)
		    {
		      tmp=gcd(ca.width, ca.height);
		      xaspect=ca.width/tmp;
		      yaspect=ca.height/tmp;
		      if (xaspect>1 && yaspect>1 && (xaspect+yaspect>100))
			{
			  xaspect>>=1;
			  xaspect>>=1;
			} else
			  break;
		    }
		  D("xaspect=%d yaspect=%d\n",xaspect,yaspect);
		}

	      place_window();
	      break;
	    }

	  usleep(2000); /* 1/50 second */
	}
      if (!unlock_done)
	{
	  my_xunlockwindow(display, wattr.root, lock_property);
	  unlock_done=1;
	}
    }
  
  while (!waitpid_done)
    {
      if (got_sigwinch)
	{
	  Window oldwindow=WINDOW;
	  got_sigwinch=0;
	  
	  D("Got SIGWINCH, new parent=%x\n",WINDOW);
	  
	  if (victim)
	    {
	      place_window();
	    } else {
	      if (my_strstr(command,"$window") &&
		  oldwindow != WINDOW)
		{
		  kill(pid, SIGKILL);
		  waitpid_done++;
		  continue;
	      }
	    }
	}
      D("Waiting for godot... %d\n",pid);
      if (flags & H_EXITS)
	{
	  exit(0);
	} else {
	  if (waitpid(pid,&status,0) > 0)
	    waitpid_done++;
	}
      D("Waid done?? (%d)\n",waitpid_done);
    }
  
  if (!WIFEXITED(status))
    {
      D("Process dumped core or something...\n");
      exit(10);
    }
  if (WEXITSTATUS(status) && !(flags & H_IGNORE_ERRORS))
    {
      D("Process exited with error code: %d\n",WEXITSTATUS(status));
      exit(WEXITSTATUS(status));
    }
  D("exited ok!\n");
}

int main(int argc, char **argv)
{
  D("Helper started.....\n");

  if (argc < 7)
  {
    fprintf(stderr,"MozPlugger version " VERSION " helper application.\n"
	    "Written by Fredrik Hubinette and Louis Bavoil.\n" 
	    "Please see 'man mozplugger' for details.\n");
    exit(1);
  }

  signal(SIGTERM, kill_child);
  signal(SIGWINCH, sigwinch);

  sscanf(argv[1],"%d,%d,%lu,%d,%d,%d,%d",
	 &flags,
	 &repeats,
	 (long unsigned int *) &windata.window,
	 (int *)&windata.x,
	 (int *)&windata.y,
	 (int *)&windata.width,
	 (int *)&windata.height);
  file=argv[2];
  winname=argv[3];
  displayname=argv[4];
  command=argv[5];
  mimetype=argv[6];

  if (!strlen(file)) file=0;
  if (!strlen(winname)) winname=0;
  if (!strlen(displayname)) displayname=0;

  D("HELPER: %s %s %s %s %s\n",
    argv[0],
    argv[1],
    file,
    displayname,
    command);

  while (repeats > 0)
  {
    char *argv[10];
    char buffer[16384];
    char *foo=buffer;
    int loops=1;
    int pid;

    /* This application will use the $repeat variable */
    if (flags & H_REPEATCOUNT)
    {
      loops=repeats;
    }

    /* This application will play the data forever */
    if (flags & H_LOOP)
    {
      D("Expecting application to loop.\n");
      loops=0x7fffffff;
    }

    /* setup environment variable $file */
    if ((flags & H_MANY) && !(flags & H_REPEATCOUNT))
      {
	int e;
	sprintf(foo,"file=%s",file);
	loops=MIN(repeats,10);
	for (e=0;e<loops;e++)
	  {
	    strcat(foo," ");
	    strcat(foo,file);
	  }
      } else {
	sprintf(foo,"file=%s",file);
      }
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $env */
    sprintf(foo,"window=%ld",(long)WINDOW);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $repeat */
    sprintf(foo,"repeat=%ld",(long)repeats);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $width */
    /* JR, 19.12.2002: Width and height to env! */
    sprintf(foo,"width=%ld",(long)windata.width);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $height */
    sprintf(foo,"height=%ld",(long)windata.height);
    putenv(foo);
    foo+=strlen(foo)+1;

    /* setup environment variable $DISPLAY */
    if (displayname)
    {
      sprintf(foo,"DISPLAY=%s",displayname);
      D("putenv(%s)\n",foo);
      putenv(foo);
      foo+=strlen(foo)+1;
    }

    /* setup environment variable $mimetype */
    if (mimetype)
    {
      sprintf(foo,"mimetype=%s",mimetype);
      D("putenv(%s)\n",foo);
      putenv(foo);
      foo+=strlen(foo)+1;
    }

    /* Create command line */
    argv[0]="/bin/sh";
    argv[1]="-c";
    argv[2]=command;
    argv[3]=0;

    D("Execing %s (repeats=%d loops=%d)\n",command,repeats,loops);

    if (flags & H_SWALLOW)
      setup_swallow();
    
    D("Running %s\n",command);

    if ((pid = fork()) == -1)
      exit(10);
      
    if (!pid)
      {
	run_app(argv);
      } else {
	D("waiting for (%d)\n", pid);
	handle_app(pid);
	D("wait done repeats=%d loops=%d\n", repeats, loops);
	if (repeats < MAXINT)
	  repeats-=loops;
      }
  }
  _exit(0);
}
