/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is mozilla.org code.
 *
 * The Initial Developer of the Original Code is
 * Netscape Communications Corporation.
 * Portions created by the Initial Developer are Copyright (C) 1998
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Stephen Mak <smak@sun.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

/*
 * npunix.c
 *
 * Netscape Client Plugin API
 * - Wrapper function to interface with the Netscape Navigator
 *
 * dp Suresh <dp@netscape.com>
 * Peter Leese - modified for mozplugger
 *
 */

#define XP_UNIX 1

#include <stdio.h>
#include <string.h>

#include "npapi.h"
#include "npupp.h"

/***********************************************************************
 *
 * Globals
 *
 ***********************************************************************/

static NPNetscapeFuncs   gNetscapeFuncs;    /* Netscape Function table */


/***********************************************************************
 *
 * Wrapper functions : plugin calling Netscape Navigator
 *
 * These functions let the plugin developer just call the APIs
 * as documented and defined in npapi.h, without needing to know
 * about the function table and call macros in npupp.h.
 *
 ***********************************************************************/

void
NPN_Version(int* plugin_major, int* plugin_minor,
            int* netscape_major, int* netscape_minor)
{
    *plugin_major = NP_VERSION_MAJOR;
    *plugin_minor = NP_VERSION_MINOR;

    /* Major version is in high byte */
    *netscape_major = gNetscapeFuncs.version >> 8;
    /* Minor version is in low byte */
    *netscape_minor = gNetscapeFuncs.version & 0xFF;
}

/******************************************************************************/
NPError
NPN_GetValue(NPP instance, NPNVariable variable, void *r_value)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_GetValueUPP func = gNetscapeFuncs.getvalue;
    if(func)
    {
        retVal = (*func)(instance, variable, r_value);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_SetValue(NPP instance, NPPVariable variable, void *value)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_SetValueUPP func = gNetscapeFuncs.setvalue;
    if(func)
    {
        retVal = (*func)(instance, variable, value);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_GetURL(NPP instance, const char* url, const char* window)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_GetURLUPP func = gNetscapeFuncs.geturl;
    if(func)
    {
        retVal = (*func)(instance, url, window);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_GetURLNotify(NPP instance, const char* url, const char* window, void* notifyData)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_GetURLNotifyUPP func = gNetscapeFuncs.geturlnotify;
    if(func)
    {
        retVal = (*func)(instance, url, window, notifyData);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_PostURL(NPP instance, const char* url, const char* window,
         uint32_t len, const char* buf, NPBool file)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_PostURLUPP func = gNetscapeFuncs.posturl;
    if(func)
    {
        retVal = (*func)(instance, url, window, len, buf, file);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_PostURLNotify(NPP instance, const char* url, const char* window, uint32_t len,
                  const char* buf, NPBool file, void* notifyData)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_PostURLNotifyUPP func = gNetscapeFuncs.posturlnotify;
    if(func)
    {
        retVal = (*func)(instance, url, window, len, buf, file, notifyData);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_RequestRead(NPStream* stream, NPByteRange* rangeList)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_RequestReadUPP func = gNetscapeFuncs.requestread;
    if(func)
    {
        retVal = (*func)(stream, rangeList);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_NewStream(NPP instance, NPMIMEType type, const char *window,
          NPStream** stream_ptr)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_NewStreamUPP func = gNetscapeFuncs.newstream;
    if(func)
    {
        retVal = (*func)(instance,  type, window, stream_ptr);
    }
    return retVal;
}

/******************************************************************************/
int32_t
NPN_Write(NPP instance, NPStream* stream, int32_t len, void* buffer)
{
    int32_t retVal = 0;
    NPN_WriteUPP func = gNetscapeFuncs.write;
    if(func)
    {
        retVal = (*func)(instance, stream, len, buffer);
    }
    return retVal;
}

/******************************************************************************/
NPError
NPN_DestroyStream(NPP instance, NPStream* stream, NPError reason)
{
    NPError retVal = NPERR_INVALID_FUNCTABLE_ERROR;
    NPN_DestroyStreamUPP func = gNetscapeFuncs.destroystream;
    if(func)
    {
        retVal = (*func)(instance, stream, reason);
    }
    return retVal;
}

/******************************************************************************/
void
NPN_Status(NPP instance, const char* message)
{
    NPN_StatusUPP func = gNetscapeFuncs.status;
    if(func)
    {
        (*func)(instance, message);
    }
}

/******************************************************************************/
const char*
NPN_UserAgent(NPP instance)
{
    const char * retVal = 0;
    NPN_UserAgentUPP func = gNetscapeFuncs.uagent;
    if(func)
    {
        retVal = (*func)(instance);
    }
    return retVal;
}

/******************************************************************************/
void*
NPN_MemAlloc(uint32_t size)
{
    void * retVal = 0;
    NPN_MemAllocUPP func = gNetscapeFuncs.memalloc;
    if(func)
    {
        retVal = (*func)(size);
    }
    return retVal;
}

/******************************************************************************/
void NPN_MemFree(void* ptr)
{
    NPN_MemFreeUPP func = gNetscapeFuncs.memfree;
    if(func)
    {
        (*func)(ptr);
    }
}

/******************************************************************************/
uint32_t NPN_MemFlush(uint32_t size)
{
    uint32_t retVal = 0;
    NPN_MemFlushUPP func = gNetscapeFuncs.memflush;
    if(func)
    {
        retVal = (*func)(size);
    }
    return retVal;
}

/******************************************************************************/
void NPN_ReloadPlugins(NPBool reloadPages)
{
    NPN_ReloadPluginsUPP func = gNetscapeFuncs.reloadplugins;
    if(func)
    {
        (*func)(reloadPages);
    }
}

/******************************************************************************/
void
NPN_InvalidateRect(NPP instance, NPRect *invalidRect)
{
    NPN_InvalidateRectUPP func = gNetscapeFuncs.invalidaterect;
    if(func)
    {
        (*func)(instance, invalidRect);
    }
}

/******************************************************************************/
void
NPN_InvalidateRegion(NPP instance, NPRegion invalidRegion)
{
    NPN_InvalidateRegionUPP func = gNetscapeFuncs.invalidateregion;
    if(func)
    {
        (*func)(instance, invalidRegion);
    }
}

/******************************************************************************/
void
NPN_ForceRedraw(NPP instance)
{
    NPN_ForceRedrawUPP func = gNetscapeFuncs.forceredraw;
    if(func)
    {
        (*func)(instance);
    }
}

/******************************************************************************/
void NPN_PushPopupsEnabledState(NPP instance, NPBool enabled)
{
    NPN_PushPopupsEnabledStateUPP func = gNetscapeFuncs.pushpopupsenabledstate;
    if(func)
    {
        (*func)(instance, enabled);
    }
}

/******************************************************************************/
void NPN_PopPopupsEnabledState(NPP instance)
{
    NPN_PopPopupsEnabledStateUPP func = gNetscapeFuncs.poppopupsenabledstate;
    if(func)
    {
        (*func)(instance);
    }
}

/******************************************************************************/
NPIdentifier NPN_GetStringIdentifier(const NPUTF8 *name)
{
    NPIdentifier retVal = 0;
    NPN_GetStringIdentifierUPP func = gNetscapeFuncs.getstringidentifier;
    if(func)
    {
        retVal = (*func)(name);
    }
    return retVal;
}

/******************************************************************************/
void NPN_GetStringIdentifiers(const NPUTF8 **names, int32_t nameCount,
                              NPIdentifier *identifiers)
{
    NPN_GetStringIdentifiersUPP func = gNetscapeFuncs.getstringidentifiers;
    if(func)
    {
        (*func)(names, nameCount, identifiers);
    }
}

/******************************************************************************/
NPIdentifier NPN_GetIntIdentifier(int32_t intid)
{
    NPIdentifier retVal = 0;
    NPN_GetIntIdentifierUPP func = gNetscapeFuncs.getintidentifier;
    if(func)
    {
        retVal = (*func)(intid);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_IdentifierIsString(NPIdentifier identifier)
{
    bool retVal = false;
    NPN_IdentifierIsStringUPP func = gNetscapeFuncs.identifierisstring;
    if(func)
    {
        retVal = (*func)(identifier);
    }
    return retVal;
}

/******************************************************************************/
NPUTF8 *NPN_UTF8FromIdentifier(NPIdentifier identifier)
{
    NPUTF8 * retVal = 0;
    NPN_UTF8FromIdentifierUPP func = gNetscapeFuncs.utf8fromidentifier;
    if(func)
    {
        retVal = (*func)(identifier);
    }
    return retVal;
}

/******************************************************************************/
int32_t NPN_IntFromIdentifier(NPIdentifier identifier)
{
    int32_t retVal = 0;
    NPN_IntFromIdentifierUPP func = gNetscapeFuncs.intfromidentifier;
    if(func)
    {
        retVal = (*func)(identifier);
    }
    return retVal;
}

/******************************************************************************/
NPObject *NPN_CreateObject(NPP npp, NPClass *aClass)
{
    NPObject * retVal = 0;
    NPN_CreateObjectUPP func = gNetscapeFuncs.createobject;
    if(func)
    {
        retVal = (*func)( npp, aClass);
    }
    return retVal;
}

/******************************************************************************/
NPObject *NPN_RetainObject(NPObject *obj)
{
    NPObject * retVal = 0;
    NPN_RetainObjectUPP func = gNetscapeFuncs.retainobject;
    if(func)
    {
        retVal = (*func)( obj);
    }
    return retVal;
}

/******************************************************************************/
void NPN_ReleaseObject(NPObject *obj)
{
    NPN_ReleaseObjectUPP func = gNetscapeFuncs.releaseobject;
    if(func)
    {
        (*func)(obj);
    }
}

/******************************************************************************/
bool NPN_Invoke(NPP npp, NPObject* obj, NPIdentifier methodName,
                const NPVariant *args, uint32_t argCount, NPVariant *result)
{
    bool retVal = false;
    NPN_InvokeUPP func = gNetscapeFuncs.invoke;
    if(func)
    {
        retVal = (*func)(npp, obj, methodName,  args, argCount, result);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_InvokeDefault(NPP npp, NPObject* obj, const NPVariant *args,
                       uint32_t argCount, NPVariant *result)
{
    bool retVal = false;
    NPN_InvokeDefaultUPP func = gNetscapeFuncs.invokeDefault;
    if(func)
    {
        retVal = (*func)(npp, obj,  args, argCount, result);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_Evaluate(NPP npp, NPObject* obj, NPString *script,
                  NPVariant *result)
{
    bool retVal = false;
    NPN_EvaluateUPP func = gNetscapeFuncs.evaluate;
    if(func)
    {
        retVal = (*func)(npp, obj, script, result);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_GetProperty(NPP npp, NPObject* obj, NPIdentifier propertyName,
                     NPVariant *result)
{
    bool retVal = false;
    NPN_GetPropertyUPP func = gNetscapeFuncs.getproperty;
    if(func)
    {
       retVal = (*func)(npp, obj, propertyName, result);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_SetProperty(NPP npp, NPObject* obj, NPIdentifier propertyName,
                     const NPVariant *value)
{
    bool retVal = false;
    NPN_SetPropertyUPP func = gNetscapeFuncs.setproperty;
    if(func)
    {
        retVal = (*func)(npp, obj, propertyName, value);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_RemoveProperty(NPP npp, NPObject* obj, NPIdentifier propertyName)
{
    bool retVal = false;
    NPN_RemovePropertyUPP func = gNetscapeFuncs.removeproperty;
    if(func)
    {
        retVal = (*func)(npp, obj, propertyName);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_HasProperty(NPP npp, NPObject* obj, NPIdentifier propertyName)
{
    bool retVal = false;
    NPN_HasPropertyUPP func = gNetscapeFuncs.hasproperty;
    if(func)
    {
        retVal = (*func)(npp, obj, propertyName);
    }
    return retVal;
}

/******************************************************************************/
bool NPN_HasMethod(NPP npp, NPObject* obj, NPIdentifier methodName)
{
    bool retVal = false;
    NPN_HasMethodUPP func = gNetscapeFuncs.hasmethod;
    if(func)
    {
        retVal = (*func)(npp, obj, methodName);
    }
    return retVal;
}

/******************************************************************************/
void NPN_ReleaseVariantValue(NPVariant *variant)
{
    NPN_ReleaseVariantValueUPP func = gNetscapeFuncs.releasevariantvalue;
    if(func)
    {
        (*func)(variant);
    }
}

/******************************************************************************/
void NPN_SetException(NPObject* obj, const NPUTF8 *message)
{
    NPN_SetExceptionUPP func = gNetscapeFuncs.setexception;
    if(func)
    {
        (*func)(obj, message);
    }
}

/*********************************************************************** 
 *
 * These functions are located automagically by netscape.
 *
 ***********************************************************************/

/*
 * NP_GetMIMEDescription
 *  - Netscape needs to know about this symbol
 *  - Netscape uses the return value to identify when an object instance
 *    of this plugin should be created.
 */
char *
NP_GetMIMEDescription(void)
{
    return NPP_GetMIMEDescription();
}

/*
 * NP_GetValue [optional]
 *  - Netscape needs to know about this symbol.
 *  - Interfaces with plugin to get values for predefined variables
 *    that the navigator needs.
 */
NPError
NP_GetValue(void* future, NPPVariable variable, void *value)
{
    return NPP_GetValue(future, variable, value);
}

/*
 * NP_Initialize
 *  - Netscape needs to know about this symbol.
 *  - It calls this function after looking up its symbol before it
 *    is about to create the first ever object of this kind.
 *
 * PARAMETERS
 *    nsTable   - The netscape function table. If developers just use these
 *        wrappers, they don't need to worry about all these function
 *        tables.
 * RETURN
 *    pluginFuncs
 *      - This functions needs to fill the plugin function table
 *        pluginFuncs and return it. Netscape Navigator plugin
 *        library will use this function table to call the plugin.
 *
 */
NPError
NP_Initialize(NPNetscapeFuncs* nsTable, NPPluginFuncs* pluginFuncs)
{
    NPError err = NPERR_NO_ERROR;

    /* validate input parameters */

    if ((nsTable == NULL) || (pluginFuncs == NULL))
    {
        err = NPERR_INVALID_FUNCTABLE_ERROR;
    }

    /*
     * Check the major version passed in Netscape's function table.
     * We won't load if the major version is newer than what we expect.
     * Also check that the function tables passed in are big enough for
     * all the functions we need (they could be bigger, if Netscape added
     * new APIs, but that's OK with us -- we'll just ignore them).
     *
     */

    if (err == NPERR_NO_ERROR) 
    {
        if ((nsTable->version >> 8) > NP_VERSION_MAJOR)
        {
            err = NPERR_INCOMPATIBLE_VERSION_ERROR;
        }

        if (nsTable->size < ((char *)&nsTable->getvalue - (char *)nsTable))
        {
            err = NPERR_INVALID_FUNCTABLE_ERROR;
        }

        if (pluginFuncs->size < ((char*)&pluginFuncs->getvalue - (char*)pluginFuncs))
        {
            err = NPERR_INVALID_FUNCTABLE_ERROR;
        }
    }
        
    
    if (err == NPERR_NO_ERROR) 
    {
        /* Zero everything */
        memset(&gNetscapeFuncs, 0, sizeof(gNetscapeFuncs));
        /*
         * Copy all the fields of Netscape function table into our
         * copy so we can call back into Netscape later.  Note that
         * we need to copy the fields one by one, rather than assigning
         * the whole structure, because the Netscape function table
         * could actually be bigger than what we expect.
         */
        gNetscapeFuncs.version       = nsTable->version;
        gNetscapeFuncs.size          = nsTable->size;
        gNetscapeFuncs.posturl       = nsTable->posturl;
        gNetscapeFuncs.geturl        = nsTable->geturl;
        gNetscapeFuncs.geturlnotify  = nsTable->geturlnotify;
        gNetscapeFuncs.requestread   = nsTable->requestread;
        gNetscapeFuncs.newstream     = nsTable->newstream;
        gNetscapeFuncs.write         = nsTable->write;
        gNetscapeFuncs.destroystream = nsTable->destroystream;
        gNetscapeFuncs.status        = nsTable->status;
        gNetscapeFuncs.uagent        = nsTable->uagent;
        gNetscapeFuncs.memalloc      = nsTable->memalloc;
        gNetscapeFuncs.memfree       = nsTable->memfree;
        gNetscapeFuncs.memflush      = nsTable->memflush;
        gNetscapeFuncs.reloadplugins = nsTable->reloadplugins;

        gNetscapeFuncs.posturlnotify = nsTable->posturlnotify;
        gNetscapeFuncs.getvalue      = nsTable->getvalue;

        /* Version 0.9?? */
        if (nsTable->size >= ((char *)&nsTable->setvalue - (char *)nsTable))
        {
            gNetscapeFuncs.setvalue  = nsTable->setvalue;
        }

        /* Version 0.11 */
        if (nsTable->size >= ((char *)&nsTable->forceredraw - (char *)nsTable))
        {
            gNetscapeFuncs.invalidaterect = nsTable->invalidaterect;
            gNetscapeFuncs.invalidateregion = nsTable->invalidateregion;
            gNetscapeFuncs.forceredraw = nsTable->forceredraw;
        }

        /* Version 0.14 */
        if (nsTable->size >= ((char *)&nsTable->setexception - (char *)nsTable))
        {
          gNetscapeFuncs.getstringidentifier = nsTable->getstringidentifier;
          gNetscapeFuncs.getstringidentifiers = nsTable->getstringidentifiers;
          gNetscapeFuncs.getintidentifier = nsTable->getintidentifier;
          gNetscapeFuncs.identifierisstring = nsTable->identifierisstring;
          gNetscapeFuncs.utf8fromidentifier = nsTable->utf8fromidentifier;
          gNetscapeFuncs.intfromidentifier = nsTable->intfromidentifier;
          gNetscapeFuncs.createobject = nsTable->createobject;
          gNetscapeFuncs.retainobject = nsTable->retainobject;
          gNetscapeFuncs.releaseobject = nsTable->releaseobject;
          gNetscapeFuncs.invoke = nsTable->invoke;
          gNetscapeFuncs.invokeDefault = nsTable->invokeDefault;
          gNetscapeFuncs.evaluate = nsTable->evaluate;
          gNetscapeFuncs.getproperty = nsTable->getproperty;
          gNetscapeFuncs.setproperty = nsTable->setproperty;
          gNetscapeFuncs.removeproperty = nsTable->removeproperty;
          gNetscapeFuncs.hasproperty = nsTable->hasproperty;
          gNetscapeFuncs.hasmethod = nsTable->hasmethod;
          gNetscapeFuncs.releasevariantvalue = nsTable->releasevariantvalue;
          gNetscapeFuncs.setexception = nsTable->setexception;
        }

        /* Version 0.16 */
        if (nsTable->size >=
            ((char *)&nsTable->poppopupsenabledstate - (char *)nsTable))
        {
          gNetscapeFuncs.pushpopupsenabledstate = nsTable->pushpopupsenabledstate;
          gNetscapeFuncs.poppopupsenabledstate  = nsTable->poppopupsenabledstate;
        }

        /* Version 0.18 */
        if (nsTable->size >=
            ((char *)&nsTable->enumerate - (char *)nsTable))
        {
          gNetscapeFuncs.enumerate = nsTable->enumerate;
        }

        /* Version 0.19 */
        if (nsTable->size >=
            ((char *)&nsTable->pluginthreadasynccall - (char *)nsTable))
        {
          gNetscapeFuncs.pluginthreadasynccall = nsTable->pluginthreadasynccall;
        }

        /* Version 0.19?? */
        if (nsTable->size >=
            ((char *)&nsTable->construct - (char *)nsTable))
        {
          gNetscapeFuncs.construct = nsTable->construct;
        }

        /* Version 0.20 */
        if (nsTable->size >=
            ((char *)&nsTable->setvalueforurl - (char *)nsTable))
        {
          gNetscapeFuncs.getvalueforurl = nsTable->getvalueforurl;
          gNetscapeFuncs.setvalueforurl = nsTable->setvalueforurl;
        }

        /* Version 0.21 */
        if (nsTable->size >=
            ((char *)&nsTable->getauthenticationinfo - (char *)nsTable))
        {
          gNetscapeFuncs.getauthenticationinfo = nsTable->getauthenticationinfo;
        }


        /*
         * Set up the plugin function table that Netscape will use to
         * call us.  Netscape needs to know about our version and size
         * and have a UniversalProcPointer for every function we
         * implement.
         */
        pluginFuncs->version    = (NP_VERSION_MAJOR << 8) + NP_VERSION_MINOR;
        pluginFuncs->newp          = NPP_New;
        pluginFuncs->destroy       = NPP_Destroy;
        pluginFuncs->setwindow     = NPP_SetWindow;
        pluginFuncs->newstream     = NPP_NewStream;
        pluginFuncs->destroystream = NPP_DestroyStream;
        pluginFuncs->asfile        = NPP_StreamAsFile;
        pluginFuncs->writeready    = NPP_WriteReady;
        pluginFuncs->write         = NPP_Write;
        pluginFuncs->print         = NPP_Print;
        pluginFuncs->event         = NULL;
        pluginFuncs->urlnotify     = NPP_URLNotify;

        pluginFuncs->javaClass  = NULL; /* = NPP_GetJavaClass(); deprecated */

        /* All this is so we keep backwards compatibility with old browsers! */ 
        if (pluginFuncs->size > ((char*)&pluginFuncs->getvalue - (char*)pluginFuncs))
        {     
            pluginFuncs->getvalue = (NPP_GetValueUPP) NPP_GetValue;
            if (pluginFuncs->size > ((char*)&pluginFuncs->setvalue - (char*)pluginFuncs))
            {    
                pluginFuncs->setvalue = NPP_SetValue;
            }   
        }

        if(pluginFuncs->size > sizeof(NPPluginFuncs))
        {   
            /* Do this so browser knows the upper limit of plugin functions supported */
            pluginFuncs->size = sizeof(NPPluginFuncs);
        } 
        err = NPP_Initialize();
    }
    
    return err;
}

/*
 * NP_Shutdown [optional]
 *  - Netscape needs to know about this symbol.
 *  - It calls this function after looking up its symbol after
 *    the last object of this kind has been destroyed.
 *
 */
NPError
NP_Shutdown(void)
{
    NPP_Shutdown();
    return NPERR_NO_ERROR;
}
