 /*** 
   * The contents of this file are subject to the Mozilla Public
   * License Version 1.1 (the "License"); you may not use this file
   * except in compliance with the License. You may obtain a copy of
   * the License at http://www.mozilla.org/MPL/
   *
   * Software distributed under the License is distributed on an "AS
   * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
   * implied. See the License for the specific language governing
   * rights and limitations under the License.
   *
   * The Original Code is this file as it was released on
   * June 7, 2002.
   *
   * The Initial Developer of the Original Code is HJ van Rantwijk.
   * Portions created by HJ van Rantwijk are Copyright (C) 2002-2006
   * HJ van Rantwijk.  All Rights Reserved.
   *
   */

const Cc = Components.classes;
const Ci = Components.interfaces;
const Cr = Components.results;

const DEFAULT_ENABLED = true;
const DEFAULT_DELAY = 10000; // 10 seconds
const DEFAULT_RETRY = 30000; // 5 minutes aka 5*60*1000
const DEFAULT_FREQUENCY = 1;

function debug(aMsg) {
  aMsg = ("mzUpdate Notifier: " + aMsg).replace(/\S{80}/g, "$&\n");
  Cc["@mozilla.org/consoleservice;1"].getService(Ci.nsIConsoleService).logStringMessage(aMsg);
}


/* :::::::: The Update Notifier Service ::::::::::::::: */

var mzUpdateNotifier = 
{
  mTimer: null, // need to hold on to timer ref
  updateEnabled: DEFAULT_ENABLED,
  updateDelay: DEFAULT_DELAY,
  isInitialized: 0,
  updateFrequency: DEFAULT_FREQUENCY,

  onProfileStartup: function(aProfileName)
  {
    var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
    observerService.addObserver(this, "domwindowopened", false);
  },

  /**
   * Handle notifications
   */
  observe: function(aSubject, aTopic, aData)
  {
    try {
    var self = this;

    switch (aTopic) {
      case "domwindowopened":
        this.mTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
        // we are no longer interested in the ``domwindowopened'' topic
        var observerService = Cc["@mozilla.org/observer-service;1"].getService(Ci.nsIObserverService);
        observerService.removeObserver(this, "domwindowopened");
        // remove our timer reference on XPCOM shutdown to avoid a leak.
        observerService.addObserver(this, "xpcom-shutdown", false);

        this.prefBranch = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefService).getBranch("multizilla.updates.");
        this.prefBranch.QueryInterface(Ci.nsIPrefBranch2);

        this.updateEnabled = this._getPref("enabled", DEFAULT_ENABLED);
        this.prefBranch.addObserver("enabled", this, false);

        this.updateDelay = this._getPref("check-after-ms", DEFAULT_DELAY);
        this.prefBranch.addObserver("check-after-ms", this, false);

        this.updateFrequency = this._getPref("frequency", DEFAULT_FREQUENCY); 
        this.prefBranch.addObserver("frequency", this, false);

        this.mTimer.init(this, this.updateDelay, Ci.nsITimer.TYPE_ONE_SHOT);
        debug("Initialized with a " + this.updateDelay + "ms delay");
        break;

      case "timer-callback":
        // dump("\nthis.isInitialized: " + this.isInitialized);
        switch(this.isInitialized) {
          case 0: 
            var browerWindow = this.getMostRecentBrowserWindow();
            this.isInitialized = (browerWindow == null) ? 0 : 1;
            this.mTimer.init(this, DEFAULT_RETRY, Ci.nsITimer.TYPE_ONE_SHOT);
            break;
          case 1: 
            this.isInitialized = 2;
            this.mTimer.init(this, this.updateDelay, Ci.nsITimer.TYPE_ONE_SHOT);
            break;
          case 2:
            this.mTimer = null; // free up timer so it can be gc'ed
            this.checkForUpdate();
            break;
        }
        break;

      case "nsPref:changed":
        switch(aData) {
          case "enabled":
            this.updateEnabled = this._getPref(aData, this.updateEnabled);

            if (this.updateEnabled)
              this.mTimer.init(this, this.updateDelay, Ci.nsITimer.TYPE_ONE_SHOT);
            break;

          case "check-after-ms":
            this.updateDelay = this._getPref(aData, this.updateDelay);
            break;

          case "frequency":
            this.updateFrequency = this._getPref(aData, this.updateFrequency);
            break;
        }
        break;

      case "xpcom-shutdown":
        /* 
         * We need to drop our timer reference here to avoid a leak
         * since the timer keeps a reference to the observer. 
         */
        this.mTimer = null;
    }
    } catch(ex) { debug("ex: " + ex); }
  },

  checkForUpdate: function()
  {
    if (this.getOfflineStatus()) {
      debug("Status: off-line (automatic retry in 5 minutes)");
      this.mTimer = Cc["@mozilla.org/timer;1"].createInstance(Ci.nsITimer);
      this.mTimer.init(this, DEFAULT_RETRY, Ci.nsITimer.TYPE_ONE_SHOT);
    }
    else if (this.shouldCheckForUpdate()) {
      debug("Checking for updates...");
      try {
        var prefs = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefBranch);
        var updateDatasourceURL = "http://multizilla.mozdev.org/xpi/updates.rdf";

        var xmlRequest = Cc["@mozilla.org/xmlextras/xmlhttprequest;1"].createInstance(Ci.nsIXMLHttpRequest);
        xmlRequest.open("HEAD", updateDatasourceURL, true);
        xmlRequest.onerror = function(aEvent)
        {
          var request, status;

          try {
            request = aEvent.target;
            // the following may throw (e.g. a local file or timeout)
            status = request.status;
          }
          catch(ex) {
            request = aEvent.target.channel.QueryInterface(Ci.nsIRequest);
            status = request.status;
          }
          dump("\nstatus: " + status);
        }
        xmlRequest.onload = function()
        {
          // dump("\nxmlRequest.readyState: " + xmlRequest.readyState);
          // dump("\nonload: " + xmlRequest.status);

          if (xmlRequest.readyState == 4)
            dump("\nxmlRequest.readyState == 4");

          if (xmlRequest.status == 200)
            dump("\nxmlRequest.status == 200");
        }
        var rdf = Cc["@mozilla.org/rdf/rdf-service;1"].getService(Ci.nsIRDFService);
        var ds = rdf.GetDataSource(updateDatasourceURL);
        ds = ds.QueryInterface(Ci.nsIRDFXMLSink);
        ds.addXMLSinkObserver(mzUpdateDatasourceObserver);
      }
      catch (ex) {
        debug("Exception getting updates.rdf: " + ex);
      }
    }
  },

  shouldCheckForUpdate: function()
  {
    var shouldCheck = false;

    try {
      if (this._getPref("enabled", true)) {
        const lastCheckPrefID = "last_checked";
        // The update frequency pref is (normally) set to: 1, 7, 30 days
        var freq = this._getPref("frequency", DEFAULT_FREQUENCY) * 86400; // 24 hours * 60 minutes * 60 seconds
        var now = (new Date().valueOf()) / 1000; // milliseconds -> seconds
        var prefs = this.prefBranch;

        if (!prefs.prefHasUserValue(lastCheckPrefID))
          prefs.setIntPref(lastCheckPrefID, 0);
        var lastChecked = this._getPref(lastCheckPrefID, now);

        if ((lastChecked + freq) > now)
          return false;

        prefs.setIntPref(lastCheckPrefID, now);
        prefs = prefs.QueryInterface(Ci.nsIPrefService);
        prefs.savePrefFile(null);
        shouldCheck = true;
      }
    }
    catch (ex) {
      shouldCheck = false;
      debug("Exception in shouldCheckForUpdate: " + ex);
    }
    return shouldCheck;
  },

  getMostRecentBrowserWindow: function()
  {
    return this._getMostRecentWindow("navigator:browser");
  },

  getOfflineStatus: function()
  {
    return Cc["@mozilla.org/network/io-service;1"].getService(Ci.nsIIOService).offline;
  },

  _getMostRecentWindow: function(aWindowType)
  {
    return Cc['@mozilla.org/appshell/window-mediator;1'].getService(Ci.nsIWindowMediator).getMostRecentWindow(aWindowType);
  },

  _getPref: function(aName, aDefault) {
    var prefs = this.prefBranch;

    try {
      switch (prefs.getPrefType(aName)) {
        case prefs.PREF_STRING:
             return prefs.getCharPref(aName);
        case prefBranch.PREF_BOOL:
             return prefs.getBoolPref(aName);
        case prefBranch.PREF_INT:
             return prefs.getIntPref(aName);
        default:
             return aDefault;
      }
    }
    catch(ex) {
      return aDefault;
    }
  },

  QueryInterface: function(aIID)
  {
    if (aIID.equals(Ci.nsIObserver) || 
        aIID.equals(Ci.nsIProfileStartupListener) ||
        aIID.equals(Ci.nsISupports))
          return this;
    Components.returnCode = Cr.NS_ERROR_NO_INTERFACE;
    return null;
  }
}
////////////////////////////////////////////////////////////////////////
//   mzUpdateDatasourceObserver : nsIRDFXMLSinkObserver
////////////////////////////////////////////////////////////////////////
var mzUpdateDatasourceObserver = 
{
  onBeginLoad: function(aSink) {},
  onInterrupt: function(aSink) {},
  onResume: function(aSink) {},

  onEndLoad: function(aSink)
  {
    aSink.removeXMLSinkObserver(this);
    var dataSource = aSink.QueryInterface(Ci.nsIRDFDataSource);
    var productName = this.getProductName();
    // dump("\nproductName: " + productName + "\n");
    var updateInfo = this.getUpdateInfo(dataSource, productName);
    var productObj = { name    : updateInfo.productName,
                       version : updateInfo.version,
                       revision: updateInfo.revision };
    var updateAvailable = this.newerVersionAvailable(productObj);
    // The following three lines restore the used version/revision info!
    var browserWindow = this.getBrowserWindow();
    productObj.version = browserWindow.gBrowser.mVersion;
    productObj.revision = browserWindow.gBrowser.mVersionRev;

    if (updateInfo && updateAvailable) {
      debug("Update for: " + updateInfo.productName + " available!");
      var promptService = Cc["@mozilla.org/embedcomp/prompt-service;1"].getService(Ci.nsIPromptService);
      var winWatcher = Cc["@mozilla.org/embedcomp/window-watcher;1"].getService(Ci.nsIWindowWatcher);
      var unBundle = this.getBundle("chrome://multiviews/locale/update-notifications.properties");

      if (!unBundle)
        return;

      var title = unBundle.formatStringFromName("updateDialogTitle", [updateInfo.productName], 1);
      var description = unBundle.formatStringFromName("updateDialogText", [updateInfo.productName, productObj.version, productObj.revision, updateInfo.productName, updateInfo.version, updateInfo.revision, updateInfo.productName], 7);
      var button0Text = unBundle.GetStringFromName("getItNow");
      var button1Text = unBundle.GetStringFromName("noThanks");
      var checkMsg = unBundle.GetStringFromName("dontAskAgain");
      var checkVal = {value:0};
      var result = promptService.confirmEx(winWatcher.activeWindow, title, description, 
                                           (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_IS_STRING) +
                                           (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_IS_STRING),
                                           button0Text, button1Text, null, checkMsg, checkVal);

      if (result == 0) {
        var ass = Cc["@mozilla.org/appshell/appShellService;1"].getService(Ci.nsIAppShellService);
        var installationText = updateInfo.text;
        var installationURL = updateInfo.URL;
        ass.hiddenDOMWindow.InstallTrigger.install({installationText : installationURL});
      }
      if (checkVal.value) {
        var prefs = Cc["@mozilla.org/preferences-service;1"].getService(Ci.nsIPrefBranch);
        prefs.setBoolPref("multizilla.update_notifications.enabled", false);
      }
    }
    else {
      debug("No update for: " + updateInfo.productName + " available!");
    }
  },

  onError: function(aSink, aStatus, aErrorMsg)
  {
    dump("\naStatus: " + aStatus);
    if (aStatus == 2152398918)
      debug("Failed (off-line)");
    else if (aStatus == 2152398878)
      debug("Failed (no connection)");
    else
      debug("Error: " + aStatus + ": " + aErrorMsg);
    aSink.removeXMLSinkObserver(this);
  },

  getBrowserWindow: function()
  {
    var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1'].getService(Components.interfaces.nsIWindowMediator);
    return windowManager.getMostRecentWindow("navigator:browser");
  },

  getBundle: function(aURL)
  {
    var strBundleService = Cc["@mozilla.org/intl/stringbundle;1"].getService(Ci.nsIStringBundleService);
    return strBundleService.createBundle(aURL);
  },

  getProductName: function()
  {
    var browserWindow = this.getBrowserWindow();
    var tabBrowser = (browserWindow) ? browserWindow.getBrowser() : null;

    return (tabBrowser && 'mProduct' in tabBrowser) ? tabBrowser.mProduct : undefined;
  },

  getUpdateInfo: function(aDS, aProductName)
  {
    var updateInfo = new Object;

    try {
      var rdf = Cc["@mozilla.org/rdf/rdf-service;1"].getService(Ci.nsIRDFService);
      var src = "urn:updates:latest";
      updateInfo.registryName = this._getTarget(rdf, aDS, src, "registryName");
      updateInfo.productName  = this._getTarget(rdf, aDS, src, "productName");

      if (aProductName == undefined || aProductName == "undetermined")
        aProductName = "Win";
      else
        aProductName += "-";

      updateInfo.version = this._getTarget(rdf, aDS, src, aProductName + "Version");
      updateInfo.revision = this._getTarget(rdf, aDS, src, aProductName + "Revision");
      updateInfo.text = this._getTarget(rdf, aDS, src, aProductName + "Text");
      updateInfo.URL  = this._getTarget(rdf, aDS, src, aProductName + "URL");
    }
    catch (ex) {
      debug("Exception getting update info: " + ex);
      updateInfo = null;
    }
    return updateInfo;
  },

  newerVersionAvailable: function(aProductObj)
  {
    var browserWindow = this.getBrowserWindow();
    
    if (!browserWindow)
      return false;

    var vc = Cc["@mozilla.org/xpcom/version-comparator;1"].getService(Ci.nsIVersionComparator);

    if (aProductObj.name == "MultiZilla") {
      if (vc.compare(String(browserWindow.gBrowser.mVersion + browserWindow.gBrowser.mVersionRev), 
                     String(aProductObj.version + aProductObj.revision)) == -1)
        return true;
    }
    /* else if (aProductObj.name == "GoogleBox") {
      var mzToolbox = browserWindow.document.getElementById("MultiZillaToolbar");

      if (mzToolbox.mToolboxActivated) { // GoogleBox installed?
        var googleBox = browserWindow.document.getElementById("internalToolBox");
        // Pre GoogleBox v1.4a versions had no version info, that's why we check for 'undefined'
        var googleBoxVersion = (googleBox.mVersion == undefined) ? 0 : googleBox.mVersion;
        var googleBoxRevision = (googleBox.mVersionRev == undefined) ? "" : googleBox.mVersionRev;

        if (vc.compare(String(googleBoxVersion + googleBoxRevision), 
                       String(aProductObj.version + aProductObj.revision)) == -1)
          return true;
      }
      return false;
    } */
  },

  _getTarget: function(aRDF, aDS, aSrc, aProp)
  {
    var src = aRDF.GetResource(aSrc);
    var arc = aRDF.GetResource("http://home.netscape.com/NC-rdf#" + aProp);
    var target = aDS.GetTarget(src, arc, true);
    return target.QueryInterface(Ci.nsIRDFLiteral).Value;
  }
}

/* :::::::: Service Registration & Initialization ::::::::::::::: */

/* ........ nsIModule .............. */

var mzUpdateNotifierModule = 
{
  mClassName:     "MultiZilla Update Notifier",
  mContractID:    "@mozilla.org/update-notifier;1",
  mClassID:       Components.ID("8b6dcf5e-3b5a-4fff-bff5-65a8fa9d71b2"),

  getClassObject: function(aCompMgr, aCID, aIID)
  {
    if (!aCID.equals(this.mClassID))
      throw Cr.NS_ERROR_NO_INTERFACE;
    if (!aIID.equals(Ci.nsIFactory))
      throw Cr.NS_ERROR_NOT_IMPLEMENTED;

    return this.mFactory;
  },

  registerSelf: function(aCompMgr, aFileSpec, aLocation, aType)
  {
    debug("Registering...\n");
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.registerFactoryLocation(this.mClassID, this.mClassName, this.mContractID, aFileSpec, aLocation, aType);
    this.getCategoryManager().addCategoryEntry("profile-startup-category", this.mContractID, "", true, true);
  },

  unregisterSelf: function(aCompMgr, aFileSpec, aLocation)
  {
    debug("Unregistering...\n");
    aCompMgr = aCompMgr.QueryInterface(Ci.nsIComponentRegistrar);
    aCompMgr.unregisterFactoryLocation(this.mClassID, aFileSpec);
    this.getCategoryManager().deleteCategoryEntry("profile-startup-category", this.mContractID, true);
  },

  canUnload: function(aCompMgr)
  {
    return true;
  },

  getCategoryManager: function()
  {
    return Cc["@mozilla.org/categorymanager;1"].getService(Ci.nsICategoryManager);
  },

  /* ........ nsIFactory .............. */

  mFactory:
  {
    createInstance: function(aOuter, aIID)
    {
      if (aOuter != null)
        throw Cr.NS_ERROR_NO_AGGREGATION;
      if (!aIID.equals(Ci.nsIObserver) &&
          !aIID.equals(Ci.nsIProfileStartupListener) &&
          !aIID.equals(Ci.nsISupports))
        throw Cr.NS_ERROR_INVALID_ARG;

      return mzUpdateNotifier.QueryInterface(aIID);
    },

    lockFactory: function(aLock)
    {
      // quiten warnings
    }
  }
};

function NSGetModule(aCompMgr, aFileSpec)
{
  return mzUpdateNotifierModule;
}


