#!/bin/python

""" Universal archive extractor. It can handle .zip, .tar, .tar.gz, tar.bz2,
and plain directories. In cases of plain directory, it just copy everything
from target directory to destination. It's equivalent to shell command 
cp target/* dest/ """

import os, sys, tarfile, shutil, zipfile

class Error(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return str(self.value)

class baseExtractor:
    def __init__(self, archive, destdir):
        self.archive = archive
        self.destdir = os.path.abspath(destdir)
        if not os.path.isdir(self.destdir):
            raise Error ("%s is not a directory" % (destdir))

class zipExtractor(baseExtractor):
    def extract(self):
        zfile = zipfile.ZipFile(self.archive)
        for member in zfile.namelist():
            memberpath = os.path.abspath(os.path.join(self.destdir, member))
            if memberpath.startswith(self.destdir):
                if member.endswith('/'):
                    if not os.path.exists(memberpath):
                        os.makedirs(memberpath)
                else:
                    basedir = os.path.dirname(memberpath)
                    if not os.path.exists(basedir):
                        os.makedirs(basedir)
                    f = open(memberpath, 'wb')
                    f.write(zfile.read(member))
                    f.close()
        zfile.close()

class tarExtractor(baseExtractor):
    def extract(self):
       tar = tarfile.open(self.archive, 'r|*')
       # security check: do not create files or directory outside of path
       for member in tar:
          memberpath = os.path.abspath(os.path.join(self.destdir, member.name))
          if memberpath.startswith(self.destdir):
             tar.extract(member, self.destdir)
       tar.close()

class plainDirExtractor(baseExtractor):
    def extract(self):
        self.copytree(self.archive, self.destdir)

    def copytree(self, src, dest):
        if not os.path.isdir(dest):
            os.makedirs(dest)

        for name in os.listdir(src):
            srcname = os.path.join(src, name)
            dstname = os.path.join(dest, name)
            if os.path.islink(srcname):
                os.symlink(os.readlink(srcname), dstname)
            if os.path.isdir(srcname):
                self.copytree(srcname, dstname)
            else:
                shutil.copy2(srcname, dstname)



def factory(archive, destdir):
    if os.path.isdir(archive):
        return plainDirExtractor(archive, destdir)

    elif archive.endswith('.tar.gz'):
        return tarExtractor(archive, destdir)

    elif archive.endswith('.tar.bz2'):
        return tarExtractor(archive, destdir)

    elif archive.endswith('.tar'):
        return tarExtractor(archive, destdir)

    elif archive.endswith('.zip'):
        return zipExtractor(archive, destdir)

    return None

def test():
    '''Test program.
        Usage: extractor archive destdir
    '''
    if len(sys.argv) < 2:
        sys.stderr.write(test.__doc__)
        sys.exit(1)
    archive = sys.argv[1]

    if len(sys.argv) >= 3:
        destdir = sys.argv[2]
    else:
        destdir = os.getcwd()
    
    try:
        extractor = factory(archive, destdir)
    except Error, inst:
        sys.stderr.write(str(inst))
        sys.exit(1)

    if extractor is None:
        sys.stderr.write(("%s is not a valid archive" % (archive)))
        sys.exit(1)
    else:
        extractor.extract()

if __name__ == "__main__":
    test()
