'''
Task Coach - Your friendly task manager
Copyright (C) 2004-2008 Frank Niessink <frank@niessink.com>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

import test
from taskcoachlib import patterns, config
from taskcoachlib.domain import task, date, category, base, effort


class TestFilter(base.Filter):
    def filter(self, items):
        return [item for item in items if item > 'b']


class FilterTests(object):
    def setUp(self):
        self.observable = self.collectionClass(['a', 'b', 'c', 'd'])
        self.filter = TestFilter(self.observable)

    def testLength(self):
        self.assertEqual(2, len(self.filter))

    def testContents(self):
        self.failUnless('c' in self.filter and 'd' in self.filter)

    def testRemoveItem(self):
        self.filter.remove('c')
        self.assertEqual(1, len(self.filter))
        self.failUnless('d' in self.filter)
        self.assertEqual(['a', 'b', 'd'], self.observable)

    def testNotification(self):
        self.observable.append('e')
        self.assertEqual(3, len(self.filter))
        self.failUnless('e' in self.filter)

        
class FilterListTest(FilterTests, test.TestCase):
    collectionClass = patterns.ObservableList


class FilterSetTest(FilterTests, test.TestCase):
    collectionClass = patterns.ObservableSet
    
    
class DummyFilter(base.Filter):
    def filter(self, items):
        return items
    
    def test(self):
        self.testcalled = 1


class StackedFilterTest(test.TestCase):
    def setUp(self):
        self.list = patterns.ObservableList(['a', 'b', 'c', 'd'])
        self.filter1 = DummyFilter(self.list)
        self.filter2 = TestFilter(self.filter1)

    def testDelegation(self):
        self.filter2.test()
        self.assertEqual(1, self.filter1.testcalled)


class ViewFilterTest(test.TestCase):
    def setUp(self):
        self.list = task.TaskList()
        self.filter = task.filter.ViewFilter(self.list)
        self.task = task.Task(subject='task')
        self.dueToday = task.Task(subject='due today', dueDate=date.Today())
        self.dueTomorrow = task.Task(subject='due tomorrow', 
            dueDate=date.Tomorrow())
        self.dueYesterday = task.Task(subject='due yesterday', 
            dueDate=date.Yesterday())
        self.child = task.Task(subject='child')

    def testCreate(self):
        self.assertEqual(0, len(self.filter))

    def testAddTask(self):
        self.filter.append(self.task)
        self.assertEqual(1, len(self.filter))
        
    def testViewActiveTasks(self):
        self.filter.append(self.task)
        self.filter.hideActiveTasks()
        self.assertEqual(0, len(self.filter))

    def testFilterCompletedTask(self):
        self.task.setCompletionDate()
        self.filter.append(self.task)
        self.assertEqual(1, len(self.filter))
        self.filter.hideCompletedTasks()
        self.assertEqual(0, len(self.filter))
        
    def testFilterCompletedTask_RootTasks(self):
        self.task.setCompletionDate()
        self.filter.append(self.task)
        self.filter.hideCompletedTasks()
        self.assertEqual(0, len(self.filter.rootItems()))

    def testMarkTaskCompleted(self):
        self.filter.hideCompletedTasks()
        self.list.append(self.task)
        self.task.setCompletionDate()
        self.assertEqual(0, len(self.filter))

    def testMarkTaskUncompleted(self):
        self.filter.hideCompletedTasks()
        self.task.setCompletionDate()
        self.list.append(self.task)
        self.task.setCompletionDate(date.Date())
        self.assertEqual(1, len(self.filter))
        
    def testChangeCompletionDateOfAlreadyCompletedTask(self):
        self.filter.hideCompletedTasks()
        self.task.setCompletionDate()
        self.list.append(self.task)
        self.task.setCompletionDate(date.Tomorrow())
        self.assertEqual(0, len(self.filter))

    def testFilterDueToday(self):
        self.filter.extend([self.task, self.dueToday])
        self.assertEqual(2, len(self.filter))
        self.filter.setFilteredByDueDate('Today')
        self.assertEqual(1, len(self.filter))
    
    def testFilterDueToday_ShouldIncludeOverdueTasks(self):
        self.filter.append(self.dueYesterday)
        self.filter.setFilteredByDueDate('Today')
        self.assertEqual(1, len(self.filter))

    def testFilterDueToday_ShouldIncludeCompletedTasks(self):
        self.filter.append(self.dueToday)
        self.dueToday.setCompletionDate()
        self.filter.setFilteredByDueDate('Today')
        self.assertEqual(1, len(self.filter))

    def testFilterDueTomorrow(self):
        self.filter.extend([self.task, self.dueTomorrow, self.dueToday])
        self.assertEqual(3, len(self.filter))
        self.filter.setFilteredByDueDate('Tomorrow')
        self.assertEqual(2, len(self.filter))
    
    def testFilterDueWeekend(self):
        dueNextWeek = task.Task(dueDate=date.Today() + \
            date.TimeDelta(days=8))
        self.filter.extend([self.dueToday, dueNextWeek])
        self.filter.setFilteredByDueDate('Workweek')
        self.assertEqual(1, len(self.filter))
        

class ViewFilterInTreeModeTest(test.TestCase):
    def setUp(self):
        self.list = task.TaskList()
        self.filter = task.filter.ViewFilter(self.list, treeMode=True)
        self.task = task.Task()
        self.dueToday = task.Task(dueDate=date.Today())
        self.dueTomorrow = task.Task(dueDate=date.Tomorrow())
        self.dueYesterday = task.Task(dueDate=date.Yesterday())
        self.child = task.Task()
        
    def testCreate(self):
        self.assertEqual(0, len(self.filter))
        
    def testAddTask(self):
        self.filter.append(self.task)
        self.assertEqual(1, len(self.filter))

    def testFilterDueToday(self):
        self.task.addChild(self.dueToday)
        self.list.append(self.task)
        self.filter.setFilteredByDueDate('Today')
        self.assertEqual(2, len(self.filter))
        
    def testFilterOverDueTasks(self):
        self.task.addChild(self.dueYesterday)
        self.list.append(self.task)
        self.filter.hideOverdueTasks()
        self.assertEqual(1, len(self.filter))

    def testFilterOverBudgetTasks(self):
        self.task.setBudget(date.TimeDelta(hours=10))
        self.task.addEffort(effort.Effort(self.task, date.Date(2000,1,1), date.Date(2000,1,2)))
        self.list.append(self.task)
        self.filter.hideOverbudgetTasks()
        self.assertEqual(0, len(self.filter))
        
        
class ViewFilter_HideCompositeTasksTest(test.TestCase):
    def setUp(self):
        self.list = task.TaskList()
        self.filter = task.filter.ViewFilter(self.list)
        self.task = task.Task(subject='task')
        self.child = task.Task(subject='child')
        self.task.addChild(self.child)
        self.filter.append(self.task)

    def testInitial(self):
        self.assertEqual(2, len(self.filter))
                
    def testTurnOn(self):
        self.filter.hideCompositeTasks()
        self.assertEqual([self.child], list(self.filter))

    def testTurnOff(self):
        self.filter.hideCompositeTasks()
        self.filter.hideCompositeTasks(False)
        self.assertEqual(2, len(self.filter))
                
    def testAddChild(self):
        self.filter.hideCompositeTasks()
        grandChild = task.Task(subject='grandchild')
        self.list.append(grandChild)
        self.child.addChild(grandChild)
        self.assertEqual([grandChild], list(self.filter))

    def testRemoveChild(self):
        self.filter.hideCompositeTasks()
        self.list.remove(self.child)
        self.assertEqual([self.task], list(self.filter))

    def _addTwoGrandChildren(self):
        self.grandChild1 = task.Task(subject='grandchild 1')
        self.grandChild2 = task.Task(subject='grandchild 2')
        self.child.addChild(self.grandChild1)
        self.child.addChild(self.grandChild2)
        self.list.extend([self.grandChild1, self.grandChild2])

    def testAddTwoChildren(self):
        self.filter.hideCompositeTasks()
        self._addTwoGrandChildren()
        self.assertEqual([self.grandChild1, self.grandChild2], 
            list(self.filter))

    def testRemoveTwoChildren(self):
        self._addTwoGrandChildren()
        self.filter.hideCompositeTasks()
        self.list.removeItems([self.grandChild1, self.grandChild2])
        self.assertEqual([self.child], list(self.filter))


class SearchFilterTest(test.TestCase):
    def setUp(self):
        self.parent = task.Task(subject='ABC')
        self.child = task.Task(subject='DEF')
        self.parent.addChild(self.child)
        self.list = task.TaskList([self.parent, self.child])
        self.filter = base.SearchFilter(self.list)

    def setSearchString(self, searchString, matchCase=False,
                        includeSubItems=False):
        self.filter.setSearchFilter(searchString, matchCase, includeSubItems)
        
    def testNoMatch(self):
        self.setSearchString('XYZ')
        self.assertEqual(0, len(self.filter))

    def testMatch(self):
        self.setSearchString('AB')
        self.assertEqual(1, len(self.filter))

    def testMatchIsCaseInSensitiveByDefault(self):
        self.setSearchString('abc')
        self.assertEqual(1, len(self.filter))

    def testMatchCaseInsensitive(self):
        self.setSearchString('abc', True)
        self.assertEqual(0, len(self.filter))

    def testMatchWithRE(self):
        self.setSearchString('a.c')
        self.assertEqual(1, len(self.filter))

    def testMatchWithEmptyString(self):
        self.setSearchString('')
        self.assertEqual(2, len(self.filter))

    def testMatchChildDoesNotSelectParentWhenNotInTreeMode(self):
        self.setSearchString('DEF')
        self.assertEqual(1, len(self.filter))

    def testMatchChildAlsoSelectsParentWhenInTreeMode(self):
        self.filter.setTreeMode(True)
        self.setSearchString('DEF')
        self.assertEqual(2, len(self.filter))
        
    def testMatchChildDoesNotSelectParentWhenChildNotInList(self):
        self.list.remove(self.child) 
        self.parent.addChild(self.child) # simulate a child that has been filtered 
        self.setSearchString('DEF')
        self.assertEqual(0, len(self.filter))

    def testAddTask(self):
        self.setSearchString('XYZ')
        taskXYZ = task.Task(subject='subject with XYZ')
        self.list.append(taskXYZ)
        self.assertEqual([taskXYZ], list(self.filter))

    def testRemoveTask(self):
        self.setSearchString('DEF')
        self.list.remove(self.child)
        self.failIf(self.filter)
        
    def testIncludeSubItems(self):
        self.setSearchString('ABC', includeSubItems=True)
        self.assertEqual(2, len(self.filter))


class CategoryFilterHelpers(object):
    def setFilterOnAnyCategory(self):
        self.filter.setFilterOnlyWhenAllCategoriesMatch(False)
        
    def setFilterOnAllCategories(self):
        self.filter.setFilterOnlyWhenAllCategoriesMatch(True)
    
    
class CategoryFilterFixtureAndCommonTests(CategoryFilterHelpers):
    def setUp(self):
        self.parent = task.Task('parent')
        self.parentCategory = category.Category('parent')
        self.parentCategory.addCategorizable(self.parent)
        self.child = task.Task()
        self.childCategory = category.Category('child')
        self.childCategory.addCategorizable(self.child)
        self.parent.addChild(self.child)
        self.unusedCategory = category.Category('unused')
        self.list = task.TaskList([self.parent, self.child])
        self.categories = category.CategoryList([self.parentCategory, 
            self.childCategory, self.unusedCategory])
        self.filter = category.filter.CategoryFilter(self.list, 
            categories=self.categories, treeMode=self.treeMode)
                              
    def testInitial(self):
        self.assertEqual(2, len(self.filter))
        
    def testInitialWhenOneCategoryFiltered(self):
        self.unusedCategory.setFiltered()
        self.filter = category.filter.CategoryFilter(self.list,
            categories=self.categories, treeMode=self.treeMode)
        self.assertEqual(0, len(self.filter))
        
    def testAddFilteredCategory(self):
        newCategory = category.Category('new')
        newCategory.setFiltered()
        self.categories.append(newCategory)
        self.assertEqual(0, len(self.filter))
        
    def testRemoveFilteredCategory(self):
        self.unusedCategory.setFiltered()
        self.categories.remove(self.unusedCategory)
        self.assertEqual(2, len(self.filter))
        
    def testFilterOnCategoryNotPresent(self):
        self.unusedCategory.setFiltered()
        self.assertEqual(0, len(self.filter))
 
    def testFilterOnCategoryParent(self):
        self.parentCategory.setFiltered()
        self.assertEqual(2, len(self.filter))
    
    def testRemoveCategory(self):
        self.parentCategory.setFiltered()
        self.parentCategory.setFiltered(False)
        self.assertEqual(2, len(self.filter))
    
    def testRemoveCategoryThatIsNotCurrentlyUsed(self):
        self.parentCategory.setFiltered(False)
        self.assertEqual(2, len(self.filter))
    
    def testFilterOnAnyCategory(self):
        self.setFilterOnAnyCategory()
        self.parentCategory.setFiltered()
        self.childCategory.setFiltered()
        self.assertEqual(2, len(self.filter))
    
    def testFilterOnAnyCategory_OneUsedAndOneUnusedCategory(self):
        self.setFilterOnAnyCategory()
        self.unusedCategory.setFiltered()
        self.parentCategory.setFiltered()
        self.assertEqual(2, len(self.filter))

    def testFilterOnAllCategories_NoTasksMatch(self):
        self.setFilterOnAllCategories()
        self.unusedCategory.setFiltered()
        self.parentCategory.setFiltered()
        self.assertEqual(0, len(self.filter))

    def testFilterOnAnyCategory_NoCategoriesSelected(self):
        self.setFilterOnAnyCategory()
        self.assertEqual(2, len(self.filter))

    def testFilterOnAllCategories_NoCategoriesSelected(self):
        self.setFilterOnAllCategories()
        self.assertEqual(2, len(self.filter))
        
    def testAddTaskWithFilteredCategory(self):
        self.unusedCategory.setFiltered()
        newTask = task.Task()
        newTask.addCategory(self.unusedCategory)
        self.list.extend([newTask])
        self.assertEqual(1, len(self.filter))
        
    def testRemoveTaskWithFilteredCategory(self):
        self.unusedCategory.setFiltered()
        newTask = task.Task()
        newTask.addCategory(self.unusedCategory)
        self.list.extend([newTask])
        self.list.remove(newTask)
        self.assertEqual(0, len(self.filter))


class CategoryFilterInListModeTest(CategoryFilterFixtureAndCommonTests, 
                                   test.TestCase):
    treeMode = False   
    
    def testFilterOnCategoryChild(self):
        self.childCategory.setFiltered()
        self.assertEqual(1, len(self.filter))
        self.failUnless(self.child in self.filter)
        
    def testFilterOnAllCategories(self):
        self.setFilterOnAllCategories()
        self.parentCategory.setFiltered()
        self.childCategory.setFiltered()
        self.assertEqual(1, len(self.filter))
        self.failUnless(self.child in self.filter)


class CategoryFilterInTreeModeTest(CategoryFilterFixtureAndCommonTests, 
                                   test.TestCase):
    treeMode = True
    
    def testFilterOnCategoryChild(self):
        self.childCategory.setFiltered()
        self.assertEqual(2, len(self.filter))

    def testFilterOnAllCategories(self):
        self.setFilterOnAllCategories()
        self.parentCategory.setFiltered()
        self.childCategory.setFiltered()
        self.assertEqual(2, len(self.filter))

                
class OriginalLengthTest(test.TestCase, CategoryFilterHelpers):
    def setUp(self):
        self.list = task.TaskList()
        self.settings = config.Settings(load=False)
        self.categories = category.CategoryList()
        self.filter = category.filter.CategoryFilter(self.list,
            categories=self.categories, settings=self.settings)
        
    def testEmptyList(self):
        self.assertEqual(0, self.filter.originalLength())
        
    def testNoFilter(self):
        self.list.append(task.Task())
        self.assertEqual(1, self.filter.originalLength())
    '''    
    def testFilter(self):
        aTask = task.Task()
        aTask.addCategory('test')
        self.list.append(aTask)
        self.addCategory('nottest')
        self.assertEqual(0, len(self.filter))
        self.assertEqual(1, self.filter.originalLength())
    '''
     
