'''
Task Coach - Your friendly task manager
Copyright (C) 2004-2008 Frank Niessink <frank@niessink.com>

Task Coach is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Task Coach is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

import test, sets
from unittests import asserts
from taskcoachlib import patterns
from taskcoachlib.domain import task, effort, date


class TaskListTest(test.TestCase):
    def setUp(self):
        self.taskList = task.TaskList()
        self.task1 = task.Task(dueDate=date.Date(2010,1,1))
        self.task2 = task.Task(dueDate=date.Date(2011,1,1))
        self.task3 = task.Task()
    
    def testNrCompleted(self):
        self.assertEqual(0, self.taskList.nrCompleted())
        self.taskList.append(self.task1)
        self.assertEqual(0, self.taskList.nrCompleted())
        self.task1.setCompletionDate()
        self.assertEqual(1, self.taskList.nrCompleted())
    
    def testNrOverdue(self):
        self.assertEqual(0, self.taskList.nrOverdue())
        self.taskList.append(self.task1)
        self.assertEqual(0, self.taskList.nrOverdue())
        self.task1.setDueDate(date.Date(1990, 1, 1))
        self.assertEqual(1, self.taskList.nrOverdue())

    def testAllCompleted(self):
        self.failIf(self.taskList.allCompleted())
        self.task1.setCompletionDate()
        self.taskList.append(self.task1)
        self.failUnless(self.taskList.allCompleted())

    def testNrDueToday(self):
        self.assertEqual(0, self.taskList.nrDueToday())
        self.taskList.append(task.Task(dueDate=date.Today()))
        self.assertEqual(1, self.taskList.nrDueToday())
        
    def testNrBeingTracked(self):
        self.assertEqual(0, self.taskList.nrBeingTracked())
        activeTask = task.Task()
        activeTask.addEffort(effort.Effort(activeTask))
        self.taskList.append(activeTask)
        self.assertEqual(1, self.taskList.nrBeingTracked())

    def testMinDate_EmptyTaskList(self):
        self.assertEqual(date.Date(), self.taskList.minDate())
        
    def testMinDate_TaskWithoutDates(self):
        self.taskList.append(task.Task(startDate=date.Date()))
        self.assertEqual(date.Date(), self.taskList.minDate())
        
    def testMinDate_TaskWithStartDate(self):
        self.taskList.append(task.Task())
        self.assertEqual(date.Today(), self.taskList.minDate())
        
    def testMinDate_TaskWithDueDate(self):
        self.taskList.append(task.Task(dueDate=date.Yesterday()))
        self.assertEqual(date.Yesterday(), self.taskList.minDate())

    def testMinDate_TaskWithCompletionDate(self):
        self.taskList.append(task.Task(completionDate=date.Yesterday()))
        self.assertEqual(date.Yesterday(), self.taskList.minDate())

    def testMaxDate_EmptyTaskList(self):
        self.assertEqual(date.Date(), self.taskList.maxDate())
        
    def testMaxDate_TaskWithoutDates(self):
        self.taskList.append(task.Task(startDate=date.Date()))
        self.assertEqual(date.Date(), self.taskList.maxDate())
        
    def testMaxDate_TaskWithStartDate(self):
        self.taskList.append(task.Task())
        self.assertEqual(date.Today(), self.taskList.maxDate())

    def testMaxDate_TaskWithDueDate(self):
        self.taskList.append(task.Task(dueDate=date.Tomorrow()))
        self.assertEqual(date.Tomorrow(), self.taskList.maxDate())
    
    def testMaxDate_TaskWithCompletionDate(self):
        self.taskList.append(task.Task(completionDate=date.Tomorrow()))
        self.assertEqual(date.Tomorrow(), self.taskList.maxDate())
        
    def testOriginalLength(self):
        self.assertEqual(0, self.taskList.originalLength())

    def testMinPriority_EmptyTaskList(self):
        self.assertEqual(0, self.taskList.minPriority())
        
    def testMinPriority_OneTaskWithDefaultPriority(self):
        self.taskList.append(self.task1)
        self.assertEqual(self.task1.priority(), self.taskList.minPriority())
        
    def testMinPriority_OneTaskWithNonDefaultPriority(self):
        self.taskList.append(task.Task(priority=-5))
        self.assertEqual(-5, self.taskList.minPriority())
        
    def testMinPriority_TwoTasks(self):
        self.taskList.extend([task.Task(priority=3), task.Task(priority=5)])
        self.assertEqual(3, self.taskList.minPriority())
        
    def testMaxPriority_EmptyTaskList(self):
        self.assertEqual(0, self.taskList.maxPriority())
        
    def testMaxPriority_OneTaskWithDefaultPriority(self):
        self.taskList.append(self.task1)
        self.assertEqual(self.task1.priority(), self.taskList.maxPriority())
        
    def testMaxPriority_OneTaskWithNonDefaultPriority(self):
        self.taskList.append(task.Task(priority=-5))
        self.assertEqual(-5, self.taskList.maxPriority())
        
    def testMaxPriority_TwoTasks(self):
        self.taskList.extend([task.Task(priority=3), task.Task(priority=5)])
        self.assertEqual(5, self.taskList.maxPriority())

