/**
 * Cleversafe open-source code header - Version 1.1 - December 1, 2006
 *
 * Cleversafe Dispersed Storage(TM) is software for secure, private and
 * reliable storage of the world's data using information dispersal.
 *
 * Copyright (C) 2005-2007 Cleversafe, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 *
 * Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
 * Chicago IL 60616
 * email licensing@cleversafe.org
 *
 * Author: Greg Dhuse <gdhuse@cleversafe.com>
 *
 */
#ifndef DSD_H
#define DSD_H

/* Uncomment for debugging */
#if 0
#define DSD_DEBUG
#endif

#include "dsd_common.h"

#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/blkdev.h>
#include <linux/bio.h>
#include <linux/random.h>
#include <linux/ioctl.h>
#include <linux/kthread.h>
#include <linux/hdreg.h>
#include <linux/fs.h>
#include <linux/syscalls.h>
#include <linux/socket.h>
#include <net/sock.h>
#include <linux/in.h>

#define DSD_MAX_OUTSTANDING_REQUESTS 20

extern int dsd_bus_major;           /* Bus major number */
extern int dsd_major;               /* Device major number */

/* Device object */
struct dsd_dev
{
   int device_num;                  /* Device number */
   atomic_t use_count;              /* Number of current device users */
   atomic_t pending_work;           /* Number of pending work items */
   atomic_t error;                  /* When < 1, fail all IO requests */
   
   uint64_t sectors;                /* Number of sectors in device */
   uint32_t sector_size;            /* Size of a sector in bytes */

   spinlock_t lock;                 /* Request queue lock */
   struct request_queue* queue;     /* Device request queue */
   struct gendisk* gd;              /* Gendisk structure */

   struct socket* sock;             /* Daemon communication socket */
   struct task_struct* thread;      /* Device kthread */

   struct dsd_dev* next;            /* Linked list */
};

/* Device list */
extern int dsd_next_device;             /* Next device number */
extern struct dsd_dev* dsd_devices;     /* Linked list head */

/* Module initialization */
int  __init dsd_init( void );
void __exit dsd_exit( void );
int  __init dsd_bus_init( void );
void __exit dsd_bus_exit( void );
int  __init dsd_net_init( void );
void __exit dsd_net_exit( void );

/* Device management */
int dsd_add_device( uint64_t sectors, 
                    uint32_t sector_size, 
                    struct socket* sk,
                    struct dsd_dev** dev_ptr );
int dsd_remove_device( struct dsd_dev* dev );

/* Daemon communication */
struct socket* dsd_connect( uint32_t ip_addr, uint16_t port );
int dsd_recv( struct socket* sk, uint8_t** msg );
int dsd_send( struct socket* sk, int msg_type, uint8_t* msg, int len );
int dsd_send_atomic( struct socket* sk, 
                     int msg_type, 
                     uint8_t* msg, 
                     int len,
                     atomic_t* pending_work );

/* 64-bit byteswapping functions */
static inline uint64_t htonll( uint64_t val )
{
   int i;
   uint64_t ret = 0;
   uint8_t* bs = (uint8_t*)&val;
   uint8_t* bd = (uint8_t*)&ret + 7;

   for( i = 0; i < 8; ++i )
   {
      *bd-- = *bs++;
   }
      
   return ret;
}

static inline uint64_t ntohll( uint64_t val )
{
   return htonll( val );
}

/* Debugging */
#ifdef DSD_DEBUG
void dsd_hexdump( uint8_t* buf, unsigned int len );
#endif

#endif

