/**
 * Cleversafe open-source code header - Version 1.1 - December 1, 2006
 *
 * Cleversafe Dispersed Storage(TM) is software for secure, private and
 * reliable storage of the world's data using information dispersal.
 *
 * Copyright (C) 2005-2007 Cleversafe, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 * USA.
 *
 * Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
 * Chicago IL 60616
 * email licensing@cleversafe.org
 *
 * Author: Greg Dhuse <gdhuse@cleversafe.com>
 *
 */

#include "org_cleversafe_block_SystemBlockDevice.h"
#include "dsd_common.h"

#include <stdlib.h>
#include <string.h>

#ifdef linux

#include <sys/ioctl.h>
#include <arpa/inet.h>
#define DSD_BUS_FILE "/dev/" DSD_BUS_DEVICE_NAME
#define max(a,b) ( (a < b) ? b : a )

#elif defined(WIN32)

#include <windows.h>
#include <winsock.h>
#include <setupapi.h>
#define snprintf _snprintf
   
#endif



/* Exceptions */
static void ThrowByName( JNIEnv* env, const char* name, const char* msg );


/**
 * Initiate creation of a system device with the ioctl: 
 *  DSD_BUS_IOCTL_CREATE_DEVICE
 *
 * @param env JNI environment
 * @param this SystemBlockDevice instance
 * @param numBlocks IN - Number of blocks in the virtual device
 * @param blockSize IN - Block size in bytes
 * @param ipAddress IN - IP address of daemon in network byte order
 * @param port IN - Port daemon is listening on
 * @param deviceName OUT - Name of system device (eg. /dev/dsd0)
 * @throws IOException
 */
JNIEXPORT void JNICALL 
Java_org_cleversafe_block_SystemBlockDevice_createSystemDevice(
   JNIEnv* env,            
   jobject this,           
   jlong numBlocks,        /* IN */
   jint blockSize,         /* IN */
   jbyteArray ipAddress,   /* IN */
   jint port               /* IN */ ) /* throws IOException */
{
#ifdef linux
/*******************************************************************************
 * BEGIN LINUX
 *******************************************************************************/

   int status;
   FILE* device;
   uint8_t* data;
   jstring retval;
   jclass thisClass;
   jfieldID deviceNameFID;
   struct dsd_bus_ioctl_create_device* request;
   struct dsd_bus_ioctl_create_device_rsp* response;

   /**
    * The request and response will use the same memory, so we need
    * to allocate enough to hold the larger of the two
    */ 
   data = malloc(max( 
      sizeof(struct dsd_bus_ioctl_create_device), 
      sizeof(struct dsd_bus_ioctl_create_device_rsp) 
   ));
   if( !data )
   {
      jclass exception = (*env)->FindClass( env, "java/lang/OutOfMemoryError" );
      if( exception )
      {
         (*env)->ThrowNew( env, exception, "Native C malloc failed" );
      }
      return;
   }

   /* Prepare request */
   request = (struct dsd_bus_ioctl_create_device*)data;
   memset( request, 0, sizeof(struct dsd_bus_ioctl_create_device) );
   request->sectors     = numBlocks;
   request->sector_size = blockSize;
   request->port        = port;

   /* The IP address comes in network byte order, but we ignore endianness */
   memcpy( &request->ip_addr, 
           (uint8_t*)((*env)->GetByteArrayElements( env, ipAddress, NULL )), 
           sizeof(request->ip_addr) );
   request->ip_addr = ntohl( request->ip_addr );
    

   /* Send ioctl */
   device = fopen( DSD_BUS_FILE, "r+" );
   if( !device )
   {
      jclass exception = (*env)->FindClass( env, "java/io/FileNotFoundException" );
      if( exception )
      {
         (*env)->ThrowNew( env, exception, 
            "Bus device not found: " DSD_BUS_FILE );
      }

      free( data );
      return;
   }

   status = ioctl( fileno( device ), 
                   DSD_BUS_IOCTL_CREATE_DEVICE, 
                   (void*)request );
   fclose( device );
   if( status < 0 )
   {
      char msg[64];
      jclass exception = (*env)->FindClass( env, "java/io/IOException" );

      if( exception )
      {
         snprintf( msg, 64, "ioctl() failed: %d", status );
         (*env)->ThrowNew( env, exception, msg );
      }

      free( data );
      return;
   }

   /* Process response */
   response = (struct dsd_bus_ioctl_create_device_rsp*)data;
   if( response->status < 0 )
   {
      char msg[64];
      jclass exception = (*env)->FindClass( env, "java/io/IOException" );

      if( exception )
      {
         snprintf( msg, 64, "ioctl() returned error: %d", response->status );
         (*env)->ThrowNew( env, exception, msg );
      }

      free( data );
      return;
   }

   // Set device name
   thisClass = (*env)->GetObjectClass( env, this );
   deviceNameFID = (*env)->GetFieldID( env, 
                                       thisClass, 
                                       "deviceName", 
                                       "Ljava/lang/String;" );
   if( deviceNameFID )
   {
      jstring deviceName = (*env)->NewStringUTF( env, response->device_name );
      if( !deviceName )
      {
         return;  /* Out of menory */
      }
      (*env)->SetObjectField( env, this, deviceNameFID, deviceName );
   }
   
   free( data );



#elif defined(WIN32)
/*******************************************************************************
 * BEGIN WINDOWS
 *******************************************************************************/

   int bytes;
   BOOL status;
   HANDLE device;
   jstring retval;
   jclass thisClass;
   HDEVINFO classDevices;
   jfieldID deviceNameFID;
   SP_DEVICE_INTERFACE_DATA deviceInformation;
   PSP_DEVICE_INTERFACE_DETAIL_DATA deviceDetails;
   struct dsd_bus_ioctl_create_device request;
   struct dsd_bus_ioctl_create_device_rsp response;

   /* Prepare request */
   memset( &request, 0, sizeof(request) );
   request.sectors      = numBlocks;
   request.sector_size  = blockSize;
   request.port         = port;

   /* The IP address comes in network byte order, but we ignore endianness */
   memcpy( &request.ip_addr, 
           (uint8_t*)((*env)->GetByteArrayElements( env, ipAddress, NULL )), 
           sizeof(request.ip_addr) );
   request.ip_addr = ntohl( request.ip_addr );

   /* Get the bus device's interface name */
   classDevices = SetupDiGetClassDevs( 
      &GUID_DEVINTERFACE_BUSENUM_DISPERSED_STORAGE, 
      NULL, 
      NULL, 
      DIGCF_DEVICEINTERFACE | DIGCF_PRESENT );
   if( classDevices == INVALID_HANDLE_VALUE )
   {
      jclass exception = (*env)->FindClass( env, "java/io/FileNotFoundException" );
      if( exception )
      {
         char msg[64];
         snprintf( msg, 64, "Interface not found(%d)", GetLastError() );
         (*env)->ThrowNew( env, exception, msg );
      }
      return;
   }

   deviceInformation.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);
   status = SetupDiEnumDeviceInterfaces( 
      classDevices, 
      NULL,
      &GUID_DEVINTERFACE_BUSENUM_DISPERSED_STORAGE,
      0,  /* Only one interface */
      &deviceInformation );
   if( !status )
   {
      jclass exception = (*env)->FindClass( env, "java/io/FileNotFoundException" );
      if( exception )
      {
         char msg[64];
         snprintf( msg, 64, "Interface could not be enumerated(%d)", GetLastError() );
         (*env)->ThrowNew( env, exception, msg );
      }
      return;
   }

   /* Determine required buffer size */
   SetupDiGetDeviceInterfaceDetail(
      classDevices,
      &deviceInformation,
      NULL,
      0,
      &bytes,
      NULL );

   deviceDetails = malloc( bytes + sizeof(TCHAR) );
   if( !deviceDetails )
   {
      jclass exception = (*env)->FindClass( env, "java/lang/OutOfMemoryError" );
      if( exception )
      {
         (*env)->ThrowNew( env, exception, "Native C malloc failed" );
      }
      return;
   }
   deviceDetails->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

   status = SetupDiGetDeviceInterfaceDetail(
      classDevices,
      &deviceInformation,
      deviceDetails,
      bytes,
      NULL,
      NULL );
   if( !status )
   {
      jclass exception = (*env)->FindClass( env, "java/io/FileNotFoundException" );
      if( exception )
      {
         char msg[64];
         snprintf( msg, 64, 
            "Interface details could not be retrieved(%d)", 
            GetLastError() );
         (*env)->ThrowNew( env, exception, msg );
      }
      return;
   }
    
   /* Send ioctl */
   device = CreateFile( deviceDetails->DevicePath,
                        0, //GENERIC_READ | GENERIC_WRITE,
                        0, //FILE_SHARE_READ | FILE_SHARE_WRITE,                                           
                        NULL,                                 
                        OPEN_EXISTING,                       
                        0,                                     
                        NULL );

   if( device == INVALID_HANDLE_VALUE )
   {
      jclass exception = (*env)->FindClass( env, "java/io/FileNotFoundException" );
      if( exception )
      {
         char msg[128];
         snprintf( msg, 128, "Bus device not found: %s(%d)", 
            deviceDetails->DevicePath, GetLastError() );
         (*env)->ThrowNew( env, exception, msg );
      }
      free( deviceDetails );
      deviceDetails = NULL;
      return;
   }
   free( deviceDetails );
   deviceDetails = NULL;

   status = DeviceIoControl( device,
                             DSD_BUS_IOCTL_CREATE_DEVICE,
                             &request, sizeof(request),     /* Input */
                             &response, sizeof(response),   /* Output */
                             &bytes,
                             NULL );
   CloseHandle( device );
   if( !status )
   {
      char msg[64];
      jclass exception = (*env)->FindClass( env, "java/io/IOException" );

      if( exception )
      {
         snprintf( msg, 64, "ioctl() failed(%d,%d)", GetLastError(), bytes );
         (*env)->ThrowNew( env, exception, msg );
      }

      return;
   }

   /* Process response */
   if( response.status < 0 )
   {
      char msg[64];
      jclass exception = (*env)->FindClass( env, "java/io/IOException" );

      if( exception )
      {
         snprintf( msg, 64, "ioctl() returned error: %d", response.status );
         (*env)->ThrowNew( env, exception, msg );
      }

      return;
   }

   // Set device name
   thisClass = (*env)->GetObjectClass( env, this );
   deviceNameFID = (*env)->GetFieldID( env, 
                                       thisClass, 
                                       "deviceName", 
                                       "Ljava/lang/String;" );
   if( deviceNameFID )
   {
      jstring deviceName = (*env)->NewStringUTF( env, response.device_name );
      if( !deviceName )
      {
         return;  /* Out of menory */
      }
      (*env)->SetObjectField( env, this, deviceNameFID, deviceName );
   }
   
#endif
}

/**
 * Throw an exception by name, from the book:
 *  The Java Native Interface Programmer’s Guide and Specification
 * 
 * @param env JNI environment
 * @param name Exception class name (eg. IOException)
 * @param msg Message string
 */
static void
ThrowByName( JNIEnv *env, const char *name, const char *msg )
{
   jclass cls = (*env)->FindClass(env, name);

   /* if cls is NULL, an exception has already been thrown */
   if( cls != NULL ) 
   {
      (*env)->ThrowNew(env, cls, msg);
   }

   /* free the local ref */
   (*env)->DeleteLocalRef(env, cls);
}

