//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 20, 2007
//---------------------

package org.cleversafe.layer.access;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.security.KeyPair;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.StandardMBean;

import org.apache.log4j.Logger;
import org.apache.log4j.extras.DOMConfigurator;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.authentication.PropertiesFileCredentialsManager;
import org.cleversafe.authentication.exceptions.CredentialsException;
import org.cleversafe.authentication.exceptions.CredentialsIOException;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.daemon.Daemonizer;
import org.cleversafe.jmx.JMXService;
import org.cleversafe.layer.access.cli.ExitStatusException;
import org.cleversafe.layer.access.exceptions.AccessConfigurationException;
import org.cleversafe.layer.access.exceptions.AccessLayerException;
import org.cleversafe.layer.access.exceptions.DuplicateServiceException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceStillRunningException;
import org.cleversafe.layer.access.jmx.GridAccessMBean;
import org.cleversafe.layer.access.jmx.VaultStatusMBean;
import org.cleversafe.util.JSAPCommandLineParser;
import org.cleversafe.util.Log4jReloader;
import org.cleversafe.util.Tuple3;
import org.cleversafe.vault.Vault;
import org.hibernate.exception.GenericJDBCException;

import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.StringParser;

// TODO: Describe class or interface
public class GridAccessDaemon extends Daemonizer implements Runnable
{
   private static Logger _logger = Logger.getLogger(GridAccessDaemon.class);

   private static final String COMMAND_NAME = "dsnet-accesser";

   private static final String JMX_AGENT_HOST_OPTION = "jmx_host";
   private static final String JMX_AGENT_PORT_OPTION = "jmx_port";
   private static final String DIRECTORY_URL_OPTION = "directory";

   public static ObjectName GRID_ACCESS_OBJECT_NAME;
   public static ObjectName VAULT_STATUS_OBJECT_NAME;

   public static final int OPTION_ERROR = 255;
   public static final int SECURITY_ERROR = 30;
   public static final int CONF_ERROR = 20;
   public static final int IO_ERROR = 10;
   public static final int UNKNOWN_ERROR = 1;

   private final String jmxHost;
   private final int jmxPort;
   private final URI directory;

   private GridAccessManager manager;

   static
   {
      try
      {
         GRID_ACCESS_OBJECT_NAME =
               new ObjectName("org.cleversafe.layer.access.jmx:type=GridAccess");
         VAULT_STATUS_OBJECT_NAME =
               new ObjectName("org.cleversafe.layer.access.jmx:type=VaultStatus");
      }
      catch (MalformedObjectNameException e)
      {
         _logger.fatal("Static object name not properly formed", e);
      }
   }

   ////////////////////////////////////////////////////////////////////////////////////////////////
   // JMX MBeans

   private class VaultStatus implements VaultStatusMBean
   {

      public String createVault(String descriptor, String name) throws Exception
      {
         try
         {
            Credentials credentials = _getCredentials(directory);
            UUID identifier =
                  manager.getVaultManager().createVault(name, descriptor,
                        credentials.getAccountIdentifier(), _getCredentialsManager(directory));
            return identifier.toString();
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      @SuppressWarnings("unchecked")
      public void deleteVault(String vaultIdentifier) throws Exception
      {
         try
         {
            final UUID vid = UUID.fromString(vaultIdentifier);

            Credentials credentials = _getCredentials(directory);
            for (UUID id : manager.getVaultIdentifiers(true))
            {
               if (id.equals(vid))
                  throw new ServiceStillRunningException("Vault is currently running: " + id);
            }
            try
            {
               while (true)
               {
                  Service service = manager.getService(vid);
                  _logger.debug("Removing " + service.getName() + " for vault " + vaultIdentifier);
                  ((ServiceInterface<Service>) manager.getServiceInterface(service.getType())).remove(service);
               }
            }
            catch (ServiceNotFoundException e)
            {
               _logger.debug(String.format("Reached end of service list for vault %s",
                     vaultIdentifier), e);
            }

            manager.getVaultManager().deleteVault(UUID.fromString(vaultIdentifier),
                  credentials.getAccountIdentifier(), _getCredentialsManager(directory));
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public String getVaultDescriptor(String vaultIdentifier) throws Exception
      {
         try
         {
            return manager.getVaultManager().getVaultDescriptor(UUID.fromString(vaultIdentifier));
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public String getVaultIdentifier(String vaultName) throws Exception
      {
         try
         {
            return manager.getVaultManager().getVaultIdentifier(vaultName).toString();
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public String[] getVaultList() throws Exception
      {
         try
         {
            List<UUID> ids = manager.getVaultManager().getVaults();
            String[] names = new String[ids.size()];
            for (int i = 0; i < ids.size(); i++)
            {
               names[i] = ids.get(i).toString();
            }
            return names;
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public String getVaultName(String vaultIdentifier) throws Exception
      {
         try
         {
            return manager.getVaultManager().getVaultName(UUID.fromString(vaultIdentifier));
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public boolean isVaultLoaded(String vaultIdentifier) throws Exception
      {
         try
         {
            return this.getLoadedVaultSet().contains(vaultIdentifier);
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      private Set<String> getLoadedVaultSet() throws Exception
      {
         try
         {
            Set<String> vaults = new HashSet<String>();
            List<ServiceInterface<Service>> ints = manager.getServiceInterfaces();
            for (ServiceInterface<Service> i : ints)
            {
               List<Service> services = i.getServices();
               for (Service s : services)
               {
                  vaults.add(s.getVaultIdentifier().toString());
               }
            }
            return vaults;
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }

      public String[] getLoadedVaults() throws Exception
      {
         try
         {
            return this.getLoadedVaultSet().toArray(new String[0]);
         }
         catch (Exception ex)
         {
            _logger.debug("Exception detected:", ex);
            throw ex;
         }
      }
   }

   private class GridAccess implements GridAccessMBean
   {

      @SuppressWarnings("unchecked")
      public synchronized void createService(String type, String name, String vault)
            throws Exception
      {
         try
         {
            // Check that client provided a valid service name for the given service type.
            StringParser nameParser =
                  (StringParser) ConfigurationFactory.getBindingsProvider(
                        ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("StringParser",
                        type);
            name = (String) nameParser.parse(name);

            UUID vaultIdentifier = UUID.fromString(vault);
            Credentials credentials = _getCredentials(directory);
            Vault vaultObj =
                  manager.getVaultManager().loadVault(vaultIdentifier,
                        credentials.getAccountIdentifier(), _getCredentialsManager(directory));
            Service service = _getImpl(Service.class, type).load(name, vaultObj);
            try
            {
               ((ServiceInterface<Service>) manager.getServiceInterface(type)).add(service);
            }
            catch (DuplicateServiceException e)
            {
               // Put a more descriptive message in here for end user consumption
               throw new DuplicateServiceException(type + " service already exists: " + name, e);
            }
         }
         catch (Exception e)
         {
            _logger.debug("Exception during JMX call", e);
            throw e;
         }
      }

      public synchronized void deleteServiceInterface(String type) throws Exception
      {
         manager.deleteServiceInterface(type);
      }

      public synchronized String[] getServiceInterfaces() throws Exception
      {
         Set<String> sis = new HashSet<String>();
         for (ServiceInterface<Service> si : manager.getServiceInterfaces())
         {
            sis.add(si.getType());
         }
         return sis.toArray(new String[0]);
      }

      @SuppressWarnings("unchecked")
      public String[] getServiceInterfaceSettings(String type) throws Exception
      {
         List<String> settings = new ArrayList<String>();
         ServiceInterface<Service> si =
               (ServiceInterface<Service>) manager.getServiceInterface(type);
         settings.add(si.getType());
         settings.add(si.getHost());
         settings.add(Integer.toString(si.getPort()));
         settings.add(Boolean.toString(si.isStartsAutomatically()));
         settings.add(Boolean.toString(si.isRunning()));
         return settings.toArray(new String[5]);
      }

      public String[][] getServiceSettings(String type) throws Exception
      {
         List<String[]> settings = new ArrayList<String[]>();
         Set<Tuple3<String, String, UUID>> services = manager.getServiceInfo(false);
         for (Tuple3<String, String, UUID> service : services)
         {
            if (!service.getFirst().equals(type))
               continue;
            String vaultName = manager.getVaultManager().getVaultName(service.getThird());
            settings.add(new String[]{
                  type, service.getSecond(), vaultName, service.getThird().toString()
            });
         }
         return settings.toArray(new String[settings.size()][]);
      }

      public String[] getServiceSettings(String type, String name) throws Exception
      {
         // Check that client provided a valid service name for the given service type.
         StringParser nameParser =
               (StringParser) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("StringParser", type);
         name = (String) nameParser.parse(name);

         Set<Tuple3<String, String, UUID>> services = manager.getServiceInfo(false);
         for (Tuple3<String, String, UUID> service : services)
         {
            if (service.getFirst().equals(type) && service.getSecond().equals(name))
            {
               String vaultName = manager.getVaultManager().getVaultName(service.getThird());
               return new String[]{
                     type, name, vaultName, service.getThird().toString()
               };
            }
         }
         throw new ServiceNotFoundException("No such " + type + " service: " + name);
      }

      @SuppressWarnings("unchecked")
      public void deleteService(String type, String name) throws Exception
      {
         // Check that client provided a valid service name for the given service type.
         StringParser nameParser =
               (StringParser) ConfigurationFactory.getBindingsProvider(
                     ConfigurationFactory.XML_CONFIG_TYPE).getImplementation("StringParser", type);
         name = (String) nameParser.parse(name);

         Service service = manager.getService(type, name);
         ((ServiceInterface<Service>) manager.getServiceInterface(type)).remove(service);
      }

      public synchronized void createServiceInterface(
            String type,
            String host,
            int port,
            boolean autostart) throws Exception
      {
         manager.createServiceInterface(type, host, port, autostart);
      }

      @SuppressWarnings("unchecked")
      public void changeServiceInterface(String type, boolean autostart) throws Exception
      {
         ServiceInterface<Service> si =
               (ServiceInterface<Service>) manager.getServiceInterface(type);
         si.setStartsAutomatically(autostart);
      }

      @SuppressWarnings("unchecked")
      private List<Service> getServiceObjs(String type) throws Exception
      {
         ServiceInterface<Service> serviceInterface =
               (ServiceInterface<Service>) manager.getServiceInterface(type);
         return serviceInterface.getServices();
      }

      public synchronized String[] getServices() throws Exception
      {
         Set<String> services = new HashSet<String>();
         List<ServiceInterface<Service>> interfaces = manager.getServiceInterfaces();
         for (ServiceInterface<Service> in : interfaces)
         {
            for (Service service : in.getServices())
            {
               services.add(service.getType() + "," + service.getName());
            }
         }
         return services.toArray(new String[0]);
      }

      public synchronized String[] getServices(String type) throws Exception
      {
         List<Service> services = getServiceObjs(type);
         String[] serviceNames = new String[services.size()];
         for (int i = 0; i < services.size(); i++)
         {
            serviceNames[i] = services.get(i).getName();
         }
         return serviceNames;
      }

      public synchronized void startServiceInterface(String type) throws Exception
      {
         manager.getServiceInterface(type).start();
      }

      public synchronized void stopServiceInterface(String type) throws Exception
      {
         manager.getServiceInterface(type).stop();
      }

   }

   ////////////////////////////////////////////////////////////////////////////////////////////////
   // FIXME: Static account methods. These methods will be replaced once a full account framework is
   // complete

   protected final static String ACCOUNTS_PATH = "accounts";
   protected final static String ACCOUNT_NAME = "dsnet";
   protected final static String ACCOUNT_PASSWORD = "dsnet";
   protected final static String KEY_ALGORITHM = "RSA";
   protected final static int KEY_SIZE = 1024;
   protected final static String SIGNATURE_ALGORITHM = "SHA1withRSA";

   protected static Credentials _getCredentials(URI directory) throws CredentialsException
   {
      CredentialsManager manager = _getCredentialsManager(directory);
      UUID accountIdentifier = manager.getAccountIdentifier(ACCOUNT_NAME);
      return manager.getCredentials(accountIdentifier);
   }

   protected static KeyPair _getKeyPair(URI directory) throws CredentialsException
   {
      CredentialsManager manager = _getCredentialsManager(directory);
      UUID accountIdentifier = manager.getAccountIdentifier(ACCOUNT_NAME);
      return manager.getAccountKeyPair(accountIdentifier);
   }

   protected static CredentialsManager _getCredentialsManager(URI directory)
         throws CredentialsException
   {
      File accountPath = new File(directory.getPath() + File.separator + ACCOUNTS_PATH);
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(accountPath.getAbsolutePath());

      try
      {
         if (!accountPath.exists())
            throw new FileNotFoundException("Account database does not exist");

         CredentialsManager manager = new PropertiesFileCredentialsManager();
         // check that manager can load account key pair
         UUID accountIdentifier = manager.getAccountIdentifier(ACCOUNT_NAME);

         File keyStoreFile =
               new File(directory.getPath() + File.separator + GridAccessDaemon.ACCOUNTS_PATH
                     + File.separator + accountIdentifier.toString()
                     + PropertiesFileCredentialsManager.KEYSTORE_EXTENSION);
         File credentialsFile =
               new File(directory.getPath() + File.separator + GridAccessDaemon.ACCOUNTS_PATH
                     + File.separator + accountIdentifier.toString()
                     + PropertiesFileCredentialsManager.CREDENTIALS_EXTENSION);

         if (!keyStoreFile.exists() && !credentialsFile.exists())
            throw new FileNotFoundException("account does not exist: " + accountIdentifier);

         if (!keyStoreFile.exists() || !credentialsFile.exists())
            throw new CredentialsIOException("one or more account files do not exist for "
                  + accountIdentifier);
      }
      catch (FileNotFoundException e)
      {
         _logger.debug("Creating account database", e);
         PropertiesFileCredentialsManager.create(ACCOUNT_NAME, ACCOUNT_PASSWORD, KEY_ALGORITHM,
               KEY_SIZE, SIGNATURE_ALGORITHM);
      }
      catch (CredentialsException e)
      {
         throw new CredentialsIOException("account database corrupt, repair or remove database", e);
      }

      _logger.trace("Loading credentials manager from " + accountPath);

      return new PropertiesFileCredentialsManager();
   }

   ////////////////////////////////////////////////////////////////////////////////////////////////

   public GridAccessDaemon(String args[]) throws ExitStatusException
   {
      super();
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));

      JSAPResult result = null;
      try
      {
         JSAPCommandLineParser parser =
               new JSAPCommandLineParser(GridAccessDaemon.class, "grid_access_daemon.jsap",
                     COMMAND_NAME + " [OPTION]...", "Starts access device");
         result = parser.parse(args);
      }
      catch (IOException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         throw new ExitStatusException(IO_ERROR, e);
      }
      catch (JSAPException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         throw new ExitStatusException(CONF_ERROR, e);
      }

      if (result == null)
      {
         // Parser handled argument exception
         throw new ExitStatusException("Option parser error encountered", OPTION_ERROR);
      }

      this.jmxHost = result.getString(JMX_AGENT_HOST_OPTION);
      this.jmxPort = result.getInt(JMX_AGENT_PORT_OPTION);
      URI dir = URI.create(result.getString(DIRECTORY_URL_OPTION));
      if (dir.getScheme() == null)
      {
         // directory path was probably actually a file
         this.directory = (new File(result.getString(DIRECTORY_URL_OPTION)).toURI());
      }
      else
      {
         this.directory = dir;
      }
   }

   public GridAccessDaemon(String jmxHost, int jmxPort, URI directory)
   {
      super();
      this.jmxHost = jmxHost;
      this.jmxPort = jmxPort;
      this.directory = directory;
   }

   @Override
   protected void shutdown()
   {
      try
      {
         if (this.manager != null)
         {
            this.manager.shutdown();
         }
      }
      catch (ServiceInterfaceStartStopException e)
      {
         _logger.error("Unable to stop all service interfaces cleanly", e);
      }
      catch (ServiceStartStopException e)
      {
         _logger.error("Unable to stop all services cleanly", e);
      }
   }

   private static <T> T _getImpl(final Class<T> interfaceClass, final String referral)
         throws AccessConfigurationException
   {
      try
      {
         return ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getImplementation(
               interfaceClass, referral);
      }
      catch (ConfigurationItemNotDefinedException e)
      {
         throw new AccessConfigurationException("could not load '" + referral + "' "
               + interfaceClass.getName() + " object");
      }
      catch (ObjectInstantiationException e)
      {
         throw new AccessConfigurationException("could not load '" + referral + "' "
               + interfaceClass.getName() + " object");
      }
      catch (ObjectInitializationException e)
      {
         throw new AccessConfigurationException("could not load '" + referral + "' "
               + interfaceClass.getName() + " object");
      }
      catch (ConfigurationLoadException e)
      {
         throw new AccessConfigurationException("could not load '" + referral + "' "
               + interfaceClass.getName() + " object");
      }
   }

   private static GridAccessManager setup(URI directory) throws AccessLayerException
   {
      _logger.debug("Loading grid access manager: " + directory.toString());
      return _getImpl(GridAccessManager.class, directory.getScheme()).load(directory);
   }

   public void run() throws ExitStatusException
   {
      try
      {
         this.manager = setup(this.directory);
      }
      catch (AccessLayerException e)
      {
         _logger.debug("Error occuring during access manager setup", e);
         final String error = "Grid access manager could not be loaded, exiting";
         if (e instanceof AccessConfigurationException)
            throw new ExitStatusException(error, OPTION_ERROR, e);
         else
            throw new ExitStatusException(error, IO_ERROR, e);
      }

      try
      {

         final StandardMBean gabean = new StandardMBean(new GridAccess(), GridAccessMBean.class);
         final StandardMBean vsbean = new StandardMBean(new VaultStatus(), VaultStatusMBean.class);

         JMXService jmx = JMXService.getInstance(this.jmxHost, this.jmxPort);
         MBeanServer mbs = jmx.getJMXConnectorServer().getMBeanServer();
         mbs.registerMBean(gabean, GRID_ACCESS_OBJECT_NAME);
         mbs.registerMBean(vsbean, VAULT_STATUS_OBJECT_NAME);

         System.out.println("Waiting forever...");
         try
         {
            Thread.sleep(Long.MAX_VALUE);
            this.shutdown();
         }
         catch (InterruptedException e)
         {
            _logger.debug("Thread interrupted, exiting application");
            this.shutdown();
            return;
         }
      }
      catch (MalformedURLException e)
      {
         String err =
               String.format(
                     "Configured JMX host or port invalid (%s:%s), could not connect to JMX Agent",
                     this.jmxHost, this.jmxPort);
         throw new ExitStatusException(err, OPTION_ERROR, e);
      }
      catch (IOException e)
      {
         String err =
               String.format("Could not connect to JMX Agent at %s:%s", this.jmxHost, this.jmxPort);
         throw new ExitStatusException(err, IO_ERROR, e);
      }
      catch (JMException e)
      {
         String err =
               String.format("Error accessing JMX MBeans at %s:%s", this.jmxHost, this.jmxPort);
         throw new ExitStatusException(err, OPTION_ERROR, e);
      }

      return;
   }

   public static void main(String[] args)
   {
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
      Log4jReloader.launch();
      Logger logger = Logger.getLogger(GridAccessDaemon.class);

      JSAPCommandLineParser.logEnvironment(args);

      GridAccessDaemon daemon = new GridAccessDaemon(args);
      try
      {
         daemon.run();
      }
      catch (ExitStatusException e)
      {
         _logger.fatal(e.getMessage(), e);
         System.err.println(e.getMessage());
         if (e.getCause() != null && e.getCause().getMessage() != null)
         {
            System.err.println(e.getCause().getMessage());
         }
         try
         {
            daemon.shutdown();
         }
         catch (RuntimeException ex)
         {
            _logger.warn("Could not cleanly shut down server after failure", ex);
         }
         System.exit(e.getStatus());
      }
      catch (RuntimeException e)
      {
         if (_logger.isDebugEnabled())
            _logger.debug("Fatal error exception during runtime.", e);
         logger.fatal("Fatal error: Is there another access daemon running?");
         System.err.println("Fatal error: Is there another access daemon running?");
         try
         {
            daemon.shutdown();
         }
         catch (GenericJDBCException ex)
         {
            _logger.warn("Could not properly shutdown the server!", ex);
         }
         catch (RuntimeException ex)
         {
            _logger.warn("Could not cleanly shut down server after failure", ex);
         }

         System.exit(UNKNOWN_ERROR);
      }

      logger.info("Exiting application.");

      daemon.shutdown();
      System.exit(0);

   }

}
