//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 3, 2007
//---------------------

package org.cleversafe.layer.access;

import java.net.URI;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.cleversafe.layer.access.exceptions.AccessConfigurationException;
import org.cleversafe.layer.access.exceptions.AccessIOException;
import org.cleversafe.layer.access.exceptions.AccessLayerException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStateException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.util.Tuple3;
import org.cleversafe.vault.VaultManager;

/**
 * Provides methods for performing high-level user-visible management operations such as vault
 * creation and interface setup.
 * <p>
 * When loaded, an access manager will need to be provided with the location of "grid resources" and
 * "access device state". Grid resources are centrally stored configuration elements (e.g., vault
 * information and account lists) and access device state is dynamic configuration information that
 * is stored on the access device instead of a central location.
 * <p>
 * The current interface does not require grid account information. These interfaces will be revised
 * once the application-level access control and authorization system is more complete.
 * 
 */
public interface GridAccessManager
{

   GridAccessManager load(URI directory) throws AccessLayerException;

   void shutdown() throws ServiceInterfaceStartStopException, ServiceStartStopException;
   
   VaultManager getVaultManager();

   /**
    * Returns a list of vaults manageable through this access manager.
    * 
    * @param running
    *           Limits returned list to vaults that are currently running as a service.
    * @return Returns a list of all vaults or a list of running vaults.
    */
   Set<UUID> getVaultIdentifiers(boolean running);
   
   /**
    * Returns a list of information about services. Returns (type,name,vault) tuples.
    */
   Set<Tuple3<String,String,UUID>> getServiceInfo(boolean running);
   

   Service getService(String type, String name) throws AccessIOException,
         AccessConfigurationException, ServiceNotFoundException, ServiceStartStopException;

   Service getService(UUID vaultIdentifier) throws AccessIOException,
         AccessConfigurationException, ServiceNotFoundException, ServiceStartStopException;
   
   ServiceInterface<? extends Service> getServiceInterface(String type)
         throws ServiceInterfaceNotFoundException;

   ServiceInterface<? extends Service> createServiceInterface(
         String type,
         String host,
         int port,
         boolean autostart) throws AccessIOException, AccessConfigurationException,
         ServiceInterfaceStartStopException;
   
   void deleteServiceInterface(String type) throws AccessIOException, ServiceInterfaceStateException;
   
   List<ServiceInterface<Service>> getServiceInterfaces();

}
