//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 21, 2007
//---------------------

package org.cleversafe.layer.access.cli;
import java.io.IOException;

import javax.naming.ServiceUnavailableException;

import org.apache.log4j.Logger;
import org.apache.log4j.extras.DOMConfigurator;
import org.cleversafe.layer.access.GridAccessConnector;
import org.cleversafe.util.JSAPCommandLineParser;
import org.cleversafe.util.Log4jReloader;

import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;

// TODO: Describe class or interface
public abstract class AbstractCLI implements Runnable
{
   private static Logger _logger = Logger.getLogger(AbstractCLI.class);
   
   protected static final String JMX_AGENT_HOST_OPTION = "jmx_host";
   protected static final String JMX_AGENT_PORT_OPTION = "jmx_port";
   
   
   public static final int OPTION_ERROR = 255;
   public static final int SECURITY_ERROR = 30;
   public static final int CONF_ERROR = 20;
   public static final int IO_ERROR = 10;
   public static final int UNKNOWN_ERROR = 1;
   
   
   private final String commandName;
   
   private final JSAPResult result;
   
   private final String jmxHost;
   private final int jmxPort;
   
   
   
   public AbstractCLI(String commandName, String jmxHost, int jmxPort)
   {
      this.commandName = commandName;
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
      Log4jReloader.launch();
      
      this.jmxHost = jmxHost;
      this.jmxPort = jmxPort;
      this.result = null;
   }
   
   public AbstractCLI(
         Class<? extends AbstractCLI> implClass,
         String commandName,
         String jsapResource,
         String help,
         String[] args) throws ExitStatusException
   {
      this.commandName = commandName;
      DOMConfigurator.configure(System.getProperty("log4j.configuration"));
      Log4jReloader.launch();
      
      if (args != null)
         JSAPCommandLineParser.logEnvironment(args);
      
      
      try
      {
         JSAPCommandLineParser parser = new JSAPCommandLineParser(
               implClass,
               jsapResource,
               this.commandName + " [OPTION]...",
               help);
         this.result = parser.parse(args);
      }
      catch (IOException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         throw new ExitStatusException(IO_ERROR, e);
      }
      catch (JSAPException e)
      {
         System.err.println("error: could not parse command line arguments: " + e.getMessage());
         throw new ExitStatusException(CONF_ERROR, e); 
      }
      
      if ( result == null )
      {
         // Parser handled argument exception
         throw new ExitStatusException("Option parser error encountered", OPTION_ERROR);
      }
      
      this.jmxHost = this.result.getString(JMX_AGENT_HOST_OPTION);
      this.jmxPort = this.result.getInt(JMX_AGENT_PORT_OPTION);
   }
   
   protected JSAPResult getJSAPResult()
   {
      return this.result;
   }
   
   protected String getJmxHost()
   {
      return jmxHost;
   }

   protected int getJmxPort()
   {
      return jmxPort;
   }

   protected GridAccessConnector getGridAccessConnector() throws ExitStatusException
   {
      try
      {
         return new GridAccessConnector(this.getJmxHost(), this.getJmxPort());
      }
      catch (IOException e)
      {
         // The ServiceUnavailableException has a really bad message format. Here we fix
         // it up a little bit so it is more readable.
         if (e.getCause() instanceof ServiceUnavailableException)
         {
            throw new ExitStatusException(
                  "Unable to establish connection with access server",
                  IO_ERROR,
                  new ServiceUnavailableException(
                        String.format("Connection refused to host (%s:%s)",
                              this.jmxHost, this.jmxPort)));
         }
         else
         {
            throw new ExitStatusException(
                  "Unable to establish connection with access server", IO_ERROR, e.getCause());   
         }
         
      }
   }

   /**
    * Executes {@link #call()} and calls {@link System#exit(int)} with the return status.
    */
   public void finish() throws ExitStatusException
   {
      try
      {
         this.run();
      }
      catch (ExitStatusException e)
      {
         _logger.fatal(e.getMessage(), e);
         System.err.println(e.getMessage());
         Throwable t = e.getCause(); 
         while (t!= null) 
         {
            if (t.getMessage() != null) 
            {
               System.err.println(e.getCause().getMessage());
            }
            t = t.getCause();
         }
         throw e;
         
      }
      catch (RuntimeException e)
      {
         _logger.fatal("Fatal error", e);
         System.err.println("Fatal error.");
         System.err.println(e.getMessage());
         throw new ExitStatusException("Unknown fatal error", UNKNOWN_ERROR, e);
      }
      
      _logger.info("Exiting application.");
   }
   
   
   
   
   
   
}


