//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 7, 2007
//---------------------

package org.cleversafe.layer.access.managers.state;
import java.util.List;
import java.util.concurrent.ConcurrentMap;

import org.cleversafe.layer.access.Service;
import org.cleversafe.layer.access.ServiceInterface;
import org.cleversafe.layer.access.exceptions.AccessIOException;
import org.cleversafe.layer.access.exceptions.AccessStateModificationException;
import org.cleversafe.layer.access.exceptions.DuplicateServiceException;
import org.cleversafe.layer.access.exceptions.InvalidServiceTypeException;
import org.cleversafe.layer.access.exceptions.ServiceConfigurationException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceStillRunningException;
import org.cleversafe.layer.access.managers.LocalFileGridAccessManager;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.managers.state.VaultState;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.classic.Session;

/**
 * Persistence wrapper for {@link ServiceInterface} objects returned from the
 * {@link LocalFileGridAccessManager} implementation.
 */
public class LocalFileServiceInterfaceWrapper implements ServiceInterface<Service>
{
   private ServiceInterfaceState state;
   private final ServiceInterface<Service> serviceInterface;
   private final SessionFactory factory;
   private ConcurrentMap<Tuple2<String,String>,Service> loadedServices;
   

   public LocalFileServiceInterfaceWrapper(
         SessionFactory factory,
         ServiceInterfaceState state,
         ServiceInterface<Service> serviceInterface,
         ConcurrentMap<Tuple2<String,String>,Service> loadedServices)
   {
      this.factory = factory;
      this.state = state;
      this.serviceInterface = serviceInterface;
      this.loadedServices = loadedServices;
   }

   public void add(Service service) throws AccessStateModificationException
   {
      if ( !this.getType().equals(service.getType()) )
      {
         throw new InvalidServiceTypeException(
               "service interface does not support service type: " + this.getType());
      }
         
      
      Session session = factory.openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         this.state = (ServiceInterfaceState)session.merge(this.state);
         
         // If the service is already persistent, it is attached to this interface or another
         ServiceState serviceState = (ServiceState)session
               .createQuery("from ServiceState s where (s.name is :name and s.type is :type)")
               .setParameter("name", service.getName())
               .setParameter("type", service.getType())
               .uniqueResult();
         if ( serviceState != null )
         {
            // serviceState.serviceInterface and this.state should be the same instance
            // (they attach to the same Hibernate persistence context) 
            if ( serviceState.getServiceInterfaces().contains(this.state) )
            {
               throw new DuplicateServiceException(
                     "service already attached to another service interface: " + service.getName());
            }
            else
            {
               // service already added to this service interface
               tx.commit();
               return;
            }
         }
         
         // Check that service instance is not already loaded
         for ( Service currentService : this.serviceInterface.getServices() )
         {
            // services are keyed by (name, type), but type is already checked above
            if (currentService == service || currentService.getName().equals(service.getName()))
            {
               throw new DuplicateServiceException(
                     "service already added to interface: " + service.getName());
            }
         }
         
         // Load vault state to attach to new service state
         VaultState vaultState = (VaultState)session
               .createQuery("from VaultState v where v.identifier is :identifier")
               .setParameter("identifier", service.getVaultIdentifier().toString())
               .uniqueResult();
         if ( vaultState == null )
            throw new AccessStateModificationException(
                  "vault does not exist: " + service.getVaultIdentifier());
         
         // Create new service state
         serviceState = new ServiceState();
         serviceState.setName(service.getName());
         serviceState.setType(service.getType());
         serviceState.setVault(vaultState);
         
         // Add service state to service interface state
         this.state.getServices().add(serviceState);
         
         // Add service to service interface object
         this.serviceInterface.add(service);
         
         // Add service to manager's loaded services map
         this.loadedServices.put(
               new Tuple2<String,String>(service.getType(),service.getName()), service);
         
         tx.commit();
         
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
      
   }

   public Service getService(String serviceName) throws ServiceNotFoundException
   {
      return this.serviceInterface.getService(serviceName);
   }

   public String[] getServiceNames()
   {
      return this.serviceInterface.getServiceNames();
   }

   public List<Service> getServices()
   {
      return this.serviceInterface.getServices();
   }

   public String getType()
   {
      return this.serviceInterface.getType();
   }

   public boolean isRunning()
   {
      return this.serviceInterface.isRunning();
   }

   public ServiceInterface<Service> load(String host, int port) throws ServiceConfigurationException
   {
      return this.serviceInterface.load(host, port);
   }

   public Service remove(Service service) throws ServiceStillRunningException,
         AccessStateModificationException, ServiceNotFoundException
   {
      Session session = factory.openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         this.state = (ServiceInterfaceState)session.merge(this.state);
         
         ServiceState serviceState = null;
         
         if ( service instanceof LocalFileServiceWrapper )
         {
            // Attempt to merge an existing state
            serviceState = (ServiceState)session.merge(
                  ((LocalFileServiceWrapper)service).getState());
         }
         else
         {
            // Query the database for a state matching the given service object
            serviceState = (ServiceState)session
               .createQuery("from ServiceState s where (s.name is :name and s.type is :type)")
               .setParameter("name", service.getName())
               .setParameter("type", service.getType())
               .uniqueResult();
            if ( state == null )
               throw new AccessStateModificationException(
                     "service does not exist: " + service.getName());
         }
         
         // Remove service state from service interface state
         this.state.getServices().remove(serviceState);
         
         // Remove service itself if it no longer belongs to any service interfaces
         if (serviceState.getServiceInterfaces().size() == 0)
         {
            session.delete(serviceState);
            
            // Also remove the orphaned service instance from the manager's loaded services map
            this.loadedServices.remove(
                  new Tuple2<String,String>(service.getType(),service.getName()));
         }
         
         tx.commit();
         
         // Remove service from service interface object
         return this.serviceInterface.remove(service);
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   public void start() throws ServiceInterfaceStartStopException, ServiceStartStopException
   {
      this.serviceInterface.start();
   }

   public void stop() throws ServiceInterfaceStartStopException, ServiceStartStopException
   {
      this.serviceInterface.stop();
   }

   public String getHost()
   {
      return this.serviceInterface.getHost();
   }

   public int getPort()
   {
      return this.serviceInterface.getPort();
   }

   public boolean isStartsAutomatically() throws AccessIOException
   {
      Session session = factory.openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         this.state = (ServiceInterfaceState)session.merge(this.state);
         
         tx.commit();
         
         return state.isStartAutomatically();
         
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }

   public ServiceInterface<Service> setStartsAutomatically(boolean autostart)
         throws AccessStateModificationException
   {
      Session session = factory.openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         this.state = (ServiceInterfaceState)session.merge(this.state);
         
         // Save configuration to service interface state
         state.setStartAutomatically(autostart);
         
         tx.commit();
         
         return (ServiceInterface<Service>)this;
      }
      finally
      {
         if (!tx.wasCommitted())
            tx.rollback();
         session.close();
      }
   }
   
   public ServiceInterfaceState getState()
   {
      return this.state;
   }
   
   
   
}


