//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: May 2, 2007
//---------------------

package org.cleversafe.layer.block;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.block.exceptions.BlockCorruptException;
import org.cleversafe.layer.block.exceptions.BlockIOException;
import org.cleversafe.layer.block.exceptions.BlockTransactionException;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.ReadControllerFactory;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;

/**
 * Handles block requests to the grid
 */
public class BlockDeviceController
{
   private static Logger _logger = Logger.getLogger(BlockDeviceController.class);

   // Vault containing device data
   protected BlockDeviceVault vault;

   // Prototype controller -- a clone is made for each transaction
   protected GridController gridController;

   protected List<CorruptedSliceObserver> corruptedSliceObservers =
         new ArrayList<CorruptedSliceObserver>();

   // Internal statistics
   protected AtomicInteger statsNumReads = new AtomicInteger();
   protected AtomicInteger statsNumWrites = new AtomicInteger();

   protected boolean started = false;
   protected boolean requestedStarted = false;

   private final PerformanceStatistics readPerformanceStatistics =
         new PerformanceStatistics(60, Level.INFO, "Overall READ statistics:",
               "Last 60 seconds READ statistics:", true, 10, 50, 100, 500, 1000, 2000, 5000, 10000,
               15000);

   private final PerformanceStatistics writePerformanceStatistics =
         new PerformanceStatistics(60, Level.INFO, "Overall WRITE statistics:",
               "Last 60 seconds WRITE statistics:", true, 10, 50, 100, 500, 1000, 2000, 5000,
               10000, 15000);

   public BlockDeviceController(BlockDeviceVault vault) throws ConfigurationException
   {
      BindingsProvider bindingsProvider =
            ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);

      ReadControllerFactory rfc =
            bindingsProvider.getDefaultImplementation(ReadControllerFactory.class);
      WriteControllerFactory wfc =
            bindingsProvider.getDefaultImplementation(WriteControllerFactory.class);

      this.vault = vault;
      this.gridController = new GridController(vault, wfc, rfc);
   }

   public void addCorruptedSliceObserver(CorruptedSliceObserver observer)
   {
      this.corruptedSliceObservers.add(observer);
   }

   /**
    * Construct a controller without an actual system device
    * 
    * @param vault
    *           Vault that represents this device
    * @param controller
    *           Controller responsible for slice storage to the grid
    */
   public BlockDeviceController(BlockDeviceVault vault, GridController controller)
   {
      this.vault = vault;
      this.gridController = controller;
   }

   /**
    * Initialize the controller
    */
   public void startup()
   {
      this.requestedStarted = true;
   }

   protected synchronized void startupImpl()
   {
      assert this.requestedStarted == true;

      if (!this.started)
      {
         this.vault.startSessions();
         this.started = true;
      }
   }

   /**
    * Cleanup controller
    */
   public void shutdown()
   {
      if (this.started)
      {
         this.vault.endSessions();
         this.started = false;
      }
      this.requestedStarted = false;
   }

   /**
    * Read one or more sequential blocks from the grid
    * 
    * @param firstBlock
    *           Index of first block to read
    * @param numBlocks
    *           Number of blocks to read starting from firstBlock
    * @return data Block data of size blockSize * numBlocks
    * @throws ControllerException
    * @throws Exception
    */
   public byte[] readBlocks(long firstBlock, int numBlocks) throws BlockIOException,
         BlockCorruptException
   {
      // Lazy initialization
      if (!this.started)
      {
         startupImpl();
      }

      long startTime = System.nanoTime();
      ReadController readController;
      try
      {
         readController = this.gridController.getReadController();
      }
      catch (final Exception ex)
      {
         throw new RuntimeException("Unable to instantiate read controller", ex);
      }

      // Add corrupted slice observers
      // TODO: Consider configuring observers from the vault definition or client config
      for (CorruptedSliceObserver observer : this.corruptedSliceObservers)
      {
         readController.addCorruptedSliceObserver(observer);
      }

      // Perform multi-block read
      int blockSize = getBlockSize();
      List<SourceName> dataSources = new ArrayList<SourceName>(numBlocks);
      for (int i = 0; i < numBlocks; ++i)
      {
         dataSources.add(new SourceName(Long.toString(firstBlock + i)));
      }
      byte[] data = new byte[numBlocks * blockSize];

      String readOperationDescription =
            String.format("firstBlock=%d, numBlocks=%d", firstBlock, numBlocks);

      try
      {
         int numReads = this.statsNumReads.incrementAndGet();
         if (_logger.isDebugEnabled())
            _logger.debug(String.format("Read<%d> started: %s", numReads, readOperationDescription));

         List<DataSource> results = readController.read(dataSources);

         // Construct data for response
         for (int i = 0; i < results.size(); ++i)
         {
            long blockNo = Long.parseLong(results.get(i).getName().getName());
            byte[] readData = results.get(i).getData();
            // If no data read for a block null will be returned
            if (readData != null)
            {
               System.arraycopy(readData, 0, data, (int) (blockNo - firstBlock) * blockSize,
                     blockSize);
            }
         }

         long endTime = System.nanoTime();

         if (_logger.isEnabledFor(this.readPerformanceStatistics.getLogLevel()))
         {
            this.readPerformanceStatistics.log((long) (startTime / 1.e6), (long) (endTime / 1.e6));
         }

         if (_logger.isDebugEnabled())
            _logger.debug(String.format("Read<%d> successful: %s (%dms)", numReads,
                  readOperationDescription, (long) ((endTime - startTime) / 1e6)));

         return data;
      }
      catch (ControllerDataTransformationException ex)
      {
         throw new BlockCorruptException("Corrupt data for block request: "
               + readOperationDescription, ex);
      }
      catch (ControllerIOException ex)
      {
         throw new BlockIOException("IO error reading from grid: " + readOperationDescription, ex);
      }
      catch (ControllerStoresNotFoundException ex)
      {
         throw new BlockIOException("Non-temporary storage problems, check & repair vault: "
               + readOperationDescription, ex);
      }
      catch (ControllerGridStateUnknownException ex)
      {
         throw new BlockIOException("Indeterminate grid error: " + readOperationDescription, ex);
      }
      catch (ControllerInformationDispersalException ex)
      {
         // The IDA should not throw an error on read unless there is a configuration problem
         throw new RuntimeException("IDA error on read, likely a configuration problem: "
               + readOperationDescription, ex);
      }
      catch (ControllerIllegalSourceNameException ex)
      {
         // We create the source name locally, so this is an internal error
         throw new RuntimeException("Internal error: illegal source name: "
               + readOperationDescription, ex);
      }
   }

   /**
    * Write one or more sequential blocks to the grid
    * 
    * @param firstBlock
    *           Index of first block to write
    * @param numBlocks
    *           Number of blocks to write starting from firstBlock
    * @param data
    *           Block data of size blockSize * numBlocks
    * @throws BlockIOException
    * @throws BlockTransactionException
    */
   public void writeBlocks(long firstBlock, int numBlocks, byte[] data) throws BlockIOException,
         BlockTransactionException
   {
      // Lazy initialization
      if (!this.started)
      {
         startupImpl();
      }

      long startTime = System.nanoTime();
      WriteController writeController;

      try
      {
         // Create a new controller w/transaction
         writeController = this.gridController.getWriteController();

         for (CorruptedSliceObserver observer : this.corruptedSliceObservers)
         {
            writeController.addCorruptedSliceObserver(observer);
         }
      }
      catch (final Exception ex)
      {
         throw new RuntimeException("Unable to instantiate write controller");
      }

      // Build multi-block write request
      int blockSize = getBlockSize();
      List<DataSource> request = new ArrayList<DataSource>(numBlocks);
      for (int i = 0; i < numBlocks; ++i)
      {
         // Isolate a block
         byte[] block = new byte[blockSize];
         System.arraycopy(data, i * blockSize, block, 0, blockSize);

         request.add(new DataSource(new SourceName(Long.toString(firstBlock + i)),
               writeController.getTransactionId(), block));
      }

      // Send the result to the device
      String writeOperationDescription =
            String.format("firstBlock=%d, numBlocks=%d, txid=%d", firstBlock, numBlocks,
                  writeController.getTransactionId());
      try
      {
         int numWrites = this.statsNumWrites.incrementAndGet();
         if (_logger.isDebugEnabled())
            _logger.debug(String.format("Write<%d> started: %s", numWrites,
                  writeOperationDescription));

         // Write & commit data
         writeController.write(request);
         writeController.commit();

         long endTime = System.nanoTime();
         if (_logger.isEnabledFor(this.writePerformanceStatistics.getLogLevel()))
         {
            this.writePerformanceStatistics.log((long) (startTime / 1.e6), (long) (endTime / 1.e6));
         }

         if (_logger.isDebugEnabled())
            _logger.debug(String.format("Write<%d> successful: %s (%dms)", numWrites,
                  writeOperationDescription, (long) ((endTime - startTime) / 1e6)));
      }
      catch (final ControllerIllegalSourceNameException ex)
      {
         // Programmer error, since we determine the name here
         throw new RuntimeException("Invalid source name:" + writeOperationDescription, ex);
      }
      catch (final ControllerDataTransformationException ex)
      {
         // Likely a configuration error
         throw new RuntimeException("Cannot encode data, likely a configuration error: "
               + writeOperationDescription, ex);
      }
      catch (final ControllerInformationDispersalException ex)
      {
         // Likely a configuration error
         throw new RuntimeException("Cannot disperse data, likely a configuration error:"
               + writeOperationDescription, ex);
      }
      catch (final ControllerIOException ex)
      {
         throw new BlockIOException("Write error:" + writeOperationDescription, ex);
      }
      catch (final ControllerTransactionException ex)
      {
         throw new BlockTransactionException("Grid transaction conflict:"
               + writeOperationDescription, ex);
      }
      catch (final ControllerGridStateUnknownException ex)
      {
         throw new BlockIOException("Indeterminate grid state:" + writeOperationDescription, ex);
      }
      catch (final ControllerStoresNotFoundException ex)
      {
         throw new BlockIOException("Non-temporary storage problems, check & repair vault:"
               + writeOperationDescription, ex);
      }
      finally
      {
         // Rolls back if necessary
         try
         {
            writeController.shutdown();
         }
         catch (final ControllerException ex)
         {
            _logger.debug("Unable to rollback transaction", ex);
            _logger.error("Unable to rollback transaction for txid: "
                  + writeController.getTransactionId() + ": " + ex.getMessage());
         }
      }
   }

   /**
    * Returns the size of the device, in bytes
    * 
    * @return Size of the device, in bytes
    */
   public long getDeviceSize()
   {
      return this.vault.getMaxDeviceSize();
   }

   /**
    * Returns the size of a block, in bytes
    * 
    * @return Size of a block, in bytes
    */
   public int getBlockSize()
   {
      return this.vault.getBlockSize();
   }

   /**
    * Returns the number of blocks on the device
    * 
    * @return Number of blocks on the device
    */
   public long getNumBlocks()
   {
      return this.vault.getNumBlocks();
   }

   public int getStatsNumReads()
   {
      return this.statsNumReads.get();
   }

   public int getStatsNumWrites()
   {
      return this.statsNumWrites.get();
   }
}
