//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: May 8, 2007
//---------------------

package org.cleversafe.layer.block;

import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.codec.Codec;
import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.vault.BaseVault;

// TODO: Describe class or interface
public class BlockDeviceVault extends BaseVault
{
   private static Logger _logger = Logger.getLogger(BlockDeviceVault.class);

   /* Number of blocks in this vault */
   public static final String TYPE = "block";

   private long numBlocks = -1;

   /* Block size in bytes */
   private int blockSize = -1;

   public BlockDeviceVault()
   {
      super(TYPE);
   }

   /**
    * 
    * @param blockSize
    * @param numBlocks
    * @param idaCodec
    * @param codecs
    * @param sliceStores
    * @throws IDAInvalidParametersException
    */
   public BlockDeviceVault(
         final int blockSize,
         final long numBlocks,
         InformationDispersalCodec idaCodec,
         List<Codec> codecs,
         List<Codec> sliceCodecs,
         List<SliceStore> sliceStores) throws IDAInvalidParametersException
   {
      super(TYPE, idaCodec, codecs, sliceCodecs, sliceStores);
      this.blockSize = blockSize;
      this.numBlocks = numBlocks;

      this.trimIDAChunkSize();

      this.initialize();
   }

   public long getSize()
   {
      return getMaxDeviceSize();
   }

   /**
    * Returns the maximum-size block device that can be created in this vault
    * 
    * @return long Maximum size of device that can fit into this vault
    */
   public long getMaxDeviceSize()
   {
      // FIXME: Calculate the size with num blocks, slice overhead, and coding
      // taken into account
      return this.getBlockSize() * this.getNumBlocks();
   }

   public int getBlockSize()
   {
      return blockSize;
   }

   @RequiredForInitialization
   public void setBlockSize(int size)
   {
      this.blockSize = size;
   }

   public long getNumBlocks()
   {
      return numBlocks;
   }

   @RequiredForInitialization
   public void setNumBlocks(long numBlocks)
   {
      this.numBlocks = numBlocks;
   }

   @Override
   public void initialize() throws IDAInvalidParametersException
   {
      super.initialize();

      if (this.blockSize <= 0)
      {
         throw new InitializationException("Invalid block-size parameter: " + this.blockSize);
      }

      // TODO: Is numBlocks = 0 allowed? maybe for testing 0-sized vault can be created
      if (this.numBlocks < 0)
      {
         throw new InitializationException("Invalid num-blocks parameter: " + this.numBlocks);
      }
   }

   protected int getEncodedBlockSlize()
   {
      return ((int) getEncodedDatasourceSize(this.blockSize));
   }

   /**
    * This method adjusts the chunk size of the IDA to be just small enough to accommodate a single
    * encoded block of the block device
    * 
    * @throws IDAInvalidParametersException
    */
   protected void trimIDAChunkSize() throws IDAInvalidParametersException
   {
      // This optimizes the IDA chunk size to be the minimum needed
      // to accommodate the device's block size after codecs are applied
      this.idaCodec.setChunkSize(this.getEncodedBlockSlize());
      this.idaCodec.initialize();
   }

   public long getMaxSliceSize()
   {
      return this.getSliceSize(this.blockSize);
   }

   @Override
   public long getSliceStoreSize(int sliceStoreIndex)
   {
      return this.getNumBlocks() * this.getMaxSliceSize();
   }

   public String toString()
   {
      StringBuffer buf = new StringBuffer();
      buf.append("BlockDevice[").append(this.getBlockSize()).append(" x ").append(
            this.getNumBlocks()).append(" = ").append((double) getMaxDeviceSize() / (1024 * 1024)).append(
            "MB ] ");
      buf.append(super.toString());
      return buf.toString();
   }

   /**
    * Makes the optimization of adjusting the IDA chunk size to match the expected size of a
    * device's block after applying datasource codecs
    */
   public void optimizeVault()
   {
      try
      {
         trimIDAChunkSize();
      }
      catch (IDAInvalidParametersException e)
      {
         _logger.error("Unable to optimize IDA chunk size", e);
      }
   }
}
