//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Apr 15, 2008
//---------------------

package org.cleversafe.layer.block;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.cleversafe.util.NamedThreadFactory;

// TODO: Describe class or interface
public class PerformanceStatistics
{
   private static Logger _logger = Logger.getLogger(PerformanceStatistics.class);

   private static ScheduledExecutorService executor =
         new ScheduledThreadPoolExecutor(1, new NamedThreadFactory(
               "Block Device Performance Thread"));

   private Runnable statisticsLogger = null;

   // Intervals in milliseconds
   private final int[] intervals;

   private final int[] statArray;
   private final int[] rollingStatArray;

   private final String statHeader;
   private final String rollingStatHeader;

   private final int logFrequencyMillis; // in seconds

   private final Level logLevel;

   private boolean printZeros = true;

   private class StatisticsLogRunnable implements Runnable
   {
      public void run()
      {
         int[] statArray;
         int[] rollingStatArray;
         synchronized (PerformanceStatistics.this)
         {
            // copy the array so the logging doesn't need to be synchronized
            statArray = PerformanceStatistics.this.statArray.clone();
            rollingStatArray = PerformanceStatistics.this.rollingStatArray.clone();
            resetRollingStatArray();
         }

         log(PerformanceStatistics.this.rollingStatHeader, rollingStatArray);
         log(PerformanceStatistics.this.statHeader, statArray);
      }
   }

   public PerformanceStatistics(
         int logFrequency,
         Level logLevel,
         String statHeader,
         String rollingStatHeader,
         boolean printZeros,
         int... intervals)
   {
      this.intervals = intervals;

      this.statArray = new int[intervals.length + 1];
      this.rollingStatArray = new int[intervals.length + 1];

      for (int i = 0; i <= intervals.length; i++)
      {
         if (i < intervals.length - 1 && intervals[i] >= intervals[i + 1])
         {
            throw new IllegalArgumentException("Intervals must be in ascending order");
         }
         this.statArray[i] = 0;
         this.rollingStatArray[i] = 0;
      }

      this.logFrequencyMillis = logFrequency * 1000;
      this.statHeader = statHeader;
      this.rollingStatHeader = rollingStatHeader;
      this.logLevel = logLevel;
      this.printZeros = printZeros;
   }

   public Level getLogLevel()
   {
      return this.logLevel;
   }

   /**
    * 
    * @param startTime
    *           in milliseconds
    * @param endTime
    *           in milliseconds
    */
   public void log(long startTime, long endTime)
   {
      long diff = endTime - startTime;

      for (int i = 0; i <= this.intervals.length; i++)
      {
         if (i == this.intervals.length || diff < this.intervals[i])
         {
            synchronized (this)
            {
               this.statArray[i]++;
               this.rollingStatArray[i]++;
            }
            break;
         }
      }

      if (this.statisticsLogger == null)
      {
         this.statisticsLogger = new StatisticsLogRunnable();

         // Start logging
         executor.scheduleAtFixedRate(this.statisticsLogger, this.logFrequencyMillis,
               this.logFrequencyMillis, TimeUnit.MILLISECONDS);
      }
   }

   private void log(String header, int[] array)
   {
      StringBuilder msg = new StringBuilder();
      msg.append(String.format("%n%s%n", header));

      List<Integer> indeces = new ArrayList<Integer>();

      for (int i = 0; i <= PerformanceStatistics.this.intervals.length; i++)
      {
         if (PerformanceStatistics.this.printZeros || array[i] != 0)
         {
            msg.append(String.format("%-15s", getIntervalString(i)));
            indeces.add(i);
         }
      }

      msg.append(System.getProperty("line.separator"));

      for (int i : indeces)
      {
         msg.append(String.format("%-15d", array[i]));
      }

      msg.append(System.getProperty("line.separator"));

      _logger.log(PerformanceStatistics.this.logLevel, msg.toString());
   }

   private void resetRollingStatArray()
   {
      for (int i = 0; i < this.rollingStatArray.length; i++)
         this.rollingStatArray[i] = 0;
   }

   private String formatInterval(int interval)
   {
      if (interval >= 1000) // in seconds
      {
         return String.format("%tss", (long) interval);
      }
      else
      // milliseconds
      {
         return String.format("%tQms", (long) interval);
      }
   }

   private String getIntervalString(int intervalIndex)
   {
      if (intervalIndex == 0)
      {
         return "< " + formatInterval(PerformanceStatistics.this.intervals[intervalIndex]);
      }
      else if (intervalIndex == PerformanceStatistics.this.intervals.length)
      {
         return "> " + formatInterval(PerformanceStatistics.this.intervals[intervalIndex - 1]);
      }
      else
      {
         return formatInterval(PerformanceStatistics.this.intervals[intervalIndex - 1]) + "-"
               + formatInterval(PerformanceStatistics.this.intervals[intervalIndex]);
      }
   }
}
