//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: gdhuse
//
// Date: May 2, 2007
//---------------------

package org.cleversafe.layer.block.dsd;
import java.net.*;
import java.io.*;
import java.util.concurrent.*;
import org.apache.log4j.Logger;
import org.cleversafe.layer.block.dsd.message.DeviceMessage;
import org.cleversafe.layer.block.dsd.message.ReadBlocksDeviceMessage;
import org.cleversafe.layer.block.dsd.message.RemoveDeviceMessage;
import org.cleversafe.layer.block.dsd.message.RemoveResponseDeviceMessage;
import org.cleversafe.layer.block.dsd.message.WriteBlocksDeviceMessage;


/**
 * Interface to system-level device drivers
 */
public class SystemBlockDevice
{
   // Time to wait for a device connection, in ms
   public static final int ACCEPT_TIMEOUT = 5000;
   
   // Time to wait for system activities (eg. ioctl()), in ms
   public static final int SYSTEM_TIMEOUT = 5000;
   
   private static Logger _logger = Logger.getLogger(SystemBlockDevice.class);
   
   // Number of blocks on the device
   protected long numBlocks;
   
   // Size of a block in bytes
   protected int blockSize;
   
   // Name of this device in the system (eg. /dev/dsd0)
   protected String deviceName;
   
   // Socket connection to system device
   protected Socket socket;
   
   // Socket I/O streams
   protected DataInputStream dataInputStream;
   protected DataOutputStream dataOutputStream;
   
   
   /**
    * Construct a system device
    * @param numBlocks Number of blocks on the device
    * @param blockSize Size of a block in bytes
    */
   public SystemBlockDevice( long numBlocks, int blockSize ) 
      throws IOException
   {
      this.numBlocks    = numBlocks;
      this.blockSize    = blockSize;
      this.deviceName   = null;
      this.socket       = null;
      
      // Create a system device and establish a connection with it
      System.loadLibrary( "SystemBlockDevice" );
      this.initialize();
   }
   
   /**
    * Initialize a connection to an externally-created block device on the given port
    * @param numBlocks Number of blocks on the device
    * @param blockSize Size of a block in bytes
    * @param staticPort Port to listen for external device connection
    */
   public SystemBlockDevice( long numBlocks, int blockSize, int staticPort ) throws IOException
   {
      this.numBlocks    = numBlocks;
      this.blockSize    = blockSize;
      this.deviceName   = null;
      this.socket       = null;
      
      // Wait for a connection from an externally-created device
      this.initializeStatic( staticPort );
   }
   
   /**
    * Gracefully notify the system that it's ok to destroy the device
    * before being destroyed
    */
   public void finalize()
   {
      try
      {
         this.removeSystemDevice();
      }
      catch( IOException e )
      {
         // Ignore all errors, the device driver will recover on its own
      }
         
   }
   
   /**
    * Gets the platform-specific device name (eg. /dev/dsd0)
    * @return Device name
    */
   public String getDeviceName()
   {
      return this.deviceName;
   }
   
   /**
    * Create a system device and establish a connection with it
    */
   protected void initialize() throws IOException
   {
      _logger.trace( "Initializing system device" );
      
      // Begin listening for incoming device interface connections
      InetAddress loopback = InetAddress.getByName( null );
      final ServerSocket serverSocket = new ServerSocket(
            0,    /* Any port */
            1,    /* Maximum of one connection */
            loopback );
      
      // Initiate system device creation in a short-lived thread
      class CreateTask implements Callable<Object>
      {
         public Object call() throws Exception
         {
            SystemBlockDevice sbd = SystemBlockDevice.this;
            
            try
            {
               sbd.createSystemDevice( 
                     sbd.numBlocks, 
                     sbd.blockSize,    
                     serverSocket.getInetAddress().getAddress(),
                     serverSocket.getLocalPort() );
            }
            catch( IOException e )
            {
               SystemBlockDevice._logger.error( 
                     "Unable to create system device", e );
            }
            
            return null;
         }
      };
      FutureTask<Object> createTask 
         = new FutureTask<Object>( new CreateTask() );
      createTask.run();
      
      // Set a reasonable timeout and block until a device connects
      serverSocket.setSoTimeout( SystemBlockDevice.ACCEPT_TIMEOUT );
      Socket deviceSocket = serverSocket.accept();
      serverSocket.close();
      
      // Set a reasonable timeout and wait for device creation to complete
      try
      {
         createTask.get( SystemBlockDevice.SYSTEM_TIMEOUT, 
               TimeUnit.MILLISECONDS );
      }
      catch( Exception e )
      {
         if( e instanceof IOException )
         {
            // Exception from the actual call
            throw (IOException) e;
         }
         
         // Failure to complete is likely the result of an I/O error
         throw new IOException( "System event timeout" );
      }
      
      // FIXME: Read handshake and verify secret
      
      // Setup I/O streams
      this.socket = deviceSocket;
      
      BufferedInputStream bin = new BufferedInputStream( 
            this.socket.getInputStream() );
      this.dataInputStream = new DataInputStream( bin );
      this.dataOutputStream = new DataOutputStream( this.socket.getOutputStream() );
      
      _logger.debug( "System device '" + this.deviceName + "' created" );
   }
   
   /**
    * Listen on specified port for a connection from an externally created system device
    * @param port
    */
   protected void initializeStatic( int port ) throws IOException
   {
      InetAddress loopback = InetAddress.getByName( null );
      final ServerSocket serverSocket = new ServerSocket(
            port,   
            1,   
            loopback );
      
      _logger.info( "Waiting for external device connection on port: " + port );
      Socket deviceSocket = serverSocket.accept();
      serverSocket.close();
      this.socket = deviceSocket;
      
      BufferedInputStream bin = new BufferedInputStream( 
            this.socket.getInputStream() );
      this.dataInputStream = new DataInputStream( bin );
      this.dataOutputStream = new DataOutputStream( this.socket.getOutputStream() );
      
      this.deviceName = "Unknown (external)";
      _logger.debug( "External system device connected" );
   }
   
   /**
    * Blocks until a network event is received and returns the request message
    */
   public DeviceMessage getMessage() throws IOException
   {      
      // Read header for next message
      DeviceMessage.Header header = new DeviceMessage.Header();
      header.deserialize( this.dataInputStream, true );    // Peek only
      
      // Read message
      DeviceMessage msg;
      switch( header.messageType )
      {
         case ReadBlocksDeviceMessage.MESSAGE_TYPE:
            msg = new ReadBlocksDeviceMessage();
            break;
            
         case WriteBlocksDeviceMessage.MESSAGE_TYPE:
            msg = new WriteBlocksDeviceMessage();
            break;
            
         case RemoveResponseDeviceMessage.MESSAGE_TYPE:
            msg = new RemoveResponseDeviceMessage();
            break;
         
         default:
            _logger.error( "Unknown message of type: " + header.messageType );
            throw new IOException( "Unknown message type" );
      }
      
      msg.deserialize( this.dataInputStream );
      return msg;
   }
   
   public void sendMessage( DeviceMessage message ) throws IOException
   {
      message.serialize( this.dataOutputStream );
   }

   /**
    * Create a new system device instance and sets this.deviceName to the
    * name of the system device (eg. /dev/dsd0)
    * 
    * @param numBlocks Number of blocks on the device
    * @param blockSize Size of a block in bytes
    * @param ipAddress IP where our ServerSocket is listening
    * @param port Port where ServerSocket is listening
    * @throws IOException
    */
   protected native void createSystemDevice( 
         long numBlocks,
         int blockSize,
         byte[] ipAddress,
         int port ) throws IOException;
   
   /**
    * Removes the system device associated with this instance
    * @throws IOException
    */
   protected void removeSystemDevice() throws IOException
   {
      RemoveDeviceMessage message = new RemoveDeviceMessage();
      this.sendMessage( message );
   }
}

