//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
//@author: John Quigley <jquigley#cleversafe.com>
//
//Date: Nov 29, 2007
//---------------------

package org.cleversafe.layer.iscsi;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.log4j.Logger;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.XMLPropertiesProvider;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.access.ServiceInterface;
import org.cleversafe.layer.access.exceptions.AccessStateModificationException;
import org.cleversafe.layer.access.exceptions.ServiceConfigurationException;
import org.cleversafe.layer.access.exceptions.ServiceInterfaceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceNotFoundException;
import org.cleversafe.layer.access.exceptions.ServiceStartStopException;
import org.cleversafe.layer.access.exceptions.ServiceStillRunningException;
import org.cleversafe.layer.iscsi.exceptions.ConnectionClosedException;
import org.cleversafe.layer.target.lu.GridLogicalUnit;
import org.jscsi.parser.exception.InternetSCSIException;
import org.jscsi.scsi.target.Target;

public class ISCSIPortal implements ServiceInterface<ISCSITargetService>, Runnable
{
	public static final String ISCSI_SERVICE = "iscsi";
	public static final String ISCSI_SERVICE_THREAD_NAME = "ISCSI Portal Service";

	private static Logger _logger = Logger.getLogger(ISCSIPortal.class);

	private static final int DEFAULT_TASK_QUEUE_SIZE = 32;
	
	private static final int STOP_WAIT_TIME = 4000;

	// Task queue size, per target
	private int taskQueueSize;

	// Portal listen socket parameters
	private String bindPortalHost;
	private int bindPortalPort;
	private String advertisedPortalHost;
	private int advertisedPortalPort;
	private boolean dynamicAdvertisedPortalHost = false;

	private List<PrototypeiSCSITarget> iscsiTargets;
	private Map<String, UUID> vaultIdentifiers;
	private Map<String, Target> scsiTargets;
        private List<String> activeTargets;

   private AtomicBoolean running = new AtomicBoolean(false);
   private AtomicBoolean shouldBeRunning = new AtomicBoolean(true);

	private boolean autostart;

	private Thread thread;

	////////////////////////////////////////////////////////////////////////////////////////////////

	public String getType()
	{
		return ISCSI_SERVICE;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////

	private class PortalSession extends Thread
	{
		private int targetNum;
		private Socket socket;
		private PrototypeiSCSITarget iscsiTarget;

		public PortalSession(String threadName, int targetNum, PrototypeiSCSITarget iscsiTarget, Socket socket)
		{
			super(threadName);
			this.socket = socket;
			this.targetNum = targetNum;
			this.iscsiTarget = iscsiTarget;
		}
		
		public void run()
		{
			if (_logger.isDebugEnabled())
				_logger.debug("Starting target: " + this.targetNum);

			try
			{
				_logger.debug("adding target to portal: " + iscsiTarget);
				iscsiTargets.add(iscsiTarget);
				iscsiTarget.call();
			}
			catch (final ConnectionClosedException e)
			{
				_logger.info("Remote host closed connection");
			}
			catch (final IOException e)
			{
				_logger.warn("I/O exception in target: " + this.targetNum, e);
			}
			catch (final InternetSCSIException e)
			{
				_logger.info("attempt to connect to an unknown target: " + e.getMessage());
			}
			catch (final Exception e)
			{
				_logger.warn("Unhandled exception in target: " + this.targetNum, e);
			}
			finally
			{
				_logger.debug("removing target from portal: " + this.iscsiTarget);
				try
				{
					this.socket.close();
				}
				catch (IOException ex)
				{
					_logger.debug("socket encountered an i/o error while closing: " + ex);
				}

				_logger.debug("server closed network connection with peer");

                                String targetName = this.iscsiTarget.getTargetName();
                                if (targetName != null)
                                    activeTargets.remove(targetName);
				iscsiTargets.remove(this.iscsiTarget);
			}

			if (_logger.isDebugEnabled())
				_logger.info("Target done: " + this.targetNum);
		}
	}

	////////////////////////////////////////////////////////////////////////////////////////////////

	private void setup(String host, int port)
	{
		this.bindPortalHost = host;
		this.bindPortalPort = port;
		this.advertisedPortalPort = port;

		if (this.advertisedPortalHost == null)
		{
			this.dynamicAdvertisedPortalHost = true;
		}

		this.iscsiTargets = new LinkedList<PrototypeiSCSITarget>();
		this.scsiTargets = new HashMap<String, Target>();
		this.vaultIdentifiers = new HashMap<String, UUID>();
                this.activeTargets = new LinkedList<String>();
	}

	public ServiceInterface<ISCSITargetService> load(String host, int port)
	throws ServiceConfigurationException
	{
		this.setup(host, port);
		// read in queue-depth from properties
		try
		{
			final XMLPropertiesProvider propProvider =
				(XMLPropertiesProvider) ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

			final String[] luQueueDepthElem = {
					"org", "cleversafe", "layer", "iscsi", "target", "queue-depth"
			};

			taskQueueSize = propProvider.getIntValue(luQueueDepthElem);
			if (taskQueueSize < 1)
			{
				_logger.debug("iSCSI target queue depth parameter out of range, using default of " + DEFAULT_TASK_QUEUE_SIZE);
				taskQueueSize = DEFAULT_TASK_QUEUE_SIZE;
			}
			_logger.info("Initializing iSCSI target queue depth " + taskQueueSize);
		}
		catch (final ConfigurationException ex)
		{
			_logger.warn("Initializing iSCSI target: No queue-depth property, defaulting to "
					+ DEFAULT_TASK_QUEUE_SIZE);
			taskQueueSize = DEFAULT_TASK_QUEUE_SIZE;
		}
		return this;
	}

	public ISCSIPortal() {}

	public ISCSIPortal(
			String bindPortalHost,
			int bindPortalPort,
			String advertisedPortalHost,
			int advertisedPortalPort,
			int taskQueueSize) 
	{
		this.setup(bindPortalHost, bindPortalPort);
		this.taskQueueSize = taskQueueSize;
		this.advertisedPortalHost = advertisedPortalHost;
		this.advertisedPortalPort = advertisedPortalPort;
	}

	////////////////////////////////////////////////////////////////////////////////////////////////

	/**
	 * Register a SCSI Target.  Registered targets will be advertised
	 * 
	 * @param target
	 */
	public void registerSCSITarget(UUID vaultIdentifier, String iqn, Target scsiTarget)
	{
		this.vaultIdentifiers.put(iqn, vaultIdentifier);
		this.scsiTargets.put(iqn, scsiTarget);
		if (!scsiTarget.isRunning())
			scsiTarget.start();
	}

	/**
	 * Unregister a SCSI Target.  This target will no longer be advertised
	 * 
	 * @param target
	 */
	public void unregisterSCSITarget(String iqn)
	{
		this.vaultIdentifiers.remove(iqn);
		Target target = this.scsiTargets.remove(iqn);
		if (target != null)
		{
			target.stop();
		}
	}

	/**
	 * Returns the SCSI Target corresponding to the given iQN, if it has been registered
	 * @param iqn
	 * @return
	 */
	public Target getSCSITarget(String iqn)
	{
		return this.scsiTargets.get(iqn);
	}

	/**
	 * Returns a list of iQNs corresponding to registered SCSI Targets
	 */
	public Set<String> getDiscoveryList()
	{
		return this.scsiTargets.keySet();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////


	public void add(ISCSITargetService service) throws AccessStateModificationException
	{
		this.registerSCSITarget(service.getVaultIdentifier(), service.getName(), service.getTarget());
	}

	public ISCSITargetService getService(String serviceName) throws ServiceNotFoundException
	{
		return new ISCSITargetService(
				this.scsiTargets.get(serviceName), 
				this.vaultIdentifiers.get(serviceName),
				serviceName);
	}

	public String[] getServiceNames()
	{
		return this.scsiTargets.keySet().toArray(new String[0]);
	}

	public List<ISCSITargetService> getServices()
	{
		List<ISCSITargetService> services = new ArrayList<ISCSITargetService>();
		for ( Map.Entry<String,Target> entry : this.scsiTargets.entrySet())
		{
			services.add(
					new ISCSITargetService(
							entry.getValue(), 
							vaultIdentifiers.get(entry.getKey()), 
							entry.getKey()));
		}
		return services;
	}


	public ISCSITargetService remove(ISCSITargetService service) throws ServiceStillRunningException,
	AccessStateModificationException, ServiceNotFoundException
	{
		this.vaultIdentifiers.remove(service.getName());
		this.scsiTargets.remove(service.getName());
		return null;
	}


	////////////////////////////////////////////////////////////////////////////////////////////////

	public void run()
	{
      this.runPortal();
	}


	public void start() throws ServiceInterfaceStartStopException, ServiceStartStopException
	{
      this.shouldBeRunning.set(true);
		this.thread = new Thread(this, ISCSI_SERVICE_THREAD_NAME);
		this.thread.start();
	}

	public void stop() throws ServiceInterfaceStartStopException, ServiceStartStopException
	{
		try
		{
			this.shouldBeRunning.set(false);
			this.thread.interrupt();
			this.thread.join(STOP_WAIT_TIME);
		}
		catch (InterruptedException e)
		{
			throw new ServiceStartStopException("iSCSI Portal did not stop before timeout", e);
		}
		finally
		{
		   for(Target scsiTarget : this.scsiTargets.values())
		   {
		      scsiTarget.stop();
		   }
		}
	}

	public boolean isRunning()
	{
		return this.running.get();
	}

	/**
	 * Open portal port and service target connections
	 */
	public void runPortal()
	{
	   this.running.set(true);
	   
	   ServerSocket serverSocket = null;
	   
	   try
	   {
   		if (this.bindPortalHost == null)
   		{
   			this.bindPortalHost = "0.0.0.0";
   		}
   
   		serverSocket = new ServerSocket();
   		serverSocket.bind(new InetSocketAddress(this.bindPortalHost, this.bindPortalPort));
   		serverSocket.setSoTimeout(1000);
   
   		_logger.info(String.format("Listening on %s:%d, advertising %s:%d",
   				(this.bindPortalHost != null) ? this.bindPortalHost : "all", this.bindPortalPort,
   						(this.advertisedPortalHost != null) ? this.advertisedPortalHost : "<dynamic>", this.advertisedPortalPort));
   
   		int targetNum = 0;
   
         while (this.shouldBeRunning.get())
   		{
            Socket socket;

   		   try
   		   {
      		   socket = serverSocket.accept();
            }
            catch (SocketTimeoutException e)
            {
               continue;
            }
      
   			if (this.dynamicAdvertisedPortalHost)
   			{
               this.advertisedPortalHost = socket.getLocalAddress().getHostAddress();
   			}
   
   			final PrototypeiSCSITarget target =
   				new PrototypeiSCSITarget(this, socket, this.advertisedPortalHost, this.advertisedPortalPort, this.taskQueueSize);
   
   			final int thisTargetNum = targetNum++;
   
   			PortalSession portalSession = new PortalSession("Target Thread: " + thisTargetNum, thisTargetNum, target, socket);
   			portalSession.start();
   		}
	   }
	   catch (IOException e)
	   {
	      _logger.error("Portal exiting due to IOException: " + e);
	   }
	   finally
	   {
	      GridLogicalUnit.stopExecutorPool();
	      
	      if (serverSocket != null)
	      {
	         try
            {
               serverSocket.close();
            }
            catch (IOException e)
            {
               _logger.warn("Caught exception while trying to close server socket: " + e);
            }
	      }

         this.running.set(false);
	   }
	}

	public String getHost()
	{
		return this.bindPortalHost;
	}

	public int getPort()
	{
		return this.bindPortalPort;
	}

	public boolean isStartsAutomatically()
	{
		return this.autostart;
	}

	public ServiceInterface<ISCSITargetService> setHost(String host)
	throws AccessStateModificationException
	{
		if (this.running.get())
			throw new AccessStateModificationException("cannot change port while interface running");
		this.bindPortalHost = host;
		return (ServiceInterface<ISCSITargetService>)this;
	}

	public ServiceInterface<ISCSITargetService> setPort(int port) throws AccessStateModificationException
	{
		if (this.running.get())
			throw new AccessStateModificationException("cannot change port while interface running");
		this.bindPortalPort = port;
		return (ServiceInterface<ISCSITargetService>)this;

	}

	public ServiceInterface<ISCSITargetService> setStartsAutomatically(boolean autostart)
	throws AccessStateModificationException
	{
		this.autostart = autostart;
		return (ServiceInterface<ISCSITargetService>)this;
	}  

   public void activateTarget(String targetName)
   {
      this.activeTargets.add(targetName);
   }

   public boolean isTargetActivated(String targetName)
   {
      return this.activeTargets.contains(targetName);
   }
}
