//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Jan 9, 2008
//---------------------

package org.cleversafe.layer.iscsi;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;
import org.cleversafe.layer.access.Service;
import org.cleversafe.layer.access.exceptions.InvalidVaultLoadException;
import org.cleversafe.layer.access.exceptions.InvalidVaultTypeException;
import org.cleversafe.layer.access.exceptions.ServiceConfigurationException;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.layer.target.lu.GridLogicalUnit;
import org.cleversafe.vault.Vault;
import org.jscsi.scsi.lu.LogicalUnit;
import org.jscsi.scsi.target.DefaultTarget;
import org.jscsi.scsi.target.Target;

public class ISCSITargetService implements Service
{
   private static Logger _logger = Logger.getLogger(ISCSITargetService.class);
   
   private static final List<String> supportedVaultTypes =
      Arrays.asList(new String[] { BlockDeviceVault.TYPE });

   private static Map<UUID, ISCSITargetService> loadedServices = new HashMap<UUID, ISCSITargetService>();
   
   private String name;
   private Target scsiTarget;
   private UUID vaultIdentifier;

   
   /////////////////////////////////////////////////////////////////////////////
   // Constructors
   
   
   public ISCSITargetService()
   {
   }
   
   public ISCSITargetService(Target scsiTarget, UUID vaultIdentifier, String name)
   {
      this.name = name;
      this.scsiTarget = scsiTarget;
      this.vaultIdentifier = vaultIdentifier;
   }
   
   /////////////////////////////////////////////////////////////////////////////
   // Operations

   
   // TODO: flag indicating load has taken place
   public Service load(String name, Vault vault) throws InvalidVaultTypeException,
         ServiceConfigurationException, InvalidVaultLoadException
   {
      _logger.info("loading iscsi target service for vault: " + vault.getVaultIdentifier());

      // ensure that our vault is of the proper type
      if (!(vault instanceof BlockDeviceVault))
      {
         _logger.error("iSCSI service cannot load vault of type: " + vault.getType());
         throw new InvalidVaultTypeException("iSCSI service cannot load vault of type: "
                + vault.getType());
      }
 
      // check to see if we've loaded this vault in the past, if so, we just return that object
      if (loadedServices.get(vault.getVaultIdentifier()) != null)
      {
         _logger.info("iSCSI target service was previously loaded for vault:" + vault);
         return loadedServices.get(vault.getVaultIdentifier());
      }
      
      this.name = name;
      this.vaultIdentifier = vault.getVaultIdentifier();
      
      try
      {
         this.scsiTarget = new DefaultTarget(this.name);
         this.scsiTarget.start();

         BlockDeviceController controller = createBlockDeviceController((BlockDeviceVault) vault);
         controller.startup();

         LogicalUnit lu = new GridLogicalUnit(controller);

         this.scsiTarget.registerLogicalUnit(0, lu);
      }
      catch(IDANotInitializedException e)
      {
         _logger.error("iSCSI service encountered a configuration error while loading");
         throw new ServiceConfigurationException("iSCSI service error while loading", e);         
      }
      catch (ConfigurationException e)
      {
         _logger.error("iSCSI service encountered a configuration error while loading");
         throw new ServiceConfigurationException("iSCSI service error while loading", e);
      }

      loadedServices.put(vault.getVaultIdentifier(), this);
      
      return this;
   }


   /////////////////////////////////////////////////////////////////////////////
   // Getters/Setters


   public String getName()
   {
      return this.name;
   }

   public List<String> getSupportedVaultTypes()
   {
      return supportedVaultTypes;
   }

   public String getType()
   {
      return ISCSIPortal.ISCSI_SERVICE;
   }

   public UUID getVaultIdentifier()
   {
      return this.vaultIdentifier;
   }
   
   public Target getTarget()
   {
      return this.scsiTarget;
   }
   
   private BlockDeviceController createBlockDeviceController(BlockDeviceVault vault) 
   throws ConfigurationException, IDANotInitializedException
   {
      return new BlockDeviceController(vault);
   }
}
