//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: bcilfone
//
// Date: Dec 12, 2007
//---------------------

package org.cleversafe.layer.iscsi.authentication.chap;

import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Random;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.log4j.Logger;
import org.cleversafe.layer.iscsi.ISCSIUtils;
import org.cleversafe.layer.iscsi.authentication.ISCSIAuthenticationSession;
import org.cleversafe.layer.iscsi.authentication.SourceOfSecrets;
import org.cleversafe.layer.iscsi.exceptions.ISCSILayerException;
import org.jscsi.parser.datasegment.OperationalTextKey;
import org.jscsi.parser.datasegment.SettingsMap;

/**
 * An authentication session that stores all the information about the current state
 * of a single target's login authentication
 */

public class ChapAuthenticationSession implements ISCSIAuthenticationSession
{
   private byte[] targetSecret;
   private String targetName;

   private ChapChallengePacket challengePacket;
   private ChapStatus currentChapStatus = ChapStatus.INITIAL;

   private static Random random = new SecureRandom();

   private static Logger _logger = Logger.getLogger(ChapAuthenticationSession.class);

   /**
    * The value in the iSCSI protocol for the AuthMethod key to identify the CHAP scheme
    */
   public static final String AUTH_METHOD_CHAP = "CHAP";

   /**
    * The recommended length of challenges and secrets in 128 bits
    */
   private static final int CHAP_CHALLENGE_LENGTH = 16;

   /**
    * Value from RFC 1994
    */
   public static final String CHAP_ALGORITHM_CODE_MD5 = "5";

   /**
    * The possible states of the CHAP login security negotiation
    */
   public enum ChapStatus
   {
      /**
       * Nothing has to be done
       */
      INITIAL,

      /**
       * The method (CHAP) has been agreed upon by the initiator and target
       */
      METHOD_NEGOTIATED,

      /**
       * The CHAP challenge has been sent to the initiator and we're awaiting a response
       */
      CHALLENGE_NEGOTIATED,

      /**
       * The CHAP login has completed successfully
       */
      COMPLETE
   }

   public ChapAuthenticationSession(byte[] targetSecret, String targetName)
   {
      this.targetSecret = targetSecret;
      this.targetName = targetName;
   }

   /**
    * This will step the iSCSI initiator through the handshake protocol of security negotiation.
    * This is a simple state machine that keeps track of the last processed phase and continues
    * through the sequence reading the appropriate input SCSI text keys and setting the appropriate
    * output SCSI text keys.
    * @throws ISCSILayerException 
    */
   public SecurityNegotiationStatus handleAuthenticationCheck(SettingsMap requestSettingsMap, SettingsMap responseSettingsMap, SourceOfSecrets sourceOfSecrets) throws ISCSILayerException
   {
      if (this.currentChapStatus == ChapStatus.COMPLETE)
      {
         return SecurityNegotiationStatus.NOT_NEEDED;
      }

      if (sourceOfSecrets == null)
      {
         return SecurityNegotiationStatus.COMPLETE_FAILURE;
      }

      ChapChallengePacket challenge_packet = generateChallengePacketIfNeeded();

      //  The first step of the negotiation is to negotiate the protocol.
      if (this.currentChapStatus == ChapStatus.INITIAL)
      {
         responseSettingsMap.add(OperationalTextKey.AUTH_METHOD, AUTH_METHOD_CHAP);
         this.currentChapStatus = ChapStatus.METHOD_NEGOTIATED;
         return SecurityNegotiationStatus.IN_PROGRESS;
      }

      //  The second phase of the negotiation is to negotiate the CHAP hash algorithm.
      //  The only algorithm supported is MD5
      //  Along with the algorithm, send back the chap identifier byte and chap challenge
      if (this.currentChapStatus == ChapStatus.METHOD_NEGOTIATED)
      {
         responseSettingsMap.add(OperationalTextKey.CHAP_A, CHAP_ALGORITHM_CODE_MD5);
         responseSettingsMap.add(OperationalTextKey.CHAP_I, ISCSIUtils.encodeBinaryValue(challenge_packet.getIdentifierByte()));
         responseSettingsMap.add(OperationalTextKey.CHAP_C, ISCSIUtils.encodeBinaryValue(challenge_packet.getChallenge()));
         this.currentChapStatus = ChapStatus.CHALLENGE_NEGOTIATED;
         return SecurityNegotiationStatus.IN_PROGRESS;
      }

      //  CHAP_N is the client initiator name
      String chap_n = requestSettingsMap.get(OperationalTextKey.CHAP_N);

      //  CHAP_R is the client response to the chap challenge
      String chap_r = requestSettingsMap.get(OperationalTextKey.CHAP_R);

      //  This would only be null if the initiator ignored the challenge
      if (chap_r != null)
      {
         byte[] response_bytes = ISCSIUtils.decodeBinaryValue(chap_r);

         if (_logger.isDebugEnabled())
         {
            _logger.debug("Got response bytes: " + ISCSIUtils.encodeBinaryValue(response_bytes));
         }

         //  Get the secret for this client initiator name
         byte[] secret = sourceOfSecrets.getSecret(chap_n);

         //  Validate the response against the challenge/secret combination
         if (validateResponsePacket(challenge_packet, response_bytes, secret))
         {
            //  Get CHAP input values from initiator's request
            String chap_i = requestSettingsMap.get(OperationalTextKey.CHAP_I);
            String chap_c = requestSettingsMap.get(OperationalTextKey.CHAP_C);

            //  Check if initiator wants to do mutual authentication of the target
            if (chap_i != null && chap_c != null)
            {
               byte[] identifier_byte = ISCSIUtils.decodeBinaryValue(chap_i);
               byte[] challenge = ISCSIUtils.decodeBinaryValue(chap_c);

               //  Generate the CHAP response to deliver back to the initiator
               byte[] response = generateHashValue(identifier_byte[0], this.targetSecret, challenge);

               //  Add the CHAP values to the response header
               responseSettingsMap.add(OperationalTextKey.CHAP_R, ISCSIUtils.encodeBinaryValue(response));
               responseSettingsMap.add(OperationalTextKey.CHAP_N, this.targetName);
            }

            this.currentChapStatus = ChapStatus.COMPLETE;

            return SecurityNegotiationStatus.COMPLETE_SUCCESS;
         }
      }

      return SecurityNegotiationStatus.COMPLETE_FAILURE;
   }

   /**
    * This will determine, based on lastAuthTime and authTtl, whether authentication is needed.
    * If so, this will return a challenge packet.  As long as the authentication has not completed
    * successfully, the same challenge packet will be returned.
    * 
    * @return
    */
   private ChapChallengePacket generateChallengePacketIfNeeded()
   {
      if (this.challengePacket == null)
      {
         if (_logger.isDebugEnabled())
         {
            _logger.debug("Authentication is required.  Creating new challenge packet.");
         }

         this.challengePacket = generateChallengePacket();
      }

      return this.challengePacket;
   }

   /**
    * Returns true if the given response/secret is valid against the current challenge packet.
    * As a side effect, if the response is valid, this will update lastAuthTime to the current time
    * and set the session's challengePacket to null.
    *  
    * @param response
    * @param secret
    * @return
    */
   public static boolean validateResponsePacket(ChapChallengePacket challengePacket, byte[] response, byte[] secret)
   {
      if (secret == null)
      {
         return false;
      }

      byte[] test_response_value = generateHashValue(challengePacket.getIdentifierByte(), secret, challengePacket.getChallenge());

      if (_logger.isTraceEnabled())
      {
         _logger.trace("  Identifier:        " + ISCSIUtils.encodeBinaryValue(challengePacket.getIdentifierByte()));
         _logger.trace("  Secret:            " + ISCSIUtils.encodeBinaryValue(secret));
         _logger.trace("  Challenge:         " + ISCSIUtils.encodeBinaryValue(challengePacket.getChallenge()));
         _logger.trace("  Expected Response: " + ISCSIUtils.encodeBinaryValue(test_response_value));
         _logger.trace("  Actual Response:   " + ISCSIUtils.encodeBinaryValue(response));
      }

      if (!Arrays.equals(response, test_response_value))
      {
         if (_logger.isDebugEnabled())
         {
            _logger.debug("Response packet does not validate against challenge");
         }

         return false;
      }

      if (_logger.isDebugEnabled())
      {
         _logger.debug("Response packet validates against challenge");
      }

      return true;
   }

   /**
    * Generate a CHAP response value for the given identifier byte, secret, and challenge
    * 
    * @param pIdentifierByte
    * @param pSecret
    * @param pChallenge
    * @return
    */
   public static byte[] generateHashValue(byte identifierByte, byte[] secret, byte[] challenge)
   {
      if (secret == null)
      {
         throw new IllegalArgumentException("Secret must not be null");
      }

      if (challenge == null)
      {
         throw new IllegalArgumentException("Challenge must not be null");
      }

      byte[] value_to_hash = new byte[1 + secret.length + challenge.length];

      //  Copy the input values to a single byte array
      value_to_hash[0] = identifierByte;
      System.arraycopy(secret, 0, value_to_hash, 1, secret.length);
      System.arraycopy(challenge, 0, value_to_hash, 1 + secret.length, challenge.length);

      return DigestUtils.md5(value_to_hash);
   }

   /**
    * Generate a challenge packet containing the information needed to be presented to an initiator
    * @return
    */
   public static ChapChallengePacket generateChallengePacket()
   {
      return new ChapChallengePacket((byte) random.nextInt(), generateChallenge());
   }

   /**
    * Generate a byte array suitable for use as a CHAP challenge
    * @return
    */
   private static byte[] generateChallenge()
   {
      byte[] challenge = new byte[CHAP_CHALLENGE_LENGTH];

      random.nextBytes(challenge);

      return challenge;
   }
}
