//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Oct 22, 2007
//---------------------

package org.cleversafe.layer.target.lu;

import java.util.concurrent.ExecutorService;

import org.apache.log4j.Logger;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.XMLPropertiesProvider;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.iscsi.ISCSITargetService;
import org.cleversafe.layer.target.tasks.GridTaskFactory;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.util.ThreadlessExecutor;
import org.cleversafe.util.rangelock.RangeReadWriteLock;
import org.cleversafe.util.rangelock.RangeReadWriteLockWritePreference;
import org.jscsi.scsi.lu.AbstractLogicalUnit;
import org.jscsi.scsi.protocol.inquiry.StaticInquiryDataRegistry;
import org.jscsi.scsi.protocol.mode.StaticModePageRegistry;
import org.jscsi.scsi.tasks.management.DefaultTaskManager;
import org.jscsi.scsi.tasks.management.DefaultTaskSet;
import org.jscsi.scsi.tasks.management.TaskSet;

/**
 * @author John Quigley <jquigley@cleversafe.com>
 * @version $Id$
 */
public class GridLogicalUnit extends AbstractLogicalUnit
{
   private static Logger _logger = Logger.getLogger(ISCSITargetService.class);
   private static final String THREAD_POOL_NAME = "LU Task Executor";
   private static final int DEFAULT_QUEUE_DEPTH = 16;

   private static ExecutorService executor;
   private static int queueDepth;

   private final BlockDeviceController bdc;

   private final RangeReadWriteLock rangeReadWriteLock;

   // TODO: this class initialization block may be more gracefully located at the ISCSIPortal
   // or ISCSITargetService
   static
   {
      // read in logical-unit-threads from properties, and create a bounded
      // threadpool of that property size
      try
      {
         final XMLPropertiesProvider propProvider =
               (org.cleversafe.config.XMLPropertiesProvider) ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

         final String[] numThreadsElem = {
               "org", "cleversafe", "layer", "target", "lu", "logical-unit-threads"
         };

         final int numThreads = propProvider.getIntValue(numThreadsElem);
         if (numThreads == 0)
         {
            _logger.info("Initializing grid logical unit: single-threaded");
            executor = new ThreadlessExecutor();
         }
         else
         {
            _logger.info("Initializing grid logical unit: " + numThreads + " threads");
            executor = new BoundedThreadPoolExecutor(THREAD_POOL_NAME, numThreads);
         }
      }
      catch (final ConfigurationException ex)
      {
         _logger.warn("Initializing grid logical unit: No logical-unit-threads property, defaulting to single-threaded");
         executor = new ThreadlessExecutor();
      }

      // read in logical-unit-queue-depth from properties
      try
      {
         final XMLPropertiesProvider propProvider =
               (XMLPropertiesProvider) ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

         final String[] luQueueDepthElem = {
               "org", "cleversafe", "layer", "target", "lu", "logical-unit-queue-depth"
         };

         queueDepth = propProvider.getIntValue(luQueueDepthElem);
         if (queueDepth < 1)
         {
            queueDepth = DEFAULT_QUEUE_DEPTH;
         }
         _logger.info("Initializing grid logical unit: queue depth " + queueDepth);
      }
      catch (final ConfigurationException ex)
      {
         _logger.warn("Initializing grid logical unit: No logical-unit-queue-depth property, defaulting to "
               + DEFAULT_QUEUE_DEPTH);
         queueDepth = DEFAULT_QUEUE_DEPTH;
      }
   }

   /**
    * Creates a new GridLogicalUnit, specifying concurrent processing and queueing parameters.
    * 
    * @param device The BlockDeviceControllerfor supervising grid I/O.
    */
   public GridLogicalUnit(final BlockDeviceController device)
   {
      super();

      this.bdc = device;

      //this.rangeReadWriteLock = new RangeReadWriteLockFair();
      this.rangeReadWriteLock = new RangeReadWriteLockWritePreference();

      final TaskSet taskSet = new DefaultTaskSet(queueDepth);

      setTaskSet(taskSet);
      setTaskManager(new DefaultTaskManager(executor, taskSet));

      setTaskFactory(new GridTaskFactory(device, new StaticModePageRegistry(),
            new StaticInquiryDataRegistry(), this.rangeReadWriteLock));
   }

   /**
    * Starts the BlockDeviceController for grid I/O, and then starts this logical unit itself.
    */
   @Override
   public void start()
   {
      // NOTE: The order of these components being started is significant.  We don't want to 
      // begin accepting incoming requests from the ISCSI layer before we have properly established
      // any state required to execute operations on the dsNet via the BlockDeviceController.
      this.bdc.startup();
      super.start();
   }

   /**
    * Stops this logical unit, and then shuts down the BlockDeviceController.
    */
   @Override
   public void stop()
   {
      // NOTE: Again, as with start(), the order of component shutdown here is significant.  We want to ensure
      // we close the interface between SCSI and ISCSI layers to prevent further tasks from being accepted.  The
      // BlockDeviceController may still be operating on a task or set of tasks at the time shutdown is called, so
      // we let those complete, block here, before actually disabling grid-based I/O.
      super.stop();
      this.bdc.shutdown();
   }

   /**
    * This class should be considered deprecated, and thread pool control should be put in the hands
    * of the ISCSIPortal or ISCSITargetService.
    * 
    * @deprecated
    */
   @Deprecated
   static public void stopExecutorPool()
   {
      GridLogicalUnit.executor.shutdown();
   }

   /**
    * Returns the size of the device, in bytes.
    * 
    * @return Size of device (in bytes)
    */
   public long getDeviceCapacity()
   {
      return this.bdc.getDeviceSize();
   }

   /**
    * Returns the size of a block, in bytes.
    * 
    * @return Device block size (in bytes)
    */
   public int getBlockSize()
   {
      return this.bdc.getBlockSize();
   }
}
