//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
//@author: jquigley
//
//Date: Oct 22, 2007
//---------------------

package org.cleversafe.layer.target.tasks;

import java.nio.ByteBuffer;

import org.apache.log4j.Logger;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.exceptions.BlockCorruptException;
import org.cleversafe.layer.block.exceptions.BlockIOException;
import org.cleversafe.util.RandomNumberUtils;
import org.cleversafe.util.rangelock.RangeReadWriteLock;
import org.jscsi.scsi.protocol.Command;
import org.jscsi.scsi.protocol.cdb.AbstractTransferCDB;
import org.jscsi.scsi.protocol.cdb.Read10;
import org.jscsi.scsi.protocol.cdb.Read12;
import org.jscsi.scsi.protocol.cdb.Read16;
import org.jscsi.scsi.protocol.cdb.Read6;
import org.jscsi.scsi.protocol.inquiry.InquiryDataRegistry;
import org.jscsi.scsi.protocol.mode.ModePageRegistry;
import org.jscsi.scsi.protocol.sense.exceptions.InvalidCommandOperationCodeException;
import org.jscsi.scsi.protocol.sense.exceptions.LogicalBlockAddressOutOfRangeException;
import org.jscsi.scsi.protocol.sense.exceptions.SenseException;
import org.jscsi.scsi.protocol.sense.exceptions.SynchronousDataTransferErrorException;
import org.jscsi.scsi.protocol.sense.exceptions.UnrecoveredReadErrorException;
import org.jscsi.scsi.tasks.Status;
import org.jscsi.scsi.transport.TargetTransportPort;

/**
 * @author John Quigley <jquigley@cleversafe.com>
 * @version $Id$
 */
public class ReadGridTask extends GridTask
{
   private static Logger _logger = Logger.getLogger(ReadGridTask.class);

   private static final int MIN_RETRY_SLEEP = 1000;
   private static final int MAX_RETRY_SLEEP = 5000;
   
   /**
    * SBC CDB Read command types
    */
   private enum ReadType
   {
      READ6, READ10, READ12, READ16, READ32;
   };

   /**
    * Creates a new ReadGridTask.
    */
   public ReadGridTask()
   {
      super("ReadGridTask");
   }

   /**
    * Creates a new ReadGridTask.
    * 
    * @param targetPort
    * @param command
    * @param modePageRegistry
    * @param inquiryDataRegistry
    * @param device
    */
   public ReadGridTask(
         final TargetTransportPort targetPort,
         final Command command,
         final ModePageRegistry modePageRegistry,
         final InquiryDataRegistry inquiryDataRegistry,
         final BlockDeviceController device,
         final RangeReadWriteLock rangeReadWriteLock)
   {
      super("ReadGridTask", targetPort, command, modePageRegistry, inquiryDataRegistry, device,
            rangeReadWriteLock);
   }

   /**
    * Determines the CDB READ type based upon an operation code.
    * 
    * @param operationCode The operation code we're inspecting
    * 
    * @return The ReadType of this operation
    * @throws InvalidCommandOperationCodeException 
    */
   private ReadType getReadType(final int operationCode)
         throws InvalidCommandOperationCodeException
   {
      if (operationCode == Read6.OPERATION_CODE)
      {
         return ReadType.READ6;
      }
      else if (operationCode == Read10.OPERATION_CODE)
      {
         return ReadType.READ10;
      }
      else if (operationCode == Read12.OPERATION_CODE)
      {
         return ReadType.READ12;
      }
      else if (operationCode == Read16.OPERATION_CODE)
      {
         return ReadType.READ16;
      }
      else
      {
         // READ32 and any other write types are not currently supported
         // TODO: exception should indicate that we don't support this command
         throw new InvalidCommandOperationCodeException();
      }
   }
   
   /**
    * Executes this particular task from within the TaskManager.
    * 
    * @throws InterruptedException
    * @throws SenseException
    */
   @Override
   protected void execute() throws InterruptedException, SenseException
   {
      if (_logger.isDebugEnabled())
      {
         _logger.debug("Executing grid read task: " + toString());
      }
      final AbstractTransferCDB cdb =
            (AbstractTransferCDB) getCommand().getCommandDescriptorBlock();
      final long longTransferLength = cdb.getTransferLength();

      // inspect for an error condition before proceeding
      if (longTransferLength < 0 || longTransferLength > Integer.MAX_VALUE
            || longTransferLength > getBlockDeviceController().getDeviceSize())
      {
         final ReadType readType =
               getReadType(getCommand().getCommandDescriptorBlock().getOperationCode());

         if (readType.equals(ReadType.READ6))
         {
            throw new LogicalBlockAddressOutOfRangeException(true, true, (byte) 4, 1);
         }
         else if (readType.equals(ReadType.READ10) || readType.equals(ReadType.READ12)
               || readType.equals(ReadType.READ16))
         {
            // field pointer is 2 for Read10, Read12, Read16
            throw new LogicalBlockAddressOutOfRangeException(true, true, 2);
         }
         else if (readType.equals(ReadType.READ32))
         {
            throw new LogicalBlockAddressOutOfRangeException(true, true, 12);
         }
      }

      final int transferLength = (int) longTransferLength;
      final long lba = cdb.getLogicalBlockAddress();

      final RangeReadWriteLock.Lock readLock =
            getRangeReadWriteLock().getReadRangeLock(lba, longTransferLength);
      readLock.acquire();

      byte[] readData;
      int numBlockIOErrors = 0;
      Exception lastException = null;
      while (true)
      {
         if (isAborted())
         {
            readLock.release();
            if (_logger.isDebugEnabled())
            {
               _logger.debug("Grid read task was aborted: " + toString());
               if (lastException != null)
               {
                  _logger.debug(numBlockIOErrors
                        + " read IO errors before abort, last exception caught for this read task: ",
                        lastException);
                  _logger.info(String.format("scsi I/O occured during read operation at lba %d", lba));
               }
            }
            return;
         }

         try
         {
            _logger.debug("Reading from block device");
            readData = getBlockDeviceController().readBlocks(lba, transferLength);
            break;
         }
         catch (final BlockIOException e1)
         {
            if (_logger.isTraceEnabled())
            {
               _logger.trace("Block read IO exception in ReadGridTask (lba: " + lba + " numBlocks: "
                     + transferLength + ")", e1);
            }
            numBlockIOErrors++;
            lastException = e1;
         }
         catch (final BlockCorruptException e1)
         {
            _logger.error("An corrupt block error occurred reading blocks (" + lba + ", "
                  + transferLength + "): " + e1);
            readLock.release();

            throw new UnrecoveredReadErrorException(true, lba, 0);
         }
         
         Thread.sleep(RandomNumberUtils.getRandomInt(MIN_RETRY_SLEEP, MAX_RETRY_SLEEP));
      }

      readLock.release();
      boolean writeSuccess = this.writeData(ByteBuffer.wrap(readData));

      if (!writeSuccess)
      {
         throw new SynchronousDataTransferErrorException();
      }

      writeResponse(Status.GOOD, null);

      if (_logger.isDebugEnabled())
      {
         _logger.debug("Successfuly completed grid read task: " + toString());
      }
   }
}
