
package org.cleversafe.layer.block;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.layer.block.exceptions.BlockLayerException;
import org.cleversafe.layer.block.exceptions.BlockTransactionException;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.ReadControllerFactory;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;
import org.cleversafe.storage.ss.SliceServerConfiguration;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.cleversafe.storage.ss.configuration.ConfigurationLoader;
import org.cleversafe.storage.ss.configuration.XMLConfigurationLoader;
import org.cleversafe.test.BaseTest;
import org.cleversafe.util.BoundedThreadPoolExecutor;
import org.cleversafe.vault.FullAccountCreator;
import org.cleversafe.vault.Vault;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

public class BlockDeviceUDCTest extends BaseTest
{
   private static Logger _logger = Logger.getLogger(BlockDeviceUDCTest.class);

   // Number of requests per test
   private static int numRequests;

   // Number of concurrent requests
   private static int numConcurrentRequests;

   // Commonly-seen request sizes, in ascending order
   private static final int[] BLOCKS_PER_REQUEST = {
         1, 8, 16, 32, 128
   };

   // SliceServer object
   private static SliceServerDaemon[] ssSet = null;

   // SliceServer Threads
   private static Thread[] daemonThreads = null;

   Vault vault;
   FullAccountCreator accountCreator;
   BlockDeviceController bdc;
   Random rng;

   static SliceServerDaemon sliceServerGroup = null;

   private static int performanceBlockSize;
   private static int ingestionDelay;

   @BeforeClass
   public static void runBeforeTests() throws Exception
   {
      // Start remote stores if necessary
      String storeString = System.getProperty("org.cleversafe.storage.ss.daemons");
      int stores = storeString != null ? Integer.parseInt(storeString) : 0;

      numRequests = Integer.getInteger("test.requests", 1024);
      numConcurrentRequests = Integer.getInteger("test.concurrency", 20);
      performanceBlockSize =
            Integer.getInteger("test.performance.blocksize",
                  BLOCKS_PER_REQUEST[BLOCKS_PER_REQUEST.length - 1]);
      ingestionDelay = Integer.getInteger("test.ingestion-delay", 0);

      System.out.println("UDC test "
            + System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor")
            + "(concurrency:" + numConcurrentRequests + " requests: " + numRequests + ")");

      if (stores > 0)
      {
         // Check required system properties
         String serverConfiguration =
               System.getProperty("org.cleversafe.storage.ss.xml.configuration");
         String bindingsConfiguration = System.getProperty("org.cleversafe.xml.configuration");

         if (serverConfiguration == null)
         {
            fail("Server configuration file not specified");
         }

         if (bindingsConfiguration == null)
         {
            fail("Bindings configuration file not specified");
         }

         startSliceServers(stores);
      }

      // Wait until the grid comes up....
      Thread.sleep(5000);
   }

   private static void startSliceServers(int daemonCount)
   {
      String serverConfiguration =
            System.getProperty("org.cleversafe.storage.ss.xml.configuration");

      // Launch daemon
      try
      {
         // Create daemons with generated configurations
         ssSet = new SliceServerDaemon[daemonCount];
         for (int daemonID = 1; daemonID <= ssSet.length; daemonID++)
         {
            Properties properties = new Properties();
            properties.setProperty("DAEMON.ID", Integer.toString(daemonID));

            ConfigurationLoader configLoader = new XMLConfigurationLoader(serverConfiguration);
            SliceServerConfiguration configuration = configLoader.getConfiguration(properties);

            ssSet[daemonID - 1] = new SliceServerDaemon(configuration);
         }

         // Create threads for non-primary daemons
         daemonThreads = new Thread[ssSet.length];
         for (int a = 0; a < ssSet.length; a++)
         {
            daemonThreads[a] = new Thread(ssSet[a], "daemon-" + (a + 1));
            daemonThreads[a].start();
         }

         // Wait until servers have started
         for (SliceServerDaemon daemon : ssSet)
         {
            try
            {
               daemon.awaitStart();
            }
            catch (InterruptedException ignored)
            {
            }
         }
      }
      catch (ServerConfigurationLoadException ex)
      {
         _logger.fatal("Unable to load configuration", ex);
         fail(ex.getMessage());
      }
   }

   @AfterClass
   public static void runAfterTests() throws Exception
   {
      if (BlockDeviceUDCTest.ssSet != null)
      {
         // Shutdown SliceServers
         for (SliceServerDaemon ss : ssSet)
         {
            ss.shutdown();
         }

         // Join Threads
         for (int a = 0; a < daemonThreads.length; a++)
         {
            daemonThreads[a].join();
         }
      }
   }

   @Before
   public void setUp()
   {
      this.rng = new Random(); // Intentional repeatability

      try
      {
         String vaultDescriptorPath =
               System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor");
         assertNotNull(
               "Vault descriptor must be provided (org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor)",
               vaultDescriptorPath);

         this.accountCreator = new FullAccountCreator(vaultDescriptorPath);
         this.vault = this.accountCreator.getVault();

         System.out.println("==============");
         System.out.println(this.vault);
         System.out.println("==============");

         // Create controller
         assertTrue(this.vault instanceof BlockDeviceVault);

         BindingsProvider bindingsProvider =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);

         ReadControllerFactory rfc =
               bindingsProvider.getDefaultImplementation(ReadControllerFactory.class);
         WriteControllerFactory wfc =
               bindingsProvider.getDefaultImplementation(WriteControllerFactory.class);

         GridController controller = new GridController(this.vault, wfc, rfc);
         this.bdc = new BlockDeviceController((BlockDeviceVault) this.vault, controller);
         this.bdc.startup();
         assertNotNull(this.bdc);
      }
      catch (Exception e)
      {
         _logger.error("Initialization error", e);
         fail(e.getMessage());
      }
   }

   @After
   public void tearDown()
   {
      try
      {
         if (this.bdc != null)
         {
            this.bdc.shutdown();
            _logger.debug("Block device controller was shutdown");
         }
         if (this.accountCreator != null)
         {
            this.accountCreator.cleanup();
         }
      }
      catch (final Exception e)
      {
         _logger.warn("Exception on tearDown", e);
      }

      this.bdc = null;
      this.rng = null;
      this.accountCreator = null;
   }

   @Test
   public void noTest()
   {
   }

   /**
    * Stress test the system by sending a number of differently-sized requests concurrently.
    */
   @Test
   public void stressTest() throws InterruptedException
   {
      // Ensure that it is impossible to have two concurrent requests for the same region of blocks
      assert (numConcurrentRequests * BLOCKS_PER_REQUEST[BLOCKS_PER_REQUEST.length - 1]) <= this.bdc.getNumBlocks();

      ExecutorService executor = new BoundedThreadPoolExecutor("stress", numConcurrentRequests);

      class Request implements Callable<Boolean>
      {
         private final long requestNum;
         private final long firstBlock;
         private final int numBlocks;

         public Request(long requestNum, long firstBlock, int numBlocks)
         {
            this.requestNum = requestNum;
            this.firstBlock = firstBlock;
            this.numBlocks = numBlocks;
         }

         public Boolean call() throws Exception
         {
            _logger.debug("Beginning request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            int blockSize = bdct.bdc.getBlockSize();
            int requestSize = blockSize * this.numBlocks;

            byte[] data = new byte[requestSize];
            for (int i = 0; i < this.numBlocks; ++i)
            {
               byte[] block = new byte[blockSize];
               Arrays.fill(block, (byte) ((this.firstBlock + i + 1) % 256));
               System.arraycopy(block, 0, data, i * blockSize, blockSize);
            }

            byte[] compare;
            try
            {
               bdct.bdc.writeBlocks(this.firstBlock, this.numBlocks, data);
               compare = bdct.bdc.readBlocks(this.firstBlock, this.numBlocks);
            }
            catch (final Exception e)
            {
               throw e;
            }

            boolean success = Arrays.equals(data, compare);
            _logger.debug("Request complete: " + this.requestNum + " (" + success + ")");
            return success;
         }
      }

      _logger.info("Beginning requests...");
      long beginTime = System.currentTimeMillis();
      long bytesSent = 0;

      try
      {
         // Queue all requests for processing
         ArrayList<Future<Boolean>> results = new ArrayList<Future<Boolean>>(numRequests);
         long currentFirstBlock = 0;
         for (int i = 0; i < numRequests; i++)
         {
            long firstBlock = currentFirstBlock;
            int numBlocks =
                  (int) Math.min(BLOCKS_PER_REQUEST[this.rng.nextInt(BLOCKS_PER_REQUEST.length)],
                        this.bdc.getNumBlocks() - firstBlock);
            currentFirstBlock = (currentFirstBlock + numBlocks) % this.bdc.getNumBlocks();

            results.add(executor.submit(new Request(i, firstBlock, numBlocks)));
            bytesSent += numBlocks * this.bdc.getBlockSize();
         }

         // Check result of all requests
         _logger.info("Requests done, checking results...");
         for (int i = 0; i < numRequests; i++)
         {
            Future<Boolean> result = results.get(i);

            try
            {
               Boolean success = result.get();
               assertTrue("UDC comparison failed for request: " + i, success);
            }
            catch (final Exception e)
            {
               _logger.error("Request error", e);
               fail(e.getMessage());
            }
         }

         long duration = System.currentTimeMillis() - beginTime;
         System.out.println(String.format("Total time: %dm%.1fs, UDC: %.3fMB (%.1fMbps)", duration
               / (60 * 1000), (duration % (60 * 1000)) / 1000.0, bytesSent / (1024 * 1024.0),
               (bytesSent * 1000 * 8) / (duration * 1024 * 1024.0)));
      }
      finally
      {
         executor.shutdown();
         executor.awaitTermination(30, TimeUnit.SECONDS);
      }
   }

   /**
    * Measure peak performance by writing and then reading maximally sized chunks This test does NOT
    * check correctness of data.
    * 
    * @throws ExecutionException
    * @throws InterruptedException
    */
   @Test
   public void performanceTest() throws InterruptedException, ExecutionException
   {
      // Ensure that it is impossible to have two concurrent requests for the same region of blocks
      assert (numConcurrentRequests * BLOCKS_PER_REQUEST[BLOCKS_PER_REQUEST.length - 1]) <= this.bdc.getNumBlocks();

      // Make configurable # of blocks per request
      final int blocksPerRequest = performanceBlockSize;
      final long bytesSent = (long) numRequests * blocksPerRequest * this.bdc.getBlockSize();
      ExecutorService executor =
            new BoundedThreadPoolExecutor("performance", numConcurrentRequests);

      System.out.println(String.format(
            "Performance test: %.3fMB, %d-byte blocks, %d requests, %d blocks per request, %d threads, %dms ingestion delay",
            bytesSent / (1024 * 1024.), this.bdc.getBlockSize(), numRequests, blocksPerRequest,
            numConcurrentRequests, ingestionDelay));

      abstract class Request implements Callable<Void>
      {
         protected long requestNum;
         protected long firstBlock;

         public Request(long requestNum, long firstBlock)
         {
            this.requestNum = requestNum;
            this.firstBlock = firstBlock;
         }
      }

      class WriteRequest extends Request
      {
         public WriteRequest(long requestNum, long firstBlock)
         {
            super(requestNum, firstBlock);
         }

         public Void call() throws Exception
         {
            _logger.debug("Beginning write request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            byte[] data = new byte[blocksPerRequest * bdct.bdc.getBlockSize()];
            try
            {
               bdct.bdc.writeBlocks(this.firstBlock, blocksPerRequest, data);
            }
            catch (final Exception e)
            {
               _logger.info("Write exception", e);
               throw e;
            }

            if (ingestionDelay > 0)
            {
               Thread.sleep(ingestionDelay);
            }

            _logger.debug("Write request complete: " + this.requestNum);
            return null;
         }
      }

      class ReadRequest extends Request
      {
         public ReadRequest(long requestNum, long firstBlock)
         {
            super(requestNum, firstBlock);
         }

         public Void call() throws Exception
         {
            _logger.debug("Beginning read request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            try
            {
               bdct.bdc.readBlocks(this.firstBlock, blocksPerRequest);
            }
            catch (final Exception e)
            {
               _logger.info("Read exception", e);
               throw e;
            }

            if (ingestionDelay > 0)
            {
               Thread.sleep(ingestionDelay);
            }

            _logger.debug("Read request complete: " + this.requestNum);
            return null;
         }
      }

      /**
       * Time read and write phases
       */
      List<Future<Void>> results = null;
      long beginTime, duration;

      try
      {
         /**
          * Write requests
          */

         List<Callable<Void>> tasks = new ArrayList<Callable<Void>>(numRequests);
         long currentFirstBlock = 0;

         for (int i = 0; i < numRequests; i++)
         {
            long firstBlock = currentFirstBlock;
            currentFirstBlock = (currentFirstBlock + blocksPerRequest) % this.bdc.getNumBlocks();

            tasks.add(i, new WriteRequest(i, firstBlock));
         }

         _logger.info("Beginning write requests...");
         beginTime = System.currentTimeMillis();

         try
         {
            results = executor.invokeAll(tasks);
         }
         catch (final InterruptedException e)
         {
            fail(e.getMessage());
         }

         for (Future<Void> result : results)
         {
            result.get();
         }

         duration = System.currentTimeMillis() - beginTime;
         System.out.println(String.format("Write time: %dm%.1fs, Bytes UDC: %.3fMB (%.1fMbps)",
               duration / (60 * 1000), (duration % (60 * 1000)) / 1000.0, bytesSent
                     / (1024 * 1024.0), (bytesSent * 1000 * 8) / (duration * 1024 * 1024.0)));

         /**
          * Read requests
          */
         tasks = new ArrayList<Callable<Void>>(numRequests);
         currentFirstBlock = 0;
         for (int i = 0; i < numRequests; i++)
         {
            long firstBlock = currentFirstBlock;
            currentFirstBlock = (currentFirstBlock + blocksPerRequest) % this.bdc.getNumBlocks();

            tasks.add(i, new ReadRequest(i, firstBlock));
         }

         _logger.info("Beginning read requests...");
         beginTime = System.currentTimeMillis();

         try
         {
            results = executor.invokeAll(tasks);
         }
         catch (final InterruptedException e)
         {
            fail(e.getMessage());
         }

         for (Future<Void> result : results)
         {
            result.get();
         }

         duration = System.currentTimeMillis() - beginTime;
         System.out.println(String.format("Read time: %dm%.1fs, Bytes UDC: %.3fMB (%.1fMbps/s)",
               duration / (60 * 1000), (duration % (60 * 1000)) / 1000.0, bytesSent
                     / (1024 * 1024.0), (bytesSent * 1000 * 8) / (duration * 1024 * 1024.0)));
      }
      finally
      {
         executor.shutdown();
         executor.awaitTermination(30, TimeUnit.SECONDS);
      }
   }

   /**
    * Attempts to produce a race between writes on the same blocks
    * 
    * @throws ExecutionException
    * @throws InterruptedException
    */
   @Test
   public void transactionTest() throws InterruptedException, ExecutionException
   {
      final int NUM_WRITES = 3;

      // Always send the maximum number of blocks per request
      final int blocksPerRequest = BLOCKS_PER_REQUEST[BLOCKS_PER_REQUEST.length - 1];
      ExecutorService executor = new BoundedThreadPoolExecutor("transaction", NUM_WRITES);

      class WriteRequest implements Callable<Void>
      {
         protected long requestNum;
         protected long firstBlock;

         public WriteRequest(long requestNum, long firstBlock)
         {
            this.requestNum = requestNum;
            this.firstBlock = firstBlock;
         }

         public Void call() throws Exception
         {
            _logger.debug("Beginning write request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            while (true)
            {
               byte[] data = new byte[blocksPerRequest * bdct.bdc.getBlockSize()];
               try
               {
                  bdct.bdc.writeBlocks(this.firstBlock, blocksPerRequest, data);
                  _logger.debug("Write request complete: " + this.requestNum);
                  return null;
               }
               catch (BlockTransactionException ex)
               {
                  // Try again
                  _logger.debug("Retrying write request: " + this.requestNum);
               }
               catch (Exception ex)
               {
                  throw ex;
               }
            }
         }
      }

      // Perform test
      try
      {
         for (int i = 0; i < numRequests; ++i)
         {
            long firstBlock = this.rng.nextInt(128);

            // Begin all requests
            _logger.info(String.format("Round %d: Beginning %d requests", i, NUM_WRITES));
            ArrayList<Future<Void>> results = new ArrayList<Future<Void>>(NUM_WRITES);
            for (int j = 0; j < NUM_WRITES; j++)
            {
               results.add(executor.submit(new WriteRequest(j, firstBlock)));
            }

            // Check result of all requests
            _logger.info(String.format("Round %d: Requests done, checking results...", i));
            for (int j = 0; j < results.size(); ++j)
            {
               try
               {
                  results.get(j).get();
                  _logger.debug(String.format("Round %d: Write %d suceeded", i, j));
               }
               catch (ExecutionException ex)
               {
                  Throwable t = ex.getCause();
                  _logger.debug(String.format("Round %d: Write %d failed with exception", i, j), t);
               }
            }

            // All writes done, try a read
            try
            {
               this.bdc.readBlocks(firstBlock, blocksPerRequest);
               _logger.info(String.format("Round %d: Read successful", i));
            }
            catch (BlockLayerException ex)
            {
               _logger.error(String.format("Round %d: Read failed with exception", i), ex);
               fail(ex.getMessage());
            }
         }
      }
      finally
      {
         executor.shutdown();
         executor.awaitTermination(30, TimeUnit.SECONDS);
      }
   }

   /**
    * Attempts to produce a deadlock by reading and writing the same blocks concurrently
    * 
    * @throws ExecutionException
    * @throws InterruptedException
    */
   //@Test
   public void deadlockTest() throws InterruptedException, ExecutionException
   {
      // Always send the maximum number of blocks per request
      final int blocksPerRequest = BLOCKS_PER_REQUEST[BLOCKS_PER_REQUEST.length - 1];
      ExecutorService executor = new BoundedThreadPoolExecutor("deadlock", numConcurrentRequests);

      abstract class Request implements Callable<Void>
      {
         protected long requestNum;
         protected long firstBlock;

         public Request(long requestNum, long firstBlock)
         {
            this.requestNum = requestNum;
            this.firstBlock = firstBlock;
         }
      }

      class WriteRequest extends Request
      {
         public WriteRequest(long requestNum, long firstBlock)
         {
            super(requestNum, firstBlock);
         }

         public Void call() throws Exception
         {
            _logger.debug("Beginning write request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            byte[] data = new byte[blocksPerRequest * bdct.bdc.getBlockSize()];
            try
            {
               bdct.bdc.writeBlocks(this.firstBlock, blocksPerRequest, data);
            }
            catch (final Exception e)
            {
               throw e;
            }

            _logger.debug("Write request complete: " + this.requestNum);
            return null;
         }
      }

      class ReadRequest extends Request
      {
         public ReadRequest(long requestNum, long firstBlock)
         {
            super(requestNum, firstBlock);
         }

         public Void call() throws Exception
         {
            _logger.debug("Beginning read request: " + this.requestNum);
            BlockDeviceUDCTest bdct = BlockDeviceUDCTest.this;

            try
            {
               bdct.bdc.readBlocks(this.firstBlock, blocksPerRequest);
            }
            catch (final Exception e)
            {
               throw e;
            }

            _logger.debug("Read request complete: " + this.requestNum);
            return null;
         }
      }

      try
      {
         // Queue all requests for processing
         ArrayList<Future<Void>> results = new ArrayList<Future<Void>>(numRequests);
         for (int i = 0; i < numRequests; i++)
         {
            long firstBlock = this.rng.nextInt(128);

            results.add(executor.submit(new WriteRequest(i, firstBlock)));
            results.add(executor.submit(new ReadRequest(i, firstBlock)));
         }

         // Check result of all requests
         _logger.info("Requests done, checking results...");
         for (int i = 0; i < numRequests; i++)
         {
            results.get(i).get();
         }
      }
      finally
      {
         executor.shutdown();
         executor.awaitTermination(30, TimeUnit.SECONDS);
      }
   }

   /**
    * CLI execution
    * @param args
    */
   public static void main(String[] args) throws Exception
   {
      Logger.getRootLogger().setLevel(Level.WARN);

      try
      {
         BlockDeviceUDCTest.runBeforeTests();

         BlockDeviceUDCTest test = new BlockDeviceUDCTest();
         test.setUp();
         test.performanceTest();
         test.tearDown();
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
      finally
      {
         BlockDeviceUDCTest.runAfterTests();
      }

      System.exit(0);
   }
}
