
package org.cleversafe.layer.block;

import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.Arrays;
import java.util.Random;

import org.apache.log4j.Logger;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.storage.ss.SliceServerDaemon;
import org.cleversafe.vault.FullAccountCreator;
import org.cleversafe.vault.Vault;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * This test is designed to test the proper operation of a UDC test when either some slices are 
 * returned corrupted or when some slices are returned with the wrong transaction IDs.  Which feature
 * is under test depends on how the UnreliableSliceStore is configured to run on the server side.
 * 
 * @author Jason Resch
 *
 */
public class UnreliableBlockDeviceUDCTest
{
   private static Logger _logger = Logger.getLogger(UnreliableBlockDeviceUDCTest.class);

   // Base port for locally-started slice servers
   private static final int BASE_SERVER_PORT = 5000;

   // SliceServer object
   private static SliceServerDaemon ss = null;

   Vault vault;
   FullAccountCreator accountCreator;
   BlockDeviceController bdc;
   Random rng;

   static SliceServerDaemon sliceServerGroup = null;

   @BeforeClass
   public static void runBeforeTests() throws Exception
   {
      // Start remote stores if necessary
      String storeString = System.getProperty("org.cleversafe.ss.test.servercount");
      int stores = storeString != null ? Integer.parseInt(storeString) : 0;

      if (stores > 0)
      {
         _logger.info("Starting " + stores + " slice servers " + "(base port: " + BASE_SERVER_PORT
               + ")");

         String[] args = new String[2];
         args[0] = "--host=127.0.0.1";
         args[1] = "--port=" + BASE_SERVER_PORT;

         //UnreliableBlockDeviceUDCTest.ss = new SliceServerDaemon(args);
         //ss.startSliceServers();

         // Wait until the grid comes up....
         Thread.sleep(5000);
      }
   }

   @AfterClass
   public static void runAfterTests() throws Exception
   {
      if (UnreliableBlockDeviceUDCTest.ss != null)
      {
         UnreliableBlockDeviceUDCTest.ss.shutdown();
      }
   }

   @Before
   public void setUp()
   {
      this.rng = new Random(); // Intentional repeatability

      try
      {
         String vaultDescriptorPath =
               System.getProperty("org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor");
         assertNotNull(
               "Vault descriptor must be provided (org.cleversafe.block.BlockDeviceUDCTest.vaultDescriptor)",
               vaultDescriptorPath);

         this.accountCreator = new FullAccountCreator(vaultDescriptorPath);
         this.vault = this.accountCreator.getVault();

         // Create controller
         assertTrue(this.vault instanceof BlockDeviceVault);
         this.bdc = new BlockDeviceController((BlockDeviceVault) this.vault);
         this.bdc.startup();
         
         _logger.debug("Block device controller was started");
         assertNotNull(this.bdc);
      }
      catch (Exception e)
      {
         _logger.error("Initialization error", e);
         fail(e.getMessage());
      }
   }

   @After
   public void tearDown()
   {
      try
      {
         this.bdc.shutdown();

         _logger.debug("Block device controller was shutdown");

         this.accountCreator.cleanup();
      }
      catch (final Exception e)
      {
         _logger.warn("Exception on tearDown", e);
      }

      this.bdc = null;
      this.rng = null;
      this.accountCreator = null;
   }

   @Test
   public void testUnreliableRead()
   {
      int firstBlock = 0;
      int numBlocks = 8;
      
      int blockSize = this.bdc.getBlockSize();
      int requestSize = blockSize * numBlocks;

      byte[] data = new byte[requestSize];      
      this.rng.nextBytes(data);

      byte[] compare = null;
      try
      {
         bdc.writeBlocks(firstBlock, numBlocks, data);
         compare = this.bdc.readBlocks(firstBlock, numBlocks);
      }
      catch (final Exception e)
      {
         fail(e.getMessage());
      }
      
      assertNotNull(compare);

      boolean success = Arrays.equals(data, compare);
      assertTrue(success);
      _logger.debug("Request complete: success = " + success);
   }
}
