
package org.cleversafe.block;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import org.apache.commons.lang.NotImplementedException;
import org.apache.log4j.xml.DOMConfigurator;
import org.cleversafe.layer.block.BlockDeviceController;
import org.cleversafe.layer.block.BlockDeviceVault;
import org.cleversafe.layer.grid.CorruptedSliceObserver;
import org.cleversafe.layer.grid.DataSource;
import org.cleversafe.layer.grid.GridController;
import org.cleversafe.layer.grid.ReadController;
import org.cleversafe.layer.grid.ReadControllerFactory;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.WriteController;
import org.cleversafe.layer.grid.WriteControllerFactory;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerInformationDispersalException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.vault.Vault;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

// TODO: Describe class or interface
public class BlockDeviceControllerTest
{
   /** Size of a block in bytes */
   public static int BLOCK_SIZE = 4096;

   /** Size of the device in bytes */
   public static long DEVICE_SIZE = 1024 * 1024;

   /** Minimum time for a task to execute in ms */
   public static int MIN_TASK_TIME = 500;

   /** Maximum time for a task to execute in ms */
   public static int MAX_TASK_TIME = 1500;

   // Reproducible randomness
   Random rng = new Random();

   BlockDeviceVault vault;

   @BeforeClass
   public static void initialize()
   {
      DOMConfigurator.configure("log4j.xml");
   }

   @Before
   public void setUp()
   {
      this.vault = new TestBlockDeviceVault();
   }

   @After
   public void tearDown()
   {
      this.vault = null;
   }

   // ///////////////////////////////////////////////////////////////////////////////////////////////
   // General
   // ///////////////////////////////////////////////////////////////////////////////////////////////

   /**
    * Test various values, getters, & setters
    */
   @Test
   public void testParameters()
   {
      BlockDeviceController bdc =
            new BlockDeviceController(this.vault, new GridController(this.vault,
                  new TestWriteControllerFactory(new TestGridController()),
                  new TestReadControllerFactory(new TestGridController())));

      try
      {
         bdc.startup();

         assertEquals(bdc.getBlockSize(), this.vault.getBlockSize());
         assertEquals(bdc.getDeviceSize(), DEVICE_SIZE);
         assertEquals(bdc.getNumBlocks(), this.vault.getNumBlocks());

         bdc.shutdown();
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   // ///////////////////////////////////////////////////////////////////////////////////////////////
   // Write
   // ///////////////////////////////////////////////////////////////////////////////////////////////

   /**
    * Test writing to the beginning of the device
    */
   @Test
   public void beginningWriteTest() throws Exception
   {
      this.testWriteBlock(0);
   }

   /**
    * Test writing to the end of the device
    */
   @Test
   public void endWriteTest() throws Exception
   {
      long lastBlock = this.vault.getNumBlocks() - 1;
      this.testWriteBlock(lastBlock);
   }

   /**
    * Test writing before the beginning of the device
    */
   /*
    * FIXME: D-01124, Support this test @Test(expected=IndexOutOfBoundsException.class) public void
    * underWriteTest() throws Exception { BlockDeviceController bdc = new BlockDeviceController(
    * this.vault, new TestSliceStoreController());
    * 
    * byte[] blocks = this.generateData(10); bdc.writeBlocks(-5, 10, blocks); bdc.shutdown(); //
    * Check that the transaction has been rolled back }
    */

   /**
    * Test writing after the end of the device
    */
   /*
    * FIXME: D-01124, Support this test @Test(expected=IndexOutOfBoundsException.class) public void
    * overWriteTest() throws Exception { long lastBlock = this.vault.getNumBlocks() - 1;
    * 
    * BlockDeviceController bdc = new BlockDeviceController( this.vault, new
    * TestSliceStoreController());
    * 
    * byte[] blocks = this.generateData(10); bdc.writeBlocks(lastBlock-5, 10, blocks);
    * bdc.shutdown(); // Check that the transaction has been rolled back }
    */

   /**
    * Test write when controller throws an exception
    */
   @Test
   public void testFailedWrite()
   {
      final byte[] blocks = this.generateData(1);

      class MyWriteController extends TestGridController
      {
         @Override
         public void writeImpl(DataSource data) throws ControllerDataTransformationException,
               ControllerInformationDispersalException, ControllerIOException,
               ControllerIllegalSourceNameException, ControllerTransactionException,
               ControllerGridStateUnknownException
         {
            throw new ControllerIOException("Testing write");
         }

         @Override
         public void rollback() throws ControllerTransactionException
         {
            throw new ControllerTransactionException("Testing rollback");
         }
      }

      BlockDeviceController bdc =
            new BlockDeviceController(this.vault, new GridController(this.vault,
                  new TestWriteControllerFactory(new MyWriteController()),
                  new TestReadControllerFactory(new TestGridController())));

      try
      {
         bdc.startup();
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

      try
      {
         bdc.writeBlocks(0, 1, blocks);

         // Should not get here
         assertTrue("Exception expected", false);
      }
      catch (final Exception e)
      {
         // FIXME: Test for more meaningful exceptions
      }

      try
      {
         bdc.shutdown();
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   protected void testWriteBlock(final long blockNum) throws Exception
   {
      final byte[] blocks = this.generateData(1);

      class MyWriteController extends TestGridController
      {
         @Override
         public void writeImpl(DataSource data) throws ControllerDataTransformationException,
               ControllerInformationDispersalException, ControllerIOException,
               ControllerIllegalSourceNameException, ControllerTransactionException,
               ControllerGridStateUnknownException
         {
            super.write(data);

            assertTrue(data.getName().equals(new SourceName(Long.toString(blockNum))));

            byte[] test = data.getData();
            for (int i = 0; i < test.length; ++i)
            {
               assertTrue(test[i] == blocks[i]);
            }
         }
      }

      BlockDeviceController bdc =
            new BlockDeviceController(this.vault, new GridController(this.vault,
                  new TestWriteControllerFactory(new MyWriteController()),
                  new TestReadControllerFactory(new TestGridController())));
      bdc.writeBlocks(blockNum, 1, blocks);
      assertEquals(1, bdc.getStatsNumWrites());

      bdc.shutdown();
   }

   // ///////////////////////////////////////////////////////////////////////////////////////////////
   // Read
   // ///////////////////////////////////////////////////////////////////////////////////////////////

   /**
    * Test reading from the beginning of the device
    * 
    * @throws Exception
    */
   @Test
   public void beginningReadTest() throws Exception
   {
      this.testReadBlock(0);
   }

   /**
    * Test reading from the end of the device
    */
   @Test
   public void endReadTest() throws Exception
   {
      long lastBlock = this.vault.getNumBlocks() - 1;
      this.testReadBlock(lastBlock);
   }

   // FIXME: Support checking & testing for under and over-reads

   /**
    * Test handling of a failed read FIXME: This test fails, defect D-01125
    */
   @Test
   public void testFailedRead()
   {
      class MyReadController extends TestGridController
      {
         @Override
         public DataSource readImpl(SourceName name) throws ControllerIOException
         {
            throw new ControllerIOException("Testing read");
         }
      }

      BlockDeviceController bdc =
            new BlockDeviceController(this.vault, new GridController(this.vault,
                  new TestWriteControllerFactory(new TestGridController()),
                  new TestReadControllerFactory(new MyReadController())));

      try
      {
         bdc.startup();
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }

      try
      {
         bdc.readBlocks(0, 1);

         // Should not get here
         assertTrue("Exception expected", false);
      }
      catch (final Exception e)
      {
         // FIXME: Test for more meaningful exceptions
      }

      try
      {
         bdc.shutdown();
      }
      catch (Exception e)
      {
         fail(e.getMessage());
      }
   }

   protected void testReadBlock(final long blockNum) throws Exception
   {
      class MyReadController extends TestGridController
      {
         @Override
         public DataSource readImpl(SourceName name) throws ControllerIOException,
               ControllerDataTransformationException, ControllerInformationDispersalException,
               ControllerGridStateUnknownException, ControllerIllegalSourceNameException
         {
            assertTrue(name.equals(new SourceName(Long.toString(blockNum))));
            return super.read(name);
         }
      }

      BlockDeviceController bdc =
            new BlockDeviceController(this.vault, new GridController(this.vault,
                  new TestWriteControllerFactory(new TestGridController()),
                  new TestReadControllerFactory(new MyReadController())));
      bdc.readBlocks(blockNum, 1);
      assertEquals(1, bdc.getStatsNumReads());

      bdc.shutdown();
   }

   // ///////////////////////////////////////////////////////////////////////////////////////////////
   // Helpers
   // ///////////////////////////////////////////////////////////////////////////////////////////////

   byte[] generateData(int numBlocks)
   {
      byte[] data = new byte[this.vault.getBlockSize() * numBlocks];
      this.rng.nextBytes(data);
      return data;
   }

   class TestBlockDeviceVault extends BlockDeviceVault
   {
      public TestBlockDeviceVault()
      {
         super();
         this.setNumBlocks(DEVICE_SIZE / BLOCK_SIZE);
         this.setBlockSize(BLOCK_SIZE);
      }

      @Override
      public void startSessions()
      {
         // Do nothing
      }

      @Override
      public void endSessions()
      {
         // Do nothing
      }
   }

   class TestWriteControllerFactory implements WriteControllerFactory
   {
      TestGridController wcc;

      public TestWriteControllerFactory(TestGridController wcc)
      {
         this.wcc = wcc;
      }

      public WriteController getInstance(Vault vault) throws ControllerException
      {
         try
         {
            return (WriteController) wcc.clone();
         }
         catch (final Exception ex)
         {
            fail(ex.getMessage());
            return null;
         }
      }

      public void release(WriteController wc)
      {
         // Do nothing
      }
   }

   class TestReadControllerFactory implements ReadControllerFactory
   {
      TestGridController rcc;

      public TestReadControllerFactory(TestGridController rcc)
      {
         this.rcc = rcc;
      }

      public ReadController getInstance(Vault vault) throws ControllerException
      {
         try
         {
            return (ReadController) rcc.clone();
         }
         catch (final Exception ex)
         {
            fail(ex.getMessage());
            return null;
         }
      }

      public void release(ReadController wc)
      {
         // Do nothing
      }
   }

   class TestGridController implements WriteController, ReadController, Cloneable
   {
      BlockDeviceVault vault;
      int minTaskTime;
      int maxTaskTime;

      boolean inTransaction = false;
      long transactionId = -1;

      public TestGridController()
      {
         this.setVault(BlockDeviceControllerTest.this.vault);

         assert MIN_TASK_TIME <= MAX_TASK_TIME;
         this.minTaskTime = MIN_TASK_TIME;
         this.maxTaskTime = MAX_TASK_TIME;

         this.initialize();
      }

      private void initialize()
      {
         // Start transaction
         this.inTransaction = true;
         this.transactionId = 12345;
      }

      /**
       * Simulate execution of a task
       */
      protected void executeTask()
      {
         int waitMs =
               (int) (Math.random() * (this.maxTaskTime - this.minTaskTime)) + this.minTaskTime;

         try
         {
            Thread.sleep(waitMs);
         }
         catch (final InterruptedException e)
         {
         }
      }

      public void commit() throws ControllerTransactionException, ControllerIOException
      {
         this.executeTask();
         this.inTransaction = false;
         this.transactionId = -1;
      }

      public void rollback() throws ControllerTransactionException, ControllerIOException
      {
         this.executeTask();
         this.inTransaction = false;
         this.transactionId = -1;
      }

      public boolean inTransaction()
      {
         return this.inTransaction;
      }

      public long getTransactionId()
      {
         return this.transactionId;
      }

      public boolean existGridSource(SourceName name)
      {
         return true;
      }

      public Vault getVault()
      {
         return this.vault;
      }

      public void setVault(Vault vault)
      {
         assertTrue(vault instanceof BlockDeviceVault);
         this.vault = (BlockDeviceVault) vault;
      }

      public void shutdown()
      {
         this.executeTask();
      }

      protected void writeImpl(DataSource data) throws ControllerDataTransformationException,
            ControllerInformationDispersalException, ControllerIOException,
            ControllerIllegalSourceNameException, ControllerTransactionException,
            ControllerGridStateUnknownException
      {
         this.executeTask();
      }

      public final void write(DataSource data) throws ControllerDataTransformationException,
            ControllerInformationDispersalException, ControllerIOException,
            ControllerIllegalSourceNameException, ControllerTransactionException,
            ControllerGridStateUnknownException
      {
         this.writeImpl(data);
      }

      public final void write(List<DataSource> sources)
            throws ControllerDataTransformationException, ControllerInformationDispersalException,
            ControllerIOException, ControllerIllegalSourceNameException,
            ControllerTransactionException, ControllerGridStateUnknownException
      {
         for (DataSource source : sources)
         {
            this.writeImpl(source);
         }
      }

      public boolean isOperational()
      {
         return true;
      }

      public final List<DataSource> read(List<SourceName> names)
            throws ControllerDataTransformationException, ControllerInformationDispersalException,
            ControllerIOException, ControllerGridStateUnknownException,
            ControllerIllegalSourceNameException
      {
         List<DataSource> sources = new ArrayList<DataSource>(names.size());
         for (SourceName name : names)
         {
            sources.add(this.readImpl(name));
         }
         return sources;
      }

      public final DataSource read(SourceName name) throws ControllerDataTransformationException,
            ControllerInformationDispersalException, ControllerIOException,
            ControllerGridStateUnknownException, ControllerIllegalSourceNameException
      {
         return readImpl(name);
      }

      protected DataSource readImpl(SourceName name) throws ControllerDataTransformationException,
            ControllerInformationDispersalException, ControllerIOException,
            ControllerGridStateUnknownException, ControllerIllegalSourceNameException
      {
         this.executeTask();
         return new DataSource(name, this.transactionId, new byte[this.vault.getBlockSize()]);
      }

      @Override
      public Object clone() throws CloneNotSupportedException
      {
         TestGridController c = (TestGridController) super.clone();
         c.initialize();
         return c;
      }

      public void addCorruptedSliceObserver(CorruptedSliceObserver corruptedSliceObserver)
      {
         throw new NotImplementedException("addCorruptedSliceObserver is not implemented");
      }
   }
}
