//
// Cleversafe open-source code header - Version 1.1 - December 1, 2006
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2007 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
// Chicago IL 60616
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: wleggette
//
// Date: Dec 7, 2007
//---------------------

package org.cleversafe.layer.access.managers;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.File;
import java.net.URI;
import java.security.KeyPair;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.Credentials;
import org.cleversafe.authentication.CredentialsManager;
import org.cleversafe.authentication.PropertiesFileCredentialsManager;
import org.cleversafe.authentication.PropertiesFileCredentialsManagerTest;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.layer.access.GridAccessManager;
import org.cleversafe.layer.access.GridAccessManagerTest;
import org.cleversafe.layer.access.managers.state.ServiceInterfaceState;
import org.cleversafe.layer.access.managers.state.ServiceState;
import org.cleversafe.util.FileSystemUtils;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.managers.LocalFileVaultManagerTest;
import org.cleversafe.vault.managers.state.VaultState;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


// TODO: Describe class or interface
public class LocalFileGridAccessManagerTest extends GridAccessManagerTest
{
   private static Logger _logger = Logger.getLogger(LocalFileGridAccessManagerTest.class);

   
   
   @Override
   public GridAccessManager getGridAccessManagerInstance()
   {
      return new LocalFileGridAccessManager();
   }
   
   

   @Override
   public Tuple2<UUID, CredentialsManager> getCredentials(URI directory) throws Exception
   {
      this.createTestAccount(directory);
      return new Tuple2<UUID, CredentialsManager>(
            LocalFileGridAccessManager._getCredentials(directory).getAccountIdentifier(),
            LocalFileGridAccessManager._getCredentialsManager(directory));
   }



   @BeforeClass
   public static void setUpBeforeClass() throws Exception
   {
   }

   @AfterClass
   public static void tearDownAfterClass() throws Exception
   {
   }

   @Before
   public void setUp() throws Exception
   {
   }

   @After
   public void tearDown() throws Exception
   {
   }
   

   @Test
   public void testCheckInitialization() throws Exception
   {
      LocalFileGridAccessManager manager = new LocalFileGridAccessManager();
      try
      {
         manager.checkInitialization();
         fail("Manager initialization check passed; expected failure");
      }
      catch (InitializationException e) {}
      
      manager.load(this.getOutputDirectory("testCheckInitialization").toURI());
      
      try
      {
         manager.checkInitialization();
      }
      catch (InitializationException e)
      {
         fail("Manager initialization check failed; expected passing");
      }
   }
   
   @SuppressWarnings("unchecked")
   @Test
   public void testCrossSchemaCompatibility() throws Exception
   {
      File output = this.getOutputDirectory("testCrossSchemaCompatibility");
      FileSystemUtils.deleteDir(output);
      
      URI directory = output.toURI();
      
      _logger.debug("Performing cross-schema compatibility test at: " + directory.toString());
      
      /*
       * Create vault objects using LOCAL FILE VAULT MANAGER schema
       * 
       * VAULT_IDENTIFIER                          VAULT_NAME
       * ----------------------------------------  ---------------------------------
       * 500bea77-8eb3-4430-adb7-9b6eb1ef0050      "Vault Manager vault 1"
       * 165373cc-3176-4c3b-bfaa-ffc5ac5e72c8      "Vault Manager vault 2"
       * 22393589-f644-4aba-b81b-6fe7469b11af      "Vault Manager vault 3"
       * 
       */
      
      final UUID VAULT_UUID_1 = UUID.fromString("500bea77-8eb3-4430-adb7-9b6eb1ef0050");
      final UUID VAULT_UUID_2 = UUID.fromString("165373cc-3176-4c3b-bfaa-ffc5ac5e72c8");
      final UUID VAULT_UUID_3 = UUID.fromString("22393589-f644-4aba-b81b-6fe7469b11af");
      final String VAULT_NAME_1 = "Vault Manager vault 1";
      final String VAULT_NAME_2 = "Vault Manager vault 2";
      final String VAULT_NAME_3 = "Vault Manager vault 3";
      
      LocalFileVaultManagerTest.loadDirectory(directory);
      
      Session session = LocalFileVaultManagerTest.getSessionFactory(directory).openSession();
      Transaction tx = session.beginTransaction();
      
      try
      {
         // Add vaults to database
         VaultState vault = new VaultState(VAULT_UUID_1, VAULT_NAME_1);
         session.save(vault);
         vault = new VaultState(VAULT_UUID_2, VAULT_NAME_2);
         session.save(vault);
         vault = new VaultState(VAULT_UUID_3, VAULT_NAME_3);
         session.save(vault);
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      session = LocalFileVaultManagerTest.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Read back vaults from database
         List<VaultState> vaults = session.createQuery("from VaultState").list();
         List<UUID> ids = new ArrayList<UUID>();
         List<String> names = new ArrayList<String>();
         for ( VaultState vault : vaults )
         {
            ids.add(vault.getIdentifier());
            names.add(vault.getName());
         }
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_1));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_2));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_2));
         assertTrue("missing vault name", names.contains(VAULT_NAME_1));
         assertTrue("missing vault name", names.contains(VAULT_NAME_2));
         assertTrue("missing vault name", names.contains(VAULT_NAME_3));
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Test the following objects using LOCAL FILE GRID ACCESS MANAGER schema
       * 
       * VAULT_IDENTIFIER                          VAULT_NAME
       * ----------------------------------------  ---------------------------------
       * 500bea77-8eb3-4430-adb7-9b6eb1ef0050      "Vault Manager vault 1"
       * 165373cc-3176-4c3b-bfaa-ffc5ac5e72c8      "Vault Manager vault 2"
       * 22393589-f644-4aba-b81b-6fe7469b11af      "Vault Manager vault 3"
       * 
       * Add the following objects using LOCAL FILE GRID ACCESS MANAGER schema
       * 
       * VAULT_IDENTIFIER                          VAULT_NAME
       * ----------------------------------------  ---------------------------------
       * dfe4f6b7-dacc-4924-af8f-048edab28af9      "Grid Access Manager vault 4"
       * e7b617c2-66a0-4d09-bb36-b69084025947      "Grid Access Manager vault 5"
       * 
       */
      
      final UUID VAULT_UUID_4 = UUID.fromString("dfe4f6b7-dacc-4924-af8f-048edab28af9");
      final UUID VAULT_UUID_5 = UUID.fromString("e7b617c2-66a0-4d09-bb36-b69084025947");
      final String VAULT_NAME_4 = "Grid Access Manager vault 4";
      final String VAULT_NAME_5 = "Grid Access Manager vault 5";
      
      LocalFileGridAccessManager.loadDirectory(directory);
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Read back vaults from database
         List<VaultState> vaults = session.createQuery("from VaultState").list();
         List<UUID> ids = new ArrayList<UUID>();
         List<String> names = new ArrayList<String>();
         for ( VaultState vault : vaults )
         {
            ids.add(vault.getIdentifier());
            names.add(vault.getName());
         }
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_1));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_2));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_2));
         assertTrue("missing vault name", names.contains(VAULT_NAME_1));
         assertTrue("missing vault name", names.contains(VAULT_NAME_2));
         assertTrue("missing vault name", names.contains(VAULT_NAME_3));
         
         // Write new vaults to database
         VaultState vault = new VaultState(VAULT_UUID_4, VAULT_NAME_4);
         session.save(vault);
         vault = new VaultState(VAULT_UUID_5, VAULT_NAME_5);
         session.save(vault);
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Test the following objecst using LOCAL FILE VAULT MANAGER schema
       * 
       * VAULT_IDENTIFIER                          VAULT_NAME
       * ----------------------------------------  ---------------------------------
       * dfe4f6b7-dacc-4924-af8f-048edab28af9      "Grid Access Manager vault 4"
       * e7b617c2-66a0-4d09-bb36-b69084025947      "Grid Access Manager vault 5"
       */
      

      session = LocalFileVaultManagerTest.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Read back vaults from database
         List<VaultState> vaults = session.createQuery("from VaultState").list();
         List<UUID> ids = new ArrayList<UUID>();
         List<String> names = new ArrayList<String>();
         for ( VaultState vault : vaults )
         {
            ids.add(vault.getIdentifier());
            names.add(vault.getName());
         }
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_1));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_2));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_3));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_4));
         assertTrue("missing vault identifier", ids.contains(VAULT_UUID_5));
         assertTrue("missing vault name", names.contains(VAULT_NAME_1));
         assertTrue("missing vault name", names.contains(VAULT_NAME_2));
         assertTrue("missing vault name", names.contains(VAULT_NAME_3));
         assertTrue("missing vault name", names.contains(VAULT_NAME_4));
         assertTrue("missing vault name", names.contains(VAULT_NAME_5));
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Add the following objects using LOCAL FILE GRID ACCESS MANAGER schema
       * 
       * SERVICE_NAME         SERVICE_TYPE   SERVICE_VAULT
       * -------------------  -------------  -------------------------------------------------
       * test-vault-1         iscsi          500bea77-8eb3-4430-adb7-9b6eb1ef0050  (Vault 1)
       * test-vault-3         cifs           22393589-f644-4aba-b81b-6fe7469b11af  (Vault 3)
       * test-vault-4         iscsi          dfe4f6b7-dacc-4924-af8f-048edab28af9  (Vault 4)
       */
      
      final String SERVICE_NAME_1 = "test-vault-1";
      final String SERVICE_NAME_3 = "test-vault-3";
      final String SERVICE_NAME_4 = "test-vault-4";
      final String SERVICE_TYPE_1 = "iscsi";
      final String SERVICE_TYPE_3 = "cifs";
      final String SERVICE_TYPE_4 = "iscsi";
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Add services to database
         VaultState vault = (VaultState)session
                 .createQuery("from VaultState v where v.identifier is '" + VAULT_UUID_1 + "'")
                 .uniqueResult();
         assertTrue("expected vault not found in database", vault != null);
         
         ServiceState service = new ServiceState();
         service.setName(SERVICE_NAME_1);
         service.setType(SERVICE_TYPE_1);
         service.setVault(vault);
         session.save(service);
         
         vault = (VaultState)session
                  .createQuery("from VaultState v where v.identifier is '" + VAULT_UUID_3 + "'")
                  .uniqueResult();
         assertTrue("expected vault not found in database", vault != null);
         
         service = new ServiceState();
         service.setName(SERVICE_NAME_3);
         service.setType(SERVICE_TYPE_3);
         service.setVault(vault);
         session.save(service);
         
         vault = (VaultState)session
               .createQuery("from VaultState v where v.identifier is '" + VAULT_UUID_4 + "'")
               .uniqueResult();
         assertTrue("expected vault not found in database", vault != null);

         service = new ServiceState();
         service.setName(SERVICE_NAME_4);
         service.setType(SERVICE_TYPE_4);
         service.setVault(vault);
         session.save(service);
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Read back services from database
         ServiceState service = (ServiceState)session
               .createQuery("from ServiceState s where s.name is '" + SERVICE_NAME_1 + "'")
               .uniqueResult();
         assertTrue("expected service not found in database", service != null);
         
         assertEquals("invalid field in service", SERVICE_NAME_1, service.getName());
         assertEquals("invalid field in service", SERVICE_TYPE_1, service.getType());
         assertEquals("invalid field in service", VAULT_UUID_1, service.getVault().getIdentifier());
         assertEquals("invalid field in service", VAULT_NAME_1, service.getVault().getName());
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Removing the following objects using LOCAL FILE GRID ACCESS MANAGER schema
       * 
       * VAULT_IDENTIFIER                          VAULT_NAME
       * ----------------------------------------  ---------------------------------
       * 22393589-f644-4aba-b81b-6fe7469b11af      "Vault Manager vault 3"
       * 
       * 
       * SERVICE_NAME         SERVICE_TYPE   SERVICE_VAULT
       * -------------------  -------------  -------------------------------------------------
       * test-vault-3         cifs           22393589-f644-4aba-b81b-6fe7469b11af  (Vault 3)
       */
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         ServiceState service = (ServiceState)session
               .createQuery("from ServiceState s where s.name is '" + SERVICE_NAME_3 + "'")
               .uniqueResult();
         assertTrue("expected service not found in database", service != null);
         
         VaultState vault = (VaultState)session
               .createQuery("from VaultState v where v.identifier is '" + VAULT_UUID_3 + "'")
               .uniqueResult();
         assertTrue("expected vault not found in database", vault != null);
         
         session.delete(service);
         session.delete(vault);
         
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Creating service interface and adding services to service interface.
       * 
       * SERVICE_INTERFACE_TYPE  SERVICE_INTERFACE_HOST  SERVICE_INTERFACE_PORT  AUTOSTART
       * ----------------------  ----------------------  ----------------------  ---------
       * iscsi                   127.0.0.1               3260                    YES
       * cifs                    0.0.0.0                 445                     NO
       */
      
      final String SERVICE_INTERFACE_TYPE_1 = "iscsi";
      final String SERVICE_INTERFACE_TYPE_2 = "cifs";
      final String SERVICE_INTERFACE_HOST_1 = "127.0.0.1";
      final String SERVICE_INTERFACE_HOST_2 = "0.0.0.0";
      final int SERVICE_INTERFACE_PORT_1 = 3260;
      final int SERVICE_INTERFACE_PORT_2 = 445;
      final boolean AUTOSTART_1 = true;
      final boolean AUTOSTART_2 = false;
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         ServiceInterfaceState serviceInterface = new ServiceInterfaceState();
         serviceInterface.setType(SERVICE_INTERFACE_TYPE_1);
         serviceInterface.setHost(SERVICE_INTERFACE_HOST_1);
         serviceInterface.setPort(SERVICE_INTERFACE_PORT_1);
         serviceInterface.setStartAutomatically(AUTOSTART_1);
         
         List<ServiceState> services = session
            .createQuery("from ServiceState s where s.type is '" + SERVICE_INTERFACE_TYPE_1 + "'")
            .list();
         for ( ServiceState service : services )
         {
            assertEquals("returned service has invalid type", 
                  SERVICE_INTERFACE_TYPE_1, service.getType());
            try
            {
               service.getServiceInterfaces().add(serviceInterface);
               fail("addition of service interface to succeeded unexpectedly");
            } catch (UnsupportedOperationException e) {}
            
            
            serviceInterface.getServices().add(service);
            assertEquals("service does not have proper interfaces", 
                  1, service.getServiceInterfaces().size());
            assertEquals("service interface not set correctly",
                  serviceInterface, service.getServiceInterfaces().iterator().next()); 
         }
         
         session.save(serviceInterface);
         
         serviceInterface = new ServiceInterfaceState();
         serviceInterface.setType(SERVICE_INTERFACE_TYPE_2);
         serviceInterface.setHost(SERVICE_INTERFACE_HOST_2);
         serviceInterface.setPort(SERVICE_INTERFACE_PORT_2);
         serviceInterface.setStartAutomatically(AUTOSTART_2);
         
         session.save(serviceInterface);
         
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      /*
       * Check that the following services are added to the "iscsi" service interface
       * using LOCAL FILE GRID ACCESS MANAGER schema. Check that "cifs" service interface
       * has no services. Remove "test-vault-1" from "iscsi" service interface.
       * 
       * SERVICE_NAME         SERVICE_TYPE   SERVICE_VAULT
       * -------------------  -------------  -------------------------------------------------
       * test-vault-1         iscsi          500bea77-8eb3-4430-adb7-9b6eb1ef0050  (Vault 1)
       * test-vault-4         iscsi          dfe4f6b7-dacc-4924-af8f-048edab28af9  (Vault 4)
       */
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Check "iscsi" service interface
         ServiceInterfaceState serviceInterface = (ServiceInterfaceState)session
            .createQuery("from ServiceInterfaceState s where s.type is '" + SERVICE_INTERFACE_TYPE_1 + "'")
            .uniqueResult();
         assertTrue("expected service interface not found in database", serviceInterface != null);
         
         Set<ServiceState> services = serviceInterface.getServices();
         Set<String> serviceNames = new HashSet<String>();
         for ( ServiceState service : services )
         {
            serviceNames.add(service.getName());
            assertEquals("service does not reference service interface",
                  1, service.getServiceInterfaces().size());
            assertEquals("service references improper service interface",
                  serviceInterface, service.getServiceInterfaces().iterator().next());
         }
         assertTrue("missing service", serviceNames.contains(SERVICE_NAME_1));
         assertTrue("missing service", serviceNames.contains(SERVICE_NAME_4));
         
         
         // Remove a service from service interface
         ServiceState service = (ServiceState)session
            .createQuery("from ServiceState s where s.name is '" + SERVICE_NAME_1 + "'")
            .uniqueResult();
         assertTrue("expected service not found in database", service != null);
         
         serviceInterface.getServices().remove(service);
         
         // Check "cifs" service interface
         serviceInterface = (ServiceInterfaceState)session
            .createQuery("from ServiceInterfaceState s where s.type is '" + SERVICE_INTERFACE_TYPE_2 + "'")
            .uniqueResult();
         assertTrue("expected service interface not found in database", serviceInterface != null);
         
         assertEquals("service interface has unexpected services", 
               0, serviceInterface.getServices().size());
         
      }
      finally
      {
         tx.commit();
         session.close();
      }

      /*
       * Check that the following services are added to the "iscsi" service interface
       * using LOCAL FILE GRID ACCESS MANAGER schema.
       * 
       * SERVICE_NAME         SERVICE_TYPE   SERVICE_VAULT
       * -------------------  -------------  -------------------------------------------------
       * test-vault-4         iscsi          dfe4f6b7-dacc-4924-af8f-048edab28af9  (Vault 4)
       */
      
      session = LocalFileGridAccessManager.getSessionFactory(directory).openSession();
      tx = session.beginTransaction();
      
      try
      {
         // Check "iscsi" service interface
         ServiceInterfaceState serviceInterface = (ServiceInterfaceState)session
            .createQuery("from ServiceInterfaceState s where s.type is '" + SERVICE_INTERFACE_TYPE_1 + "'")
            .uniqueResult();
         assertTrue("expected service interface not found in database", serviceInterface != null);
         
         Set<ServiceState> services = serviceInterface.getServices();
         Set<String> serviceNames = new HashSet<String>();
         for ( ServiceState service : services )
         {
            serviceNames.add(service.getName());
         }
         assertTrue("missing service", serviceNames.contains(SERVICE_NAME_4));
         
      }
      finally
      {
         tx.commit();
         session.close();
      }
      
      
   }
   
   

   private void createTestAccount(URI directory) throws Exception
   {
      // Set default credentials path
      //String orig = PropertiesFileCredentialsManager.getDefaultCredentialsPath();
      PropertiesFileCredentialsManager.setDefaultCredentialsPath(
            directory.getPath() + File.separator + LocalFileGridAccessManager.ACCOUNTS_PATH);
      
      PropertiesFileCredentialsManagerTest.createFileFile(null, null,
            LocalFileGridAccessManager.ACCOUNT_NAME, LocalFileGridAccessManager.ACCOUNT_PASSWORD);
   }
   
   @Test
   public void test_getCredentialsManager() throws Exception
   {
      URI directory = this.getOutputDirectory("test_getCredentialsManager").toURI();
      this.createTestAccount(directory);
      
      this.getGridAccessManagerInstance().load(directory);
      
      CredentialsManager credman = LocalFileGridAccessManager._getCredentialsManager(directory);
      
      // Just try getting these things, the operations should not fail
      
      UUID accountIdentifier = 
         credman.getAccountIdentifier(LocalFileGridAccessManager.ACCOUNT_NAME);
      assertTrue("account identifier fetch failed", accountIdentifier != null);
      
      KeyPair keypair =
         credman.getAccountKeyPair(accountIdentifier);
      assertTrue("account key pair fetch failed", keypair != null);
      // TODO: Find some way to compare key pairs as is done below
      
      Credentials credentials =
         credman.getCredentials(accountIdentifier);
      assertTrue("account credentials fetch failed", credentials != null);
      assertEquals("returned credentials not equal", 
            credentials, LocalFileGridAccessManager._getCredentials(directory));
      
      assertEquals("credentials contain invalid account identifier",
            accountIdentifier,
            credentials.getAccountIdentifier());
   }


}


