#!/usr/bin/perl -w
#
# Log analyzer for dsNet Accesser log files
# Run on accesser.log.* files in chronological order:
#  > ./dsnet-accesser-status 2 accesser.log.2008-03-27 accesser.log.2008-03-28
#

use Time::Local;
use POSIX qw(floor);

$REG_IP = '\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}';
$REG_TIME = '\d{4}\-\d{2}\-\d{2}\ \d{1,2}\:\d{1,2}\:\d{1,2}\.\d{3}';
$REG_TIME_GET = '(\d{4})\-(\d{2})\-(\d{2})\ (\d{1,2})\:(\d{1,2})\:(\d{1,2})\.(\d{3})';
$REG_UP = "($REG_TIME).*MinaConnector\.sessionOpened.*endpoint\:($REG_IP)";
$REG_DOWN = "($REG_TIME).*MinaConnector\.sessionClosed.*endpoint\:($REG_IP)";

if($#ARGV < 1)
{
	print "Usage: dsnet-accesser-status threshold log1 [log2 ...]\n";
	print "Logs should be specified in chronological order, eg:\n";
	print " > ./dsnet-accesser-status 2 accesser.log.2008-03-27 accesser.log.2008-03-28\n";
	exit;
}

$threshold = shift @ARGV;
@logs = @ARGV;

%storesDown = ();
undef $outage;

while($log = shift @logs)
{
	print "Analyzing log: $log\n";
	open(LOG, $log) or die("Cannot open log: $log");

	while($line = <LOG>)
	{
		if($line =~ $REG_UP)
		{
			$time = $storesDown{$2};
			$now = date2time($1);
			if($time)
			{
				delete $storesDown{$2};
				printf("$1 store <$2> UP after [%s] - (%s)\n", 
					duration2str($now - $time), 
					status());

				my $down = keys %storesDown;
				if($down == $threshold)
				{
					printf("\n===> OUTAGE ENDS at $1 after [%s]\n\n",
						duration2str($now - $outage));
					undef $outage;
				}
			}
			else
			{
				# Store was not down
			}
		}
		elsif($line =~ $REG_DOWN)
		{
			$time = $storesDown{$2};
			if($time)
			{
				# Store was already down
				die("ERROR: store <$2> was already down since $time");
			}
			else
			{
				$now = date2time($1);
				my $down = keys %storesDown;
				if($down == $threshold)
				{
					printf("\n===> OUTAGE BEGINS at $1\n\n");
					$outage = $now;
				}

				$storesDown{$2} = $now;
				printf("$1 store <$2> DOWN - (%s)\n", 
					status());
			}
		}
	}

	close(LOG);
}

print "Done\n";


sub date2time
{
	my $date = shift;
	if($date =~ $REG_TIME_GET)
	{
		return timegm($6, $5, $4, $3, $2, $1) + ($7/1000.);
	}
	else
	{
		die("Date $date does not parse");
	}
}

sub duration2str
{
	my $dur = shift;

	my $sph = 60*60;
	my $h = floor($dur/$sph);
	my $spm = 60;
	my $m = floor(($dur - $h*$sph) / $spm);
	my $s = floor($dur - $h*$sph - $m*$spm);
	my $ms = $dur - $h*$sph - $m*$spm - $s;

	return sprintf("%02d:%02d:%02d%s", $h, $m, $s, substr(sprintf("%.3f", $ms), 1));
}

sub status
{
	$sz = keys %storesDown;
	$s = "$sz down: ";
	while(($store, $time) = each %storesDown)
	{
		$s .= "<$store>, ";
	}	
	return substr($s, 0, -2);
}
