#!/usr/bin/perl -w
#
# Log analyzer for dsNet Accesser log files
# Run on accesser.log.* files in chronological order:
#  > ./dsnet-accesser-perf 2 accesser.log.2008-03-27 accesser.log.2008-03-28
#

use Time::Local;
use POSIX qw(floor);

$REG_TIME = '\d{4}\-\d{2}\-\d{2}\ \d{1,2}\:\d{1,2}\:\d{1,2}\.\d{3}';
$REG_TIME_GET = '(\d{4})\-(\d{2})\-(\d{2})\ (\d{1,2})\:(\d{1,2})\:(\d{1,2})\.(\d{3})';
$REG_BLOCK_WRITE_BEGIN = "($REG_TIME).*Write\\<.*started";
$REG_BLOCK_WRITE_END = "($REG_TIME).*Write\\<.*successful.*?(\\d+)ms";
$REG_BLOCK_READ_BEGIN = "($REG_TIME).*Read\\<.*started";
$REG_BLOCK_READ_END = "($REG_TIME).*Read\\<.*successful.*?(\\d+)ms";
$REG_GRID_WRITE_BEGIN = "($REG_TIME).*SmartWriteController\\.write.*started";
$REG_GRID_WRITE_END = "($REG_TIME).*SmartWriteController\\.write.*successful.*?(\\d+)ms";
$REG_GRID_COMMIT_BEGIN = "($REG_TIME).*GridTransaction\\.commit.*started";
$REG_GRID_COMMIT_END = "($REG_TIME).*GridTransaction\\.commit.*successful.*?(\\d+)ms";
$REG_GRID_READ_BEGIN = "($REG_TIME).*SmartReadController\\.read.*started";
$REG_GRID_READ_END = "($REG_TIME).*SmartReadController\\.read.*successful.*>(\\d+)ms";
$REG_SS_WRITE_BEGIN = "($REG_TIME).*RemoteSliceStore\\.write.*started";
$REG_SS_WRITE_END = "($REG_TIME).*RemoteSliceStore\\.write.*successful.*?(\\d+)ms";
$REG_SS_COMMIT_BEGIN = "($REG_TIME).*RemoteSliceStore\\.commit.*started";
$REG_SS_COMMIT_END = "($REG_TIME).*RemoteSliceStore\\.commit.*successful.*?(\\d+)ms";
$REG_SS_READ_BEGIN = "($REG_TIME).*RemoteSliceStore\\.read.*started";
$REG_SS_READ_END = "($REG_TIME).*RemoteSliceStore\\.read.*successful.*?(\\d+)ms";
$REG_HANDLER_BEGIN = "($REG_TIME).*Handler\\.process.*started";
$REG_HANDLER_END = "($REG_TIME).*Handler\\.process.*successful \\- (\\d+)ms";

if($#ARGV < 0)
{
	print "Usage: dsnet-accesser-perf log1 [log2 ...]\n";
	print "Logs should be specified in chronological order, eg:\n";
	print " > ./dsnet-accesser-perf accesser.log.2008-03-27 accesser.log.2008-03-28\n";
	exit;
}

@logs = @ARGV;

%outstanding = (
	# Block layer
	"block_write" => 0,
	"block_read" => 0,
	
	# Grid layer
	"grid_write" => 0,
	"grid_commit" => 0,
	"grid_read" => 0,
		
	# SliceStore layer
	"ss_write" => 0,
	"ss_commit" => 0,
	"ss_read" => 0,
	
	# Handler layer
	"handler" => 0,
);

%duration = (
	# Block layer
	"block_write" => 0,
	"block_read" => 0,
	
	# Grid layer
	"grid_write" => 0,
	"grid_commit" => 0,
	"grid_read" => 0,
		
	# SliceStore layer
	"ss_write" => 0,
	"ss_commit" => 0,
	"ss_read" => 0,
	
	# Handler layer
	"handler" => 0,
);


$last_time = 0;
undef($begin_time);

while($log = shift @logs)
{
	#print "Analyzing log: $log\n";
	open(LOG, $log) or die("Cannot open log: $log");

	while($line = <LOG>)
	{
		undef $time;
		
		if ($line =~ $REG_BLOCK_WRITE_BEGIN)
		{
			$outstanding{"block_write"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_BLOCK_WRITE_END)
		{
			$outstanding{"block_write"} -= 1;
			$time = date2ms($1);
			$duration{"block_write"} = $2;
		}
		elsif ($line =~ $REG_BLOCK_READ_BEGIN)
		{
			$outstanding{"block_read"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_BLOCK_READ_END)
		{
			$outstanding{"block_read"} -= 1;
			$time = date2ms($1);
			$duration{"block_read"} = $2;
		}
		elsif ($line =~ $REG_GRID_WRITE_BEGIN)
		{
			$outstanding{"grid_write"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_GRID_WRITE_END)
		{
			$outstanding{"grid_write"} -= 1;
			$time = date2ms($1);
			$duration{"grid_write"} = $2;
		}
		elsif ($line =~ $REG_GRID_COMMIT_BEGIN)
		{
			$outstanding{"grid_commit"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_GRID_COMMIT_END)
		{
			$outstanding{"grid_commit"} -= 1;
			$time = date2ms($1);
			$duration{"grid_commit"} = $2;
		}
		elsif ($line =~ $REG_GRID_READ_BEGIN)
		{
			$outstanding{"grid_read"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_GRID_READ_END)
		{
			$outstanding{"grid_read"} -= 1;
			$time = date2ms($1);
			$duration{"grid_read"} = $2;
		}
		elsif ($line =~ $REG_SS_WRITE_BEGIN)
		{
			$outstanding{"ss_write"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_SS_WRITE_END)
		{
			$outstanding{"ss_write"} -= 1;
			$time = date2ms($1);
			$duration{"ss_write"} = $2;
		}
		elsif ($line =~ $REG_SS_COMMIT_BEGIN)
		{
			$outstanding{"ss_commit"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_SS_COMMIT_END)
		{
			$outstanding{"ss_commit"} -= 1;
			$time = date2ms($1);
			$duration{"ss_commit"} = $2;
		}
		elsif ($line =~ $REG_SS_READ_BEGIN)
		{
			$outstanding{"ss_read"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_SS_READ_END)
		{
			$outstanding{"ss_read"} -= 1;
			$time = date2ms($1);
			$duration{"ss_read"} = $2;
		}
		elsif ($line =~ $REG_HANDLER_BEGIN)
		{
			$outstanding{"handler"} += 1;
			$time = date2ms($1);
		}
		elsif ($line =~ $REG_HANDLER_END)
		{
			$outstanding{"handler"} -= 1;
			$time = date2ms($1);
			$duration{"handler"} = $2;
		}
		
		# Print row
		if(defined($time) && ($time > $last_time))
		{
			$last_time = $time;
			if(!defined($begin_time)) {
				$begin_time = $time;
			}
			
			printf("%s,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d\n",
				$time - $begin_time,
				$outstanding{"block_write"},
				$outstanding{"block_read"},
				$outstanding{"grid_write"},
				$outstanding{"grid_commit"},
				$outstanding{"grid_read"},
				$outstanding{"ss_write"},
				$outstanding{"ss_commit"},
				$outstanding{"ss_read"},
				$outstanding{"handler"},
				$duration{"block_write"},
				$duration{"block_read"},
				$duration{"grid_write"} + $duration{"grid_commit"},
				$duration{"grid_read"},
				$duration{"ss_write"} + $duration{"ss_commit"},
				$duration{"ss_read"},
				$duration{"handler"},
			);
		}
	}

	close(LOG);
}

# Parses a date and returns the number of ms since the epoch
sub date2ms
{
	my $date = shift;
	if($date =~ $REG_TIME_GET)
	{
		return 1e3 * timegm($6, $5, $4, $3, $2, $1) + $7;
	}
	else
	{
		die("Date $date does not parse");
	}
}

sub duration2str
{
	my $dur = shift;

	my $sph = 60*60;
	my $h = floor($dur/$sph);
	my $spm = 60;
	my $m = floor(($dur - $h*$sph) / $spm);
	my $s = floor($dur - $h*$sph - $m*$spm);
	my $ms = $dur - $h*$sph - $m*$spm - $s;

	return sprintf("%02d:%02d:%02d%s", $h, $m, $s, substr(sprintf("%.3f", $ms), 1));
}

