//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: Zachary Mark
//
// Date: Nov 29, 2007
//---------------------

package org.cleversafe.authentication;

import java.util.List;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.Configuration;
import javax.security.auth.login.LoginContext;
import javax.security.auth.login.LoginException;

import org.apache.log4j.Logger;
import org.cleversafe.authentication.credentials.PasswordCredentials;
import org.cleversafe.server.exceptions.ServerConfigurationLoadException;

// TODO: Describe class or interface
public class JAASPasswordAuthenticationFacilitator implements AuthenticationFacilitator
{
   private static Logger _logger = Logger.getLogger(JAASPasswordAuthenticationFacilitator.class);

   private AuthenticationFacilitatorConfiguration authConfiguration = null;

   public JAASPasswordAuthenticationFacilitator()
   {
   }

   public JAASPasswordAuthenticationFacilitator(
         AuthenticationFacilitatorConfiguration authConfiguration)
   {
      this.setConfiguration(authConfiguration);
   }

   /**
    * Set the authentication configuration as required by JAAS.
    * 
    * @param authConfiguration
    */
   public void setConfiguration(AuthenticationFacilitatorConfiguration authConfiguration)
   {
      this.authConfiguration = authConfiguration;
   }

   /**
    * Returns the authentication configuration used by JAAS.
    * 
    * @return The authentication configuration.
    */
   public AuthenticationFacilitatorConfiguration getConfiguration()
   {
      return this.authConfiguration;
   }

   /**
    * A JAAS CallbackHandler for handling username and password input.
    */
   private static class PasswordCallbackHandler implements CallbackHandler
   {
      private String username;
      private String password;

      /**
       * Constructs a callback handler to use the given username and password.
       * 
       * @param username
       *           The username.
       * @param password
       *           The password.
       */
      public PasswordCallbackHandler(String username, String password)
      {
         this.username = username;
         this.password = password;
      }

      /**
       * Handles any callbacks that might be made. It can process NameCallback and PasswordCallback,
       * otherwise throws an exception.
       * 
       * @throws UnsupportedCallbackException
       *            If the handler does not support a given callback type.
       */
      public void handle(Callback[] callbacks) throws UnsupportedCallbackException
      {
         for (int i = 0; i < callbacks.length; i++)
         {
            // Username handler
            if (callbacks[i] instanceof NameCallback)
            {
               _logger.debug("Setting username");
               NameCallback cb = (NameCallback) callbacks[i];
               cb.setName(this.username);
            }
            // Password handler
            else if (callbacks[i] instanceof PasswordCallback)
            {
               _logger.debug("Setting password");
               PasswordCallback cb = (PasswordCallback) callbacks[i];
               cb.setPassword(this.password.toCharArray());
            }
            // Unhandled callback
            else
            {
               _logger.warn("Unsupported callback type: " + callbacks[i]);
               throw new UnsupportedCallbackException(callbacks[i]);
            }
         }
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.authentication.AuthenticationFacilitator#attemptAuthentication(org.cleversafe.authentication.Credentials)
    */
   public boolean attemptAuthentication(Credentials credentials)
   {
      if (!(this.authConfiguration instanceof JAASConfiguration))
      {
         return false;
      }
      return this.attemptAuthentication(credentials,
            ((JAASConfiguration) authConfiguration).getDefaultMethod());
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.authentication.AuthenticationFacilitator#attemptAuthentication(org.cleversafe.authentication.Credentials,
    *      java.lang.String)
    */
   public boolean attemptAuthentication(Credentials credentials, String method)
   {
      boolean authenticated = false;

      try
      {
         // Setup JAAS module
         _logger.debug("Initializing login context");
         PasswordCallbackHandler callbackHandler;

         if (!(credentials instanceof PasswordCredentials))
         {
            throw new IllegalArgumentException("Expected PasswordCredentials object, got "
                  + credentials.getClass().toString());
         }

         if (!(authConfiguration instanceof Configuration))
         {
            throw new IllegalArgumentException("Expected Configuration object, got "
                  + authConfiguration.getClass().toString());
         }

         String username = ((PasswordCredentials) credentials).getUsername();
         String password = ((PasswordCredentials) credentials).getPassword();

         callbackHandler = new PasswordCallbackHandler(username, password);

         // This cast is acceptable because it was sanity checked above.
         LoginContext loginContext =
               new LoginContext(method, null, callbackHandler, (Configuration) authConfiguration);

         // Attempt authentication
         _logger.debug("Attempting authentication");
         loginContext.login();

         authenticated = true;
         _logger.debug("Authentication succeeded for user " + username);
      }
      catch (IllegalArgumentException ex)
      {
         _logger.warn("Illegal argument:  " + ex.getMessage());
         authenticated = false;;
      }
      catch (LoginException ex)
      {
         _logger.warn("Unable to authenticate user: " + ex.getMessage());
         authenticated = false;
      }
      catch (SecurityException ex)
      {
         _logger.error("Unable to process login attempt: " + ex.getMessage());
         authenticated = false;
      }

      return authenticated;
   }
}
