//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: Jun 1, 2007
//---------------------

package org.cleversafe.codec.misc;

import org.cleversafe.codec.AbstractCodec;
import org.cleversafe.codec.Decoder;
import org.cleversafe.codec.Encoder;
import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.exceptions.InitializationException;

public class TranspositionCodec extends AbstractCodec
{
   private static String name = "transposition";
   private static String type = "misc";

   private int threshold = -1;

   public TranspositionCodec()
   {
   }

   public TranspositionCodec(int threshold)
   {
      this.threshold = threshold;
      initialize();
   }

   @RequiredForInitialization
   public void setThreshold(int threshold)
   {
      this.threshold = threshold;
   }

   public int getThreshold()
   {
      return this.threshold;
   }

   public void initialize()
   {
      if (this.threshold == -1)
      {
         throw new InitializationException(
               "Parameters blockSize and threshold must be initialized to valid non-negative values");
      }
   }

   public Decoder getDecoder()
   {
      // New instance must be created each time because encoder or decoder objects cannot be shared amongst threads 
      return new TranspositionDecoder(this.threshold);
   }

   public Encoder getEncoder()
   {
      // New instance must be created each time because encoder or decoder objects cannot be shared amongst threads 
      return new TranspositionEncoder(this.threshold);
   }

   public String getName()
   {
      return TranspositionCodec.name;
   }

   public String getType()
   {
      return TranspositionCodec.type;
   }

   static int calculateRoundedBlockSize(int threshold, int blockSize)
   {
      return (int) Math.ceil(((double) blockSize / (double) threshold)) * threshold;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.codec.Codec#getEncodedSizeEstimate(int)
    */
   public long getEncodedSize(long inputSize)
   {      
      return inputSize + (this.threshold - (inputSize % this.threshold));
   }

   public static void main(String[] args) throws Exception
   {
      String inputString = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz";
      int threshold = 10; //  3, 5, 13, 52, 0, 1, 3000, 17, 26
      TranspositionCodec codec = new TranspositionCodec(threshold);
      System.out.println("Estimated encoded size: " + codec.getEncodedSize(52));

      byte[] encoded = codec.getEncoder().finish(inputString.getBytes());

      System.out.println("Encoded Length = " + encoded.length);

      int blockNumber = 0;
      for (int i = 0; i < encoded.length / threshold; i++)
      {
         if ((i * threshold) % (threshold * threshold) == 0)
            System.out.println("Block number: " + blockNumber++);

         byte[] buf = new byte[threshold];
         System.arraycopy(encoded, i * threshold, buf, 0, threshold);
         System.out.println(new String(buf));
      }

      System.out.println();
      byte[] decoded = codec.getDecoder().finish(encoded);
      System.out.println("Decoded Length = " + decoded.length);
      System.out.println(new String(decoded));

   }
}
