//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Oct 10, 2007
//---------------------

package org.cleversafe.codec.misc;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import org.cleversafe.codec.Encoder;
import org.cleversafe.codec.exceptions.CodecEncodeException;
import org.cleversafe.layer.grid.SourceName;

// TODO: Describe class or interface
class TranspositionEncoder implements Encoder
{
   private final int threshold;
   private final int blockSize;
   private byte[] data;
   private int position;
   
   private ByteArrayOutputStream buffer = new ByteArrayOutputStream();

   public TranspositionEncoder(int threshold)
   {
      this.position = 0;
      this.threshold = threshold;
      this.blockSize = (threshold * threshold);
      this.data = new byte[TranspositionCodec.calculateRoundedBlockSize(threshold, blockSize)];
   }

   private int transposition_index(final int arrayLength)
   {
      final int product = this.position * this.threshold;
      return ((product % arrayLength) + (product / arrayLength));
   }

   public byte[] finish() throws CodecEncodeException
   {
      // When we finish we flush out any remaining data in the buffer and 
      // add at a max (threshold) number of bytes of padding
      int paddingLength = (this.threshold) - (this.buffer.size() % this.threshold);

      // We follow the PKCS5 padding convention and append the byte value which
      // represents how many bytes to strip off.
      // FIXME: Add a check when constructing to allow a maximum threshold of 255
      byte paddingByte = (byte) paddingLength;
      for (int cnt = 0; cnt < paddingLength; cnt++)
      {
         this.buffer.write(paddingByte);
      }
      
      assert (this.buffer.size() % this.threshold == 0) : "Buffer size is not aligned to a multiple of threshold";
      
      // Perform transposition of last bit of final data and padding
      byte[] processData = this.buffer.toByteArray();
      byte[] dataPlusPadding = new byte[processData.length];
      
      int i = 0;
      while (i < dataPlusPadding.length)
      {
         dataPlusPadding[transposition_index(dataPlusPadding.length)] = processData[i++];
         this.position++;
      }


      this.buffer.reset();
      this.position = 0;
      
      return dataPlusPadding;
   }

   public byte[] finish(byte[] data) throws CodecEncodeException
   {
      try
      {
         ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
         byteOut.write(process(data));
         byteOut.write(finish());
         return byteOut.toByteArray();
      }
      catch (IOException e)
      {
         throw new CodecEncodeException("Unknown IO exception while finishing data", e);
      }
   }

   public void reset(SourceName sourceName, long transactionId)
   {
      this.position = 0;
      this.buffer.reset();
   }

   public boolean isInitialized()
   {
      return true;
   }

   public byte[] process(byte[] dataIn) throws CodecEncodeException
   {
      // This codec always keeps a remainder of data that is not a multiple of
      // the blockSize in a buffer memory for the next call to process.
      this.buffer.write(dataIn, 0, dataIn.length);

      
      
      // Only when we get a complete chunk do we do the write.
      if (this.buffer.size() >= this.blockSize)
      {
         byte[] processData = this.buffer.toByteArray();
         int remainderLength = buffer.size() % this.blockSize;
         
         this.buffer.reset();
         this.buffer.write(processData, processData.length - remainderLength, remainderLength);

         int processLength = (processData.length - remainderLength);
         
         
         // Create OutputStream for writing out transposed data
         ByteArrayOutputStream byteOut = new ByteArrayOutputStream(processLength);

         int i = 0;
         while (i < processLength)
         {
            while (i < processLength && this.position < this.blockSize)
            {
               this.data[transposition_index(this.blockSize)] = processData[i++];
               this.position++;
            }

            if (this.position == this.blockSize)
            {
               this.position = 0;
               try
               {
                  byteOut.write(this.data);
               }
               catch (IOException e)
               {
                  // This should never happen
                  throw new CodecEncodeException("Unkown IO exception while processing data", e);
               }
            }
         }

         assert (this.position == 0) : "Position was not zero, did not process an entire block";         
         
         try
         {
            byteOut.close();
         }
         catch (IOException ignore)
         {
         }

         return byteOut.toByteArray();
      }
      else
      {
         return new byte[0];
      }
   }
}
