//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: Manish Motwani <mmotwani@cleversafe.com>
//
// Date: May 14, 2007
//---------------------

package org.cleversafe.config;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.IllegalConfigurationFormatException;
import org.cleversafe.util.XMLParser;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * An XML implementation of {@link PropertiesProvider}
 * 
 * A custom properties class that reads properties from an XML DOM document.
 * 
 * @author Manish Motwani
 */
public class XMLPropertiesProvider implements PropertiesProvider
{
   private static final Logger _logger = Logger.getLogger(XMLPropertiesProvider.class);

   private static final long serialVersionUID = 6911780391299498797L;

   public static interface ValueTypes
   {
      public static final String STRING = "string";
      public static final String INTEGER = "int";
      public static final String DOUBLE = "double";
      public static final String BOOLEAN = "boolean";
   }

   public static interface AttributeNames
   {
      public static final String NAME = "name";
      public static final String TYPE = "type";
      public static final String VALUE = "value";
      public static final String CLASS = "class";
      public static final String REFERRAL = "referral";
      public static final String REQUIRES_INTERFACE = "requiresInterface";
   }

   public static interface DocumentSections
   {
      public static final String PROPERTIES = "properties";
   }

   /*
    * A map for each type of value will enable get methods (for eg. getIntValue()) to not do any
    * type conversion at run time. The conversion is done at parse time.
    */
   private final Map<List<String>, List<String>> propertiesStringMap =
         new HashMap<List<String>, List<String>>();
   private final Map<List<String>, List<Integer>> propertiesIntMap =
         new HashMap<List<String>, List<Integer>>();
   private final Map<List<String>, List<Double>> propertiesDoubleMap =
         new HashMap<List<String>, List<Double>>();
   private final Map<List<String>, List<Boolean>> propertiesBooleanMap =
         new HashMap<List<String>, List<Boolean>>();

   /**
    * Default Constructor.
    * 
    */
   public XMLPropertiesProvider()
   {
      super();
   }

   /**
    * Constructor.
    * 
    * @param xmlInputStream
    * @throws IllegalConfigurationFormatException
    */
   public XMLPropertiesProvider(final InputStream xmlInputStream)
         throws IllegalConfigurationFormatException, ConfigurationLoadException
   {
      Document document = null;
      try
      {
         document = XMLParser.parse(xmlInputStream, null);
      }
      catch (final Exception e)
      {
         throw new ConfigurationLoadException("Error loading properties configuration from XML", e);
      }

      readProperties(document);
   }

   /*
    * Returns true if the value for the given property name is set for any type; false otherwise.
    */
   private boolean isPropertySet(final List<String> propertyName)
   {
      if (this.propertiesStringMap.containsKey(propertyName)
            || this.propertiesIntMap.containsKey(propertyName)
            || this.propertiesDoubleMap.containsKey(propertyName)
            || this.propertiesBooleanMap.containsKey(propertyName))
      {
         return true;
      }
      else
      {
         return false;
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertyInterface#isSet(java.lang.String[])
    */
   public boolean isSet(final String[] qualifiedName)
   {
      assert qualifiedName != null;

      return isPropertySet(Arrays.asList(qualifiedName));
   }

   /*
    * Returns the typed list of values for the input property name.
    */
   private List<?> getValuesList(final List<String> propertyName)
         throws ConfigurationItemNotDefinedException
   {
      List<?> valuesList = null;

      // Check if the property exists in any of typed maps.
      if (this.propertiesStringMap.containsKey(propertyName))
      {
         valuesList = this.propertiesStringMap.get(propertyName);
      }
      else if (this.propertiesIntMap.containsKey(propertyName))
      {
         valuesList = this.propertiesIntMap.get(propertyName);
      }
      else if (this.propertiesDoubleMap.containsKey(propertyName))
      {
         valuesList = this.propertiesDoubleMap.get(propertyName);
      }
      else if (this.propertiesBooleanMap.containsKey(propertyName))
      {
         valuesList = this.propertiesBooleanMap.get(propertyName);
      }
      else
      {
         // If it doesn't exist in any of the maps, it means the property is not
         // set.
         assert !isPropertySet(propertyName);
         throw new ConfigurationItemNotDefinedException("Property not set: " + propertyName);
      }
      assert valuesList != null;

      return valuesList;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertyInterface#getValue(java.lang.String[])
    */
   public String getValue(final String[] qualifiedName) throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<?> valuesList = getValuesList(propertyName);

      // Throw an exception if more than 1 value exists.
      if (valuesList.size() > 1)
      {
         throw new ConfigurationItemNotDefinedException(
               "More than one value found for the property: " + propertyName
                     + " Try using getValuesList()");
      }

      // Return the first (and only) value.
      return valuesList.get(0).toString();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertyInterface#getValuesList(java.lang.String[])
    */
   public List<String> getValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<?> valuesList = getValuesList(propertyName);

      // Convert the typed values to String
      final List<String> retList = new ArrayList<String>();

      for (final Object value : valuesList)
      {
         retList.add(value.toString());
      }

      return retList;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getStringValue(java.lang.String[])
    */
   public String getStringValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<String> propertyValues = this.propertiesStringMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'String' not found: "
               + propertyName);
      }

      if (propertyValues.size() > 1)
      {
         throw new ConfigurationItemNotDefinedException(
               "More than one value found for the property: " + propertyName
                     + " Try using getStringValuesList()");
      }

      return propertyValues.get(0);
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getStringValuesList(java.lang.String[])
    */
   public List<String> getStringValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<String> propertyValues = this.propertiesStringMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'String' not found: "
               + propertyName);
      }

      return propertyValues;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultStringValue(java.lang.String[],
    *      java.lang.String)
    */
   public void setDefaultStringValue(final String[] qualifiedName, final String value)
   {
      assert qualifiedName != null;
      assert value != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default value
      if (!isPropertySet(propertyName))
      {
         final List<String> list = new ArrayList<String>();
         list.add(value);
         this.propertiesStringMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultStringValuesList(java.lang.String[],
    *      java.lang.String[])
    */
   public void setDefaultStringValuesList(final String[] qualifiedName, final String[] values)
   {
      assert qualifiedName != null;
      assert values != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default values
      if (!isPropertySet(propertyName))
      {
         final List<String> list = Arrays.asList(values);

         this.propertiesStringMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getNumStringValuesSet(java.lang.String[])
    */
   public int getNumStringValuesSet(final String[] qualifiedName)
   {
      assert qualifiedName != null;

      final List<String> list = this.propertiesStringMap.get(Arrays.asList(qualifiedName));
      return list == null ? 0 : list.size();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getIntValue(java.lang.String[])
    */
   public int getIntValue(final String[] qualifiedName) throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Integer> propertyValues = this.propertiesIntMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'int' not found: "
               + propertyName);
      }

      if (propertyValues.size() > 1)
      {
         throw new ConfigurationItemNotDefinedException(
               "More than one value found for the property: " + propertyName
                     + " Try using getIntValuesList()");
      }

      return propertyValues.get(0).intValue();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getIntValuesList(java.lang.String[])
    */
   public List<Integer> getIntValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Integer> propertyValues = this.propertiesIntMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'int' not found: "
               + propertyName);
      }

      return propertyValues;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultIntValue(java.lang.String[], int)
    */
   public void setDefaultIntValue(final String[] qualifiedName, final int value)
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default value
      if (!isPropertySet(propertyName))
      {
         final List<Integer> list = new ArrayList<Integer>();
         list.add(Integer.valueOf(value));
         this.propertiesIntMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultIntValuesList(java.lang.String[],
    *      int[])
    */
   public void setDefaultIntValuesList(final String[] qualifiedName, final int[] values)
   {
      assert qualifiedName != null;
      assert values != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default values
      if (!isPropertySet(propertyName))
      {
         final List<Integer> list = new ArrayList<Integer>();
         for (final int value : values)
         {
            list.add(Integer.valueOf(value));
         }

         this.propertiesIntMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getNumIntValuesSet(java.lang.String[])
    */
   public int getNumIntValuesSet(final String[] qualifiedName)
   {
      assert qualifiedName != null;

      final List<Integer> list = this.propertiesIntMap.get(Arrays.asList(qualifiedName));
      return list == null ? 0 : list.size();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getBooleanValue(java.lang.String[])
    */
   public boolean getBooleanValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Boolean> propertyValues = this.propertiesBooleanMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'boolean' not found: "
               + propertyName);
      }

      if (propertyValues.size() > 1)
      {
         throw new ConfigurationItemNotDefinedException(
               "More than one value found for the property: " + propertyName
                     + " Try using getBooleanValuesList()");
      }

      return propertyValues.get(0).booleanValue();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getBooleanValuesList(java.lang.String[])
    */
   public List<Boolean> getBooleanValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Boolean> propertyValues = this.propertiesBooleanMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'boolean' not found: "
               + propertyName);
      }

      return propertyValues;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultBooleanValue(java.lang.String[],
    *      boolean)
    */
   public void setDefaultBooleanValue(final String[] qualifiedName, final boolean value)
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default value
      if (!isPropertySet(propertyName))
      {
         final List<Boolean> list = new ArrayList<Boolean>();
         list.add(Boolean.valueOf(value));
         this.propertiesBooleanMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultBooleanValuesList(java.lang.String[],
    *      boolean[])
    */
   public void setDefaultBooleanValuesList(final String[] qualifiedName, final boolean[] values)
   {
      assert qualifiedName != null;
      assert values != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default values
      if (!isPropertySet(propertyName))
      {
         final List<Boolean> list = new ArrayList<Boolean>();
         for (final boolean value : values)
         {
            list.add(Boolean.valueOf(value));
         }

         this.propertiesBooleanMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getNumBooleanValuesSet(java.lang.String[])
    */
   public int getNumBooleanValuesSet(final String[] qualifiedName)
   {
      assert qualifiedName != null;

      final List<Boolean> list = this.propertiesBooleanMap.get(Arrays.asList(qualifiedName));
      return list == null ? 0 : list.size();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getDoubleValue(java.lang.String[])
    */
   public double getDoubleValue(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Double> propertyValues = this.propertiesDoubleMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'double' not found: "
               + propertyName);
      }

      if (propertyValues.size() > 1)
      {
         throw new ConfigurationItemNotDefinedException(
               "More than one value found for the property: " + propertyName
                     + " Try using getDoubleValuesList()");
      }

      return propertyValues.get(0).doubleValue();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getDoubleValuesList(java.lang.String[])
    */
   public List<Double> getDoubleValuesList(final String[] qualifiedName)
         throws ConfigurationItemNotDefinedException
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      final List<Double> propertyValues = this.propertiesDoubleMap.get(propertyName);

      if (propertyValues == null)
      {
         throw new ConfigurationItemNotDefinedException("Property of type 'double' not found: "
               + propertyName);
      }

      return propertyValues;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultDoubleValue(java.lang.String[],
    *      double)
    */
   public void setDefaultDoubleValue(final String[] qualifiedName, final double value)
   {
      assert qualifiedName != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default value
      if (!isPropertySet(propertyName))
      {
         final List<Double> list = new ArrayList<Double>();
         list.add(Double.valueOf(value));
         this.propertiesDoubleMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#setDefaultDoubleValuesList(java.lang.String[],
    *      double[])
    */
   public void setDefaultDoubleValuesList(final String[] qualifiedName, final double[] values)
   {
      assert qualifiedName != null;
      assert values != null;

      final List<String> propertyName = Arrays.asList(qualifiedName);

      // Only if the property doesn't exist, add the default values
      if (!isPropertySet(propertyName))
      {
         final List<Double> list = new ArrayList<Double>();
         for (final double value : values)
         {
            list.add(Double.valueOf(value));
         }

         this.propertiesDoubleMap.put(propertyName, Collections.unmodifiableList(list));
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.config.PropertiesProvider#getNumDoubleValuesSet(java.lang.String[])
    */
   public int getNumDoubleValuesSet(final String[] qualifiedName)
   {
      assert qualifiedName != null;

      final List<Double> list = this.propertiesDoubleMap.get(Arrays.asList(qualifiedName));
      return list == null ? 0 : list.size();
   }

   private List<?> createTypedList(final String type)
   {
      assert checkTypeValidity(type);

      List<?> list = null;
      if (type.equals(ValueTypes.STRING))
      {
         list = new ArrayList<String>();
      }
      else if (type.equals(ValueTypes.INTEGER))
      {
         list = new ArrayList<Integer>();
      }
      else if (type.equals(ValueTypes.DOUBLE))
      {
         list = new ArrayList<Double>();
      }
      else if (type.equals(ValueTypes.BOOLEAN))
      {
         list = new ArrayList<Boolean>();
      }

      assert list != null;
      return list;
   }

   /*
    * Adds a single, typed, property-value pair.
    */
   @SuppressWarnings("unchecked")
   private void addSingleValue(
         final List<String> propertyName,
         final String type,
         final String value) throws IllegalConfigurationFormatException
   {
      // Assure the type is a valid type.
      assert checkTypeValidity(type);

      Map mapToUse = null;
      Object valueObject = null;
      List list = null;

      // Initialize the map, value object and the list according to the type.
      if (type.equals(ValueTypes.STRING))
      {
         mapToUse = this.propertiesStringMap;
         valueObject = value;
         list = new ArrayList<String>();
      }
      else if (type.equals(ValueTypes.INTEGER))
      {
         mapToUse = this.propertiesIntMap;
         try
         {
            valueObject = Integer.valueOf(value);
         }
         catch (final NumberFormatException nfe)
         {
            throw new IllegalConfigurationFormatException(nfe);
         }
         list = new ArrayList<Integer>();
      }
      else if (type.equals(ValueTypes.DOUBLE))
      {
         mapToUse = this.propertiesDoubleMap;
         try
         {
            valueObject = Double.valueOf(value);
         }
         catch (final NumberFormatException nfe)
         {
            throw new IllegalConfigurationFormatException(nfe);
         }
         list = new ArrayList<Double>();
      }
      else if (type.equals(ValueTypes.BOOLEAN))
      {
         mapToUse = this.propertiesBooleanMap;
         try
         {
            valueObject = Boolean.valueOf(value);
         }
         catch (final NumberFormatException nfe)
         {
            throw new IllegalConfigurationFormatException(nfe);
         }
         list = new ArrayList<Boolean>();
      }

      // At this time the list cannot be null (since the type was valid).
      assert list != null;

      // Add the value to the list and add the unmodifiable list to the map.
      list.add(valueObject);
      mapToUse.put(propertyName, Collections.unmodifiableList(list));
   }

   /*
    * Returns a typed value from the value string. This method is used in conjunction with
    * addListValues to add list values of the same type.
    */
   private Object getTypedValue(final String type, final String value)
         throws IllegalConfigurationFormatException
   {
      assert checkTypeValidity(type);
      Object valueObject = null;

      try
      {
         if (type.equals(ValueTypes.STRING))
         {
            valueObject = value;
         }
         else if (type.equals(ValueTypes.INTEGER))
         {
            valueObject = Integer.valueOf(value);
         }
         else if (type.equals(ValueTypes.DOUBLE))
         {
            valueObject = Double.valueOf(value);
         }
         else if (type.equals(ValueTypes.BOOLEAN))
         {
            valueObject = Boolean.valueOf(value);
         }
      }
      catch (final NumberFormatException nfe)
      {
         throw new IllegalConfigurationFormatException(nfe);
      }

      return valueObject;
   }

   private boolean checkTypeValidity(final String type)
   {
      if (type.equals(ValueTypes.STRING) || type.equals(ValueTypes.INTEGER)
            || type.equals(ValueTypes.DOUBLE) || type.equals(ValueTypes.BOOLEAN))
      {
         return true;
      }
      else
      {
         return false;
      }
   }

   /*
    * Adds a list of values to the typed property-value map. This method is used after one or more
    * calls to getTypedValue() method.
    */
   @SuppressWarnings("unchecked")
   private void addListValues(final List<String> propertyName, final String type, final List values)
   {
      assert checkTypeValidity(type);
      assert !isPropertySet(propertyName);

      if (type.equals(ValueTypes.STRING))
      {
         this.propertiesStringMap.put(propertyName, values);
      }
      else if (type.equals(ValueTypes.INTEGER))
      {
         this.propertiesIntMap.put(propertyName, values);
      }
      else if (type.equals(ValueTypes.DOUBLE))
      {
         this.propertiesDoubleMap.put(propertyName, values);
      }
      else if (type.equals(ValueTypes.BOOLEAN))
      {
         this.propertiesBooleanMap.put(propertyName, values);
      }
   }

   /*
    * Recursive method to traverse the XML DOM document's property nodes.
    */
   @SuppressWarnings("unchecked")
   private void fillPropertiesMap(final List<String> parent, final Node node)
         throws IllegalConfigurationFormatException
   {
      assert node != null && parent != null;

      // Create the qualified property name from the parent and current node.
      final List<String> propertyName = new ArrayList<String>(parent);
      propertyName.add(node.getNodeName());

      // If this property is already set, throw an exception.
      if (isPropertySet(propertyName))
      {
         throw new IllegalConfigurationFormatException("Property of this name already exists: "
               + propertyName);
      }

      // Get child's parent (node)'s type
      final Node typeNode = node.getAttributes().getNamedItem(AttributeNames.TYPE);

      String type;
      if (typeNode == null) // If no type is defined, use string
      {
         type = ValueTypes.STRING;
      }
      else
      {
         type = typeNode.getNodeValue();
      }

      if (!checkTypeValidity(type))
      {
         throw new IllegalConfigurationFormatException("Invalid type defined (" + type
               + ") for property: " + propertyName);
      }

      // If the node has children
      if (node.hasChildNodes())
      {
         XMLPropertiesProvider._logger.debug("Processing property node's children: " + propertyName);
         final NodeList children = node.getChildNodes();

         boolean nonValueChildrenExist = false;

         List list = null;

         // For each child of the current property
         for (int i = 0; i < children.getLength(); i++)
         {
            final Node child = children.item(i);

            if (child.getNodeType() == Node.ELEMENT_NODE)
            {
               // If it's a <value> node
               if (child.getNodeName().equals(AttributeNames.VALUE))
               {
                  // If there was a value found for this property, it cannot
                  // have sub properties.
                  if (nonValueChildrenExist)
                  {
                     throw new IllegalConfigurationFormatException(
                           "Cannot have value(s) for a property that has sub elements:"
                                 + propertyName);
                  }

                  // Get the value from the value node
                  String value = null;
                  if (child.hasChildNodes())
                  {
                     if (child.getChildNodes().getLength() > 1)
                     {
                        throw new IllegalConfigurationFormatException(
                              "Value nodes must be leaf elements!");
                     }
                     else if (child.getFirstChild().getNodeType() != Node.TEXT_NODE)
                     {
                        throw new IllegalConfigurationFormatException(
                              "Value nodes must only contain text and not sub elements!");
                     }

                     value = child.getFirstChild().getNodeValue();
                  }
                  else
                  {
                     throw new IllegalConfigurationFormatException(
                           "Value nodes must have a value set!");
                  }

                  // Get the value as a typed value and add to the list.
                  assert value != null;
                  final Object valueObject = getTypedValue(type, value);
                  if (list == null)
                  {
                     list = createTypedList(type);
                  }
                  list.add(valueObject);
               }

               // If it's not a <value> node.
               else
               {
                  // Traverse the tree recursively
                  nonValueChildrenExist = true;
                  fillPropertiesMap(propertyName, children.item(i));
               }
            }
         }

         if (list != null)
         {
            addListValues(propertyName, type, list);
         }
      }

      // Leaf nodes that are not named "value"
      else
      {
         assert node.getNodeName() != AttributeNames.VALUE;

         final Node valueNode = node.getAttributes().getNamedItem(AttributeNames.VALUE);
         if (valueNode == null)
         {
            throw new IllegalConfigurationFormatException("Value not defined for leaf property: "
                  + propertyName);
         }
         final String value = valueNode.getNodeValue();

         addSingleValue(propertyName, type, value);
      }
   }

   /*
    * Reads the XML DOM document and populates the typed property-value maps. Calls the recursive
    * fillPropertiesMap() method for each top-level property node.
    */
   private void readProperties(final Document document) throws IllegalConfigurationFormatException
   {
      XMLPropertiesProvider._logger.debug("Parsing DOM document...");

      final Node configuration = document.getChildNodes().item(0);

      final NodeList children = configuration.getChildNodes();

      NodeList properties = null;

      // Find the "properties" section in the document
      XMLPropertiesProvider._logger.debug("Looking for '" + DocumentSections.PROPERTIES + "' tag");
      for (int i = 0; i < children.getLength(); i++)
      {
         if (children.item(i).getNodeName().equals(DocumentSections.PROPERTIES))
         {
            XMLPropertiesProvider._logger.debug("'" + DocumentSections.PROPERTIES + "' tag found!");
            properties = children.item(i).getChildNodes();
            break;
         }
      }

      // Throw exception if the section is not found
      if (properties == null)
      {
         XMLPropertiesProvider._logger.debug("'" + DocumentSections.PROPERTIES + "' tag not found!");

         throw new IllegalConfigurationFormatException("Invalid document - '"
               + DocumentSections.PROPERTIES + "' section not found!");
      }

      // For each property
      final List<String> propertiesName = new ArrayList<String>();
      for (int j = 0; j < properties.getLength(); j++)
      {
         if (properties.item(j).getNodeType() == Node.ELEMENT_NODE)
         {
            // Top level property node cannot be <value>
            if (properties.item(j).getNodeName() != AttributeNames.VALUE)
            {
               fillPropertiesMap(propertiesName, properties.item(j));
            }
            else
            {
               throw new IllegalConfigurationFormatException("Top level property name cannot be "
                     + AttributeNames.VALUE);
            }
         }
      }

      XMLPropertiesProvider._logger.debug("Parsing DOM document complete!");
   }
}
