//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: mmotwani
//
// Date: Jul 30, 2007
//---------------------

package org.cleversafe.config.evaluator;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.config.BindingsProvider;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.ExecutionContext;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.config.exceptions.ConfigurationLoadException;
import org.cleversafe.config.exceptions.IllegalConfigurationContentException;
import org.cleversafe.config.exceptions.ObjectInitializationException;
import org.cleversafe.config.exceptions.ObjectInstantiationException;
import org.cleversafe.config.exceptions.ObjectValidationException;
import org.cleversafe.util.Version;

// TODO: Describe class or interface
public class XMLEvaluator
{
   private static final Logger _logger = Logger.getLogger(XMLEvaluator.class);

   private static BindingsProvider bindingsProvider = null;

   static
   {
      // Load configuration
      try
      {
         bindingsProvider =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE);
      }
      catch (final ConfigurationLoadException cle)
      {
         throw new RuntimeException("Could not load system bindings configuration", cle);
      }
      assert XMLEvaluator.bindingsProvider != null;
   }

   /**
    * 
    * @param simpleArgument
    * @return
    * @throws IllegalConfigurationContentException
    */
   public static Evaluatable parseSimpleArgument(final SimpleArgument simpleArgument)
         throws IllegalConfigurationContentException
   {
      return parseLiteral(simpleArgument.getLiteral());
   }

   /**
    * 
    * @param simpleParameter
    * @return
    * @throws IllegalConfigurationContentException
    */
   public static NamedEvaluator parseSimpleParameter(final SimpleParameter simpleParameter)
         throws IllegalConfigurationContentException
   {
      final String name = simpleParameter.getName();

      // Since a simpleParameter is a simpleArgument, it can be passed into parseSimpleArgument
      return new NamedEvaluator(name, parseSimpleArgument(simpleParameter));
   }

   /**
    * 
    * @param argument
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static Evaluatable parseArgument(final Argument argument, final ExecutionContext ctx)
         throws IllegalConfigurationContentException, ConfigurationItemNotDefinedException,
         ObjectValidationException, ObjectInstantiationException, ObjectInitializationException
   {
      XMLEvaluator._logger.trace("Parsing 'argument' node");
      Evaluatable evaluatable = null;
      if (argument.isSetLiteral())
      {
         evaluatable = parseLiteral(argument.getLiteral());
      }
      else if (argument.isSetMethod())
      {
         evaluatable = parseMethod(argument.getMethod(), ctx);
      }
      else
      {
         // Enforced by 'choice' in schema
         assert argument.isSetReference();

         evaluatable = parseReference(argument.getReference(), ctx);
      }

      return evaluatable;
   }

   /**
    * 
    * @param literal
    * @return
    * @throws IllegalConfigurationContentException
    */
   public static LiteralEvaluator parseLiteral(final Literal literal)
         throws IllegalConfigurationContentException
   {
      XMLEvaluator._logger.trace("Parsing 'literal' node");
      LiteralEvaluator evaluator = null;
      try
      {
         evaluator =
               LiteralEvaluatorFactory.getEvaluator(literal.getType().toString(),
                     literal.getValue());
      }
      catch (final ObjectInstantiationException oie)
      {
         throw new IllegalConfigurationContentException(
               "Invalid literal node (type='" + literal.getType().toString() + "', value='"
                     + literal.getValue().toString() + "')", oie);
      }
      return evaluator;
   }

   /**
    * 
    * @param params
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static List<NamedEvaluator> parseParameterArray(
         final Parameter[] params,
         final ExecutionContext ctx) throws IllegalConfigurationContentException,
         ConfigurationItemNotDefinedException, ObjectValidationException,
         ObjectInstantiationException, ObjectInitializationException
   {
      List<NamedEvaluator> paramEvaluators = new ArrayList<NamedEvaluator>(params.length);
      for (Parameter param : params)
      {
         paramEvaluators.add(XMLEvaluator.parseParameter(param, ctx));
      }
      return paramEvaluators;
   }

   /**
    * 
    * @param parameter
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static NamedEvaluator parseParameter(final Parameter parameter, final ExecutionContext ctx)
         throws IllegalConfigurationContentException, ConfigurationItemNotDefinedException,
         ObjectValidationException, ObjectInstantiationException, ObjectInitializationException
   {
      XMLEvaluator._logger.trace("Parsing 'parameter' node");
      final String name = parameter.getName();

      // Since a parameter is an argument, it can be passed into parseArgument
      return new NamedEvaluator(name, parseArgument(parameter, ctx));
   }

   /**
    * 
    * @param referenceArray
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static List<ReferenceEvaluator> parseReferenceArray(
         final Reference[] referenceArray,
         ExecutionContext ctx) throws IllegalConfigurationContentException,
         ConfigurationItemNotDefinedException, ObjectValidationException,
         ObjectInstantiationException, ObjectInitializationException
   {
      assert referenceArray != null;

      List<ReferenceEvaluator> referenceEvaluatorList =
            new ArrayList<ReferenceEvaluator>(referenceArray.length);
      for (Reference reference : referenceArray)
      {
         referenceEvaluatorList.add(XMLEvaluator.parseReference(reference, ctx));
      }
      return referenceEvaluatorList;
   }

   /**
    * 
    * @param reference
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static ReferenceEvaluator parseReference(
         final Reference reference,
         final ExecutionContext ctx) throws IllegalConfigurationContentException,
         ConfigurationItemNotDefinedException, ObjectValidationException,
         ObjectInstantiationException, ObjectInitializationException
   {
      XMLEvaluator._logger.trace("Parsing 'reference' node");

      XMLValidator.validateReference(reference, ctx);

      Version version = null;

      // Get parameters
      final List<NamedEvaluator> paramList = parseParameterArray(reference.getParamArray(), ctx);

      if (reference.isSetVersion())
      {
         version = Version.fromString(reference.getVersion());
      }

      if (reference.isSetInterface())
      {
         return createReferenceEvaluatorFromInterface(reference.getInterface(),
               reference.isSetReferral(), reference.getReferral(), version, paramList);
      }
      else if (reference.isSetKind())
      {
         return createReferenceEvaluatorFromKind(reference.getKind().split("\\."),
               reference.isSetReferral(), reference.getReferral(), version, paramList);
      }
      else
      {
         // Assert reference type is object (otherwise XMLValidator would have failed)
         assert reference.isSetObject();
         if (ctx == null)
         {
            throw new RuntimeException("Null ExecutionContext passed for getting object");
         }

         final String objectName = reference.getObject();
         final Object referredObject = ctx.get(objectName);

         if (referredObject == null)
         {
            throw new IllegalConfigurationContentException("Object " + objectName
                  + " not found in execution context!");
         }
         return new ReferenceEvaluator(referredObject, paramList);
      }
   }

   /**
    * 
    * @param method
    * @param ctx
    * @return
    * @throws IllegalConfigurationContentException
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectValidationException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static InstanceMethodEvaluator parseMethod(final Method method, final ExecutionContext ctx)
         throws IllegalConfigurationContentException, ConfigurationItemNotDefinedException,
         ObjectValidationException, ObjectInstantiationException, ObjectInitializationException
   {
      XMLEvaluator._logger.trace("Parsing 'method' node");

      if (ctx == null)
      {
         throw new RuntimeException("Null ExecutionContext passed for invoking method on object");
      }

      // Get arguments
      final List<Evaluatable> argList = new ArrayList<Evaluatable>();
      for (final Argument arg : method.getArgArray())
      {
         argList.add(parseArgument(arg, ctx));
      }

      final String objectName = method.getObject();
      final Object referredObject = ctx.get(objectName);

      if (referredObject == null)
      {
         throw new IllegalConfigurationContentException("Object " + objectName
               + " not found in execution context!");
      }

      return new InstanceMethodEvaluator(referredObject, method.getName(), argList);
   }

   /**
    * Creates a ReferenceEvaluator from a kind and referral.
    * 
    * @param qualifiedKindName
    * @param isReferralDefined
    * @param referral
    * @param version
    * @param params
    * @return
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    */
   public static ReferenceEvaluator createReferenceEvaluatorFromKind(
         final String[] qualifiedKindName,
         final boolean isReferralDefined,
         final String referral,
         final Version version,
         final List<NamedEvaluator> params) throws ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException
   {
      assert params != null;

      Object kindObject = null;
      if (isReferralDefined)
      {
         if (version == null)
            kindObject = bindingsProvider.getImplementation(qualifiedKindName, referral);
         else
            kindObject = bindingsProvider.getImplementation(qualifiedKindName, referral, version);
      }
      else
         kindObject = bindingsProvider.getDefaultImplementation(qualifiedKindName);

      if (kindObject == null)
      {
         throw new ObjectInstantiationException("Kind: " + qualifiedKindName + " and referral: "
               + referral + " could not be instantiated");
      }

      return new ReferenceEvaluator(kindObject, params);
   }

   /**
    * Creates a ReferenceEvaluator from a interface class name.
    * 
    * @param interfaceClassName
    * @param isReferralDefined
    * @param referral
    * @param version
    * @param params
    * @return
    * @throws ConfigurationItemNotDefinedException
    * @throws ObjectInstantiationException
    * @throws ObjectInitializationException
    * @throws IllegalConfigurationContentException
    */
   public static ReferenceEvaluator createReferenceEvaluatorFromInterface(
         final String interfaceClassName,
         final boolean isReferralDefined,
         final String referral,
         final Version version,
         final List<NamedEvaluator> params) throws ConfigurationItemNotDefinedException,
         ObjectInstantiationException, ObjectInitializationException,
         IllegalConfigurationContentException
   {
      assert params != null;

      Object interfaceObject = null;

      try
      {
         if (isReferralDefined)
         {
            if (version == null)
            {
               interfaceObject =
                     bindingsProvider.getImplementation(Class.forName(interfaceClassName), referral);
            }
            else
            {
               interfaceObject =
                     bindingsProvider.getImplementation(Class.forName(interfaceClassName),
                           referral, version);
            }
         }
         else
         {
            interfaceObject =
                  bindingsProvider.getDefaultImplementation(Class.forName(interfaceClassName));
         }
      }
      catch (final ClassNotFoundException cnfe)
      {
         throw new IllegalConfigurationContentException("Interface class not found: "
               + interfaceClassName, cnfe);
      }

      if (interfaceObject == null)
      {
         throw new ObjectInstantiationException("Object for interface: " + interfaceClassName
               + " could not be instantiated");
      }

      return new ReferenceEvaluator(interfaceObject, params);
   }

}
