//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: ivolvovski
//
// Date: Dec 9, 2007
//---------------------

package org.cleversafe.control;

import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.control.FixedPunishmentSelector.ParameterRuntime;

public class FixedErrorRateSelector implements ParameterSelector
{
   private static Logger _logger = Logger.getLogger(FixedErrorRateSelector.class);

   private volatile boolean initialized = false;

   FixedPunishmentSelector selector = null;

   private Statistics statistics = null;

   // Number of reported attempts
   private int attempts;
   private double initialPunisnmentValue = 1;
   private long lastCorrectionAttempt = 0;
   private int errorCountSinceLastAdjustment = 0;
   /**
    * Configuration values
    */
   /**
    * Number of attempt to perform an operation before correction attempt could be made
    * Need to collect some statistics before trying to change mechanism
    */
   private int minAttemptBeforeCorrection = 100;
   /**
    * If error rate exceeds a given value a punishment tactics should be changed -
    * less aggressive assignment mechanism should be used
    * Should be in the range (0.0, 1.0)
    * If 1.0 is used no correction will be ever made
    */
   private double maxTolerableErrorRate = 1.0;
   /**
    * If error rate fall below a given value a punishment tactics should be changed -
    * more  aggressive assignment mechanism should be used
    * Should be in the range (0.0, 1.0) and be less then maxTolerableErrorRate
    * If 0.0 is used no correction will be ever made
    */
   private double minSensitiveErrorRate = 0.0;

   /**
    * Changes punishment by this value, 
    * either increasing or decreasing it in a given number of times
    */
   private double errorPanishmentAdjustmentValue = 1.1;

   /**
    * Statistics. Need only for external analysis
    */
   public final class Statistics
   {
      private long punishInc = 0;
      private long punishDec = 0;
      private long attemptsCopy = 0;
      private long errorCounter = 0;
      private double cumulativePunishment = 0.0;
      private double punishMin = Double.MAX_VALUE;
      private double punishMax = 0.0;

      /**
       * @return the punishInc
       */
      public synchronized long getPunishInc()
      {
         return this.punishInc;
      }

      /**
       * @return the punishDec
       */
      public synchronized long getPunishDec()
      {
         return this.punishDec;
      }

      /**
       * @return the attempts
       */
      public synchronized long getAttempts()
      {
         return this.attemptsCopy;
      }

      /**
       * @return the punishMin
       */
      public synchronized double getPunishMin()
      {
         return this.punishMin;
      }

      /**
       * @return the punishMax
       */
      public synchronized double getPunishMax()
      {
         return this.punishMax;
      }

      /**
       * @return the average punishment overtime
       */
      public synchronized double getPunishAverage()
      {
         return this.attemptsCopy != 0 ? this.cumulativePunishment / this.attemptsCopy : 0;
      }

      /**
       * @return the errorCounter
       */
      public synchronized long getErrorCounter()
      {
         return errorCounter;
      }
   }

   public FixedErrorRateSelector()
   {
      this.selector = new FixedPunishmentSelector();
   }

   public FixedErrorRateSelector(
         ParameterState state,
         int minSingleAttempt,
         double intialPunishmentValue,
         double punishmentAdjustmentValue,
         double maxTolerableRate,
         double minSensitiveRate,
         int minAttemoptBeforeCorrection)
   {
      this.selector = new FixedPunishmentSelector();
      this.selector.setState(state);
      this.selector.setMinimalSingleAttemptCount(minSingleAttempt);
      this.selector.setErrorPanishment(intialPunishmentValue);

      this.setErrorPanishmentAdjustmentValue(punishmentAdjustmentValue);
      this.setMaxTolerableErrorRate(maxTolerableRate);
      this.setMinSensitiveErrorRate(minSensitiveRate);
      this.setMinAttemptBeforeCorrection(minAttemoptBeforeCorrection);
      initialize();
   }

   public synchronized void initialize()
   {
      assert this.selector != null;
      this.statistics = new Statistics();

      this.statistics.punishMin = this.statistics.punishMax = selector.getErrorPunishment();

      // check that all values make sense
      if (this.maxTolerableErrorRate < 0 || this.maxTolerableErrorRate > 1)
      {
         throw new IllegalArgumentException("max-tolerable-error-rate must be in [0,1] range");
      }
      if (this.minSensitiveErrorRate < 0 || this.minSensitiveErrorRate > 1.0)
      {
         throw new IllegalArgumentException("min-sensitive-error-rate must be in [0,1] range");
      }
      if (this.maxTolerableErrorRate <= this.minSensitiveErrorRate)
      {
         throw new IllegalArgumentException(
               "max-tolerable-error-rate must be higher then min-sensitive-error-rate");
      }
      if (this.minAttemptBeforeCorrection < 1)
      {
         throw new IllegalArgumentException("min-attempts before correction should be positive");
      }
      if (this.errorPanishmentAdjustmentValue < 1)
      {
         throw new IllegalArgumentException("error-punishment-adjustment should be higher then 1");
      }
      this.selector.setErrorPanishment(this.initialPunisnmentValue);
      this.selector.initialize();

      this.initialized = true;
   }

   private synchronized void checkNotInitialized()
   {
      if (initialized)
      {
         throw new IllegalStateException("Values can be set only before an object is initilaized");
      }
   }

   private synchronized void checkInitialized()
   {
      if (!initialized)
      {
         throw new IllegalStateException("Values can be set only before an object is initilaized");
      }
   }

   /**
    * @return the statistics
    */
   public synchronized Statistics getStatistics()
   {
      statistics.attemptsCopy = attempts; // We need it in both places
      return this.statistics;
   }

   public synchronized Map<ParameterState, ParameterRuntime> getCurrentResults()
   {
      return selector.getCurrentResults();
   }

   /** 
    * Set the least number of success should in order to move to another state
    * @param count
    */
   public synchronized void setMinimalSingleAttemptCount(int count)
   {
      this.selector.setMinimalSingleAttemptCount(count);
   }

   /**
    * Getter of minimal number of successes within a state
    * @return
    */
   public synchronized int getMinimalSingleAttemptCount()
   {
      return selector.getMinimalSingleAttemptCount();
   }

   /**
    * The initial value, can't be changed after the object is fully initialized
    * @param initialErrorPanishmentCoef the errorPanishmentCoef to set
    */
   public synchronized void setErrorPanishment(double initialErrorPanishmentCoef)
   {
      checkNotInitialized();
      selector.setErrorPanishment(initialErrorPanishmentCoef);
   }

   /**
    * Affects how aggressively a selector odes transition to a more advanced state
    * @return the errorPanishmentCoef
    */
   public synchronized double getErrorPanishment()
   {
      return this.selector.getErrorPunishment();
   }

   /**
    * @param minAttemptBeforeCorrection the minAttemptBeforeCorrection to set
    */
   public synchronized void setMinAttemptBeforeCorrection(int minAttemptBeforeCorrection)
   {
      this.minAttemptBeforeCorrection = minAttemptBeforeCorrection;
   }

   /**
    * @return the minAttemptBeforeCorrection
    */
   public synchronized int getMinAttemptBeforeCorrection()
   {
      return this.minAttemptBeforeCorrection;
   }

   /**
    * @param maxTolerableErrorRate the maxTolerableErrorRate to set
    */
   public synchronized void setMaxTolerableErrorRate(double maxTolerableErrorRate)
   {
      this.maxTolerableErrorRate = maxTolerableErrorRate;
   }

   /**
    * @return the maxTolerableErrorRate
    */
   public synchronized double getMaxTolerableErrorRate()
   {
      return this.maxTolerableErrorRate;
   }

   /**
    * @param minSensitiveErrorRate 
    */
   public synchronized void setMinSensitiveErrorRate(double minSensitiveErrorRate)
   {
      this.minSensitiveErrorRate = minSensitiveErrorRate;
   }

   /**
    * @return the minSensitiveErrorRate
    */
   public synchronized double getMinSensitiveErrorRate()
   {
      return this.minSensitiveErrorRate;
   }

   /**
    * @param errorPanishmentAdjustmentValue the errorPanishmentAdjustmentValue to set
    */
   public synchronized void setErrorPanishmentAdjustmentValue(double errorPanishmentAdjustmentValue)
   {
      this.errorPanishmentAdjustmentValue = errorPanishmentAdjustmentValue;
   }

   /**
    * @return the errorPanishmentAdjustmentValue
    */
   public synchronized double getErrorPanishmentAdjustmentValue()
   {
      return this.errorPanishmentAdjustmentValue;
   }

   /**
    * @param initialPunisnmentValue the initialPunisnmentValue to set
    */
   public synchronized void setInitialPunisnmentValue(double initialPunisnmentValue)
   {
      this.initialPunisnmentValue = initialPunisnmentValue;
   }

   /**
    * @return the initialPunisnmentValue
    */
   public synchronized double getInitialPunisnmentValue()
   {
      return this.initialPunisnmentValue;
   }

   /**
    * This method is needed by framework, can be only called before object was full initialized
    * @param state
    */
   public synchronized void setState(ParameterState state)
   {
      if (this.initialized == true)
      {
         throw new IllegalStateException("Can't change selectors state after initialization");
      }
      this.selector.setState(state);
   }

   public ParameterState getState()
   {
      return this.selector.getState();
   }

   /**
    * On success we record changes and move to more desirable state
    */
   public synchronized void onSuccess()
   {
      checkInitialized();
      this.selector.onSuccess();
      this.adjustParameters();
   }

   public synchronized void onFailure()
   {
      checkInitialized();
      this.errorCountSinceLastAdjustment++;
      this.statistics.errorCounter++;
      this.selector.onFailure();

   }

   private void adjustParameters()
   {
      this.attempts++;
      this.statistics.cumulativePunishment += selector.getErrorPunishment();

      long attemptsSinceLastCorrection = this.attempts - this.lastCorrectionAttempt;
      // Is it time to try correction attempt
      if (attemptsSinceLastCorrection > this.minAttemptBeforeCorrection)
      {
         double currentErrorRate =
               ((double) (this.errorCountSinceLastAdjustment)) / attemptsSinceLastCorrection;
         boolean correctionPerformed = false;
         double adjustedPunishValue = selector.getErrorPunishment();
         
         //System.out.println (adjustedPunishValue + ", " + this.attempts + ", " + this.statistics.cumulativePunishment + ", " + this.statistics.cumulativePunishment/this.attempts);
         
         if (currentErrorRate > this.maxTolerableErrorRate)
         {
            correctionPerformed = true;
            adjustedPunishValue *= this.errorPanishmentAdjustmentValue;
            statistics.punishInc++;
            if (adjustedPunishValue > this.statistics.punishMax)
            {
               this.statistics.punishMax = adjustedPunishValue;
            }
            _logger.debug("Panishment value is increased " + adjustedPunishValue + " after "
                  + attemptsSinceLastCorrection + " attempts");
         }
         else if (currentErrorRate < this.minSensitiveErrorRate)
         {
            correctionPerformed = true;
/*            
            if (adjustedPunishValue > 1.e-300) // Arbitrary small value to prevent it go to 0
            {
               adjustedPunishValue /= this.errorPanishmentAdjustmentValue;
            }
*/            
            adjustedPunishValue /= this.errorPanishmentAdjustmentValue;
            statistics.punishDec++;
            if (adjustedPunishValue < this.statistics.punishMin)
            {
               this.statistics.punishMin = adjustedPunishValue;
            }
            _logger.debug("Panishment value is decreased " + adjustedPunishValue + " after "
                  + attemptsSinceLastCorrection + " attempts");
         }
         if (correctionPerformed)
         {
            this.lastCorrectionAttempt = this.attempts;
            this.errorCountSinceLastAdjustment = 0;
            this.selector.setErrorPanishment(adjustedPunishValue);
         }
      }
   }

   public String toString()
   {
      return "A:" + this.attempts + " C:" + this.lastCorrectionAttempt + " {" + " T:"
            + this.maxTolerableErrorRate + " S:" + this.minSensitiveErrorRate
            + " A:" + this.minAttemptBeforeCorrection + "} " + this.selector;
   }
}
