//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: ivolvovski
//
// Date: May 23, 2007
//---------------------

package org.cleversafe.daemon;

import org.apache.log4j.Logger;
import org.tanukisoftware.wrapper.WrapperManager;
import org.tanukisoftware.wrapper.event.WrapperControlEvent;
import org.tanukisoftware.wrapper.event.WrapperEvent;
import org.tanukisoftware.wrapper.event.WrapperEventListener;
import org.tanukisoftware.wrapper.event.WrapperServiceControlEvent;

/*
 * 
 * This class is used to create demonizable UNIX or NT Service java application
 * Uses JavaWrapper http://wrapper.tanukisoftware.org/
 * A derived class needs to overwrite Shutdown
 * 
 * A class that implements main() method has to derive either from Daemonizer 
 * or a class derived from Daemonizer. 
 * shutdown() - will be called before daemon and service is shutdown
 * reconfigure() - provides a hook for reconfiguration
 * 
 * TODO: Implement stop/resume commands. Now stop performs shutdown, resume is not implemented
 * TODO: research how logoff should be implemented
 */
public abstract class Daemonizer implements WrapperEventListener
{
   public static final String DAEMONIZE_PROPERTY = "org.cleversafe.daemonize";
   
   private static Logger _logger = Logger.getLogger(Daemonizer.class);

   private Object latch = new Object();
   private boolean shutdown = false;
   private boolean inShutdown = false;

   public Daemonizer()
   {
      if ( Boolean.getBoolean(DAEMONIZE_PROPERTY) )
      {
         WrapperManager.addWrapperEventListener(this, WrapperEventListener.EVENT_FLAG_CONTROL);
         WrapperManager.addWrapperEventListener(this, WrapperEventListener.EVENT_FLAG_SERVICE);
      }
   }

   public final void fired(WrapperEvent event)
   {
      _logger.info("Event of type " + event.getClass().getName() + " flag=" + event.getFlags());

      synchronized (latch)
      {
         if (this.shutdown == true)
         {
            _logger.warn("Shutdown has alredy been performed. No event processing at this point");
            return;
         }

         if (event instanceof WrapperControlEvent)
         {
            WrapperControlEvent ctrlEvent = (WrapperControlEvent) event;
            if (    ctrlEvent.getControlEvent() == WrapperManager.WRAPPER_CTRL_C_EVENT
                 || ctrlEvent.getControlEvent() == WrapperManager.WRAPPER_CTRL_TERM_EVENT
                 || ctrlEvent.getControlEvent() == WrapperManager.WRAPPER_CTRL_SHUTDOWN_EVENT
                 || ctrlEvent.getControlEvent() == WrapperManager.WRAPPER_CTRL_CLOSE_EVENT )
            {
               shutdownMe();
            }
            else if (ctrlEvent.getControlEvent() == WrapperManager.WRAPPER_CTRL_HUP_EVENT)
            {
               reconfigure();
            }
            else
            {
               _logger.info("Dropping an event: " + ctrlEvent.getControlEvent());
            }
         }
         else if (event instanceof WrapperServiceControlEvent)
         {
            WrapperServiceControlEvent serviceEvent = (WrapperServiceControlEvent) event;
            if (    serviceEvent.getServiceControlCode() == WrapperManager.SERVICE_CONTROL_CODE_STOP
                 || serviceEvent.getServiceControlCode() == WrapperManager.SERVICE_CONTROL_CODE_SHUTDOWN)
            {
               shutdownMe();
            }
            else if (serviceEvent.getServiceControlCode() == WrapperManager.SERVICE_CONTROL_CODE_INTERROGATE)
            {
               //reconfigure();
               _logger.warn("Interrogate service is not implemented");
            }
            else
            {
               _logger.info("Dropping an event: " + serviceEvent.getServiceControlCode());
            }
         }
         else 
         {
            _logger.error("Unknown event of type:" + event.getClass().getName() + " ignoring....");
         }
      }
   }

   private void shutdownMe()
   {
      this.inShutdown = true;
      shutdown();
      this.shutdown = true;
   }

   public final boolean isShutdownInProgress()
   {
      return inShutdown;
   }
   /**
    * This method is invoked upon SIG HUP or Service interrogate. A derived
    * class has to provide its own means of reconfiguring
    */
   protected void reconfigure()
   {
      _logger.debug("Null-op reconfigure() is called");
   }

   /**
    * This method is invoked daemon or service shutdown.
    */
   protected abstract void shutdown();

}
