//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.ida;

import org.apache.log4j.Logger;
import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;

/**
 * Base abstract class that implements {@link InformationDispersalCodec} and
 * contains common code for all codecs.
 * 
 * @author Manish Motwani
 */
public abstract class InformationDispersalCodecBase
      implements InformationDispersalCodec
{
   public static int DEFAULT_CHUNK_SIZE = 4096;
   
   boolean isInitialized = false;
   
   // Number of slices
   protected int numSlices;

   // Number of acceptable lost slices
   protected int threshold;
   
   // Size of the chunk of data processed as input with each call to encode
   protected int chunkSize;

   // Encoder and decoder
   private InformationDispersalEncoder encoder;
   private InformationDispersalDecoder decoder;
   
   protected abstract InformationDispersalEncoder getNewEncoder() throws IDAInvalidParametersException;
   protected abstract InformationDispersalDecoder getNewDecoder() throws IDAInvalidParametersException;
   
   protected InformationDispersalCodecBase()
   {
      this.numSlices = -1;
      this.threshold = -1;
      this.chunkSize = DEFAULT_CHUNK_SIZE;
   }

   /**
    * This constructor exists only for convenience and should not be used
    * outside of tests.  The chunk size ought to be calculated based on the
    * disk block size and the expected size increase caused by the datasource
    * codecs. - JKR
    * 
    * @deprecated
    */ 
   protected InformationDispersalCodecBase(int numSlices, int threshold)
   {      
      this.numSlices = numSlices;
      this.threshold = threshold;
      this.chunkSize = DEFAULT_CHUNK_SIZE;
   }
   
   protected InformationDispersalCodecBase(int numSlices, int threshold, int chunkSize)
   {      
      this.numSlices = numSlices;
      this.threshold = threshold;
      this.chunkSize = chunkSize;
   }
   
   /**
    * Initializes IDA's encoder and decoder, cannot be changed thereafter
    */
   public synchronized void initialize() throws IDAInvalidParametersException
   {
      assert numSlices != -1 : "Number of slices has not been set";
      assert threshold != -1 : "Threshold has not been set";
      
      // Configuration already calls initialize, but we need to reinitialize with the
      // calculated block size after datasource codecs are applied.  This is a safety 
      // measure which should be added back in the future, but is being removed as a quick
      // fix -- JKR
      //if (this.isInitialized == true)
      //{
      //   throw new IDAInvalidParametersException("IDA already initialized, object may only be initialized once");
      //}
      
      this.decoder = this.getNewDecoder();
      this.encoder = this.getNewEncoder();
      
      this.isInitialized = true;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getDecoder()
    */
   public InformationDispersalDecoder getDecoder() throws IDANotInitializedException
   {
      if ((this.isInitialized == false) || (this.decoder == null))
      {
         throw new IDANotInitializedException("IDA not initialized, call initialize first");
      }
      return this.decoder;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getEncoder()
    */
   public InformationDispersalEncoder getEncoder() throws IDANotInitializedException
   {
      if ((this.isInitialized == false) || (this.encoder == null))
      {
         throw new IDANotInitializedException("IDA not initialized, call initialize first");
      }
      return this.encoder;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getNumSlices()
    */
   public int getNumSlices()
   {
      return numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getThreshold()
    */
   public int getThreshold()
   {
      return threshold;
   }
   
   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getThreshold()
    */
   public int getChunkSize()
   {
      return chunkSize;
   }

   /**
    * @param numSlices the numSlices to set
    */
   @RequiredForInitialization
   public void setNumSlices(int numSlices)
   {
      this.numSlices = numSlices;
   }

   /**
    * @param threshold the threshold to set
    */
   @RequiredForInitialization
   public void setThreshold(int threshold)
   {
      this.threshold = threshold;
   }
   
   /**
    * @param chunkSize the chunkSize to set
    */
   public void setChunkSize(int chunkSize)
   {
      this.chunkSize = chunkSize;
   }
}
