//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.ida.replicate;

import org.cleversafe.config.evaluator.XMLValidator.RequiredForInitialization;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.ida.InformationDispersalCodecBase;
import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;

/**
 * Replicate data {@link InformationDispersalCodec} implementation.
 * 
 * @author Manish Motwani
 */
public class ReplicateInformationDispersalCodec extends InformationDispersalCodecBase
      implements
         InformationDispersalCodec
{
   public static final String NAME = "replicate";

   public ReplicateInformationDispersalCodec()
   {
   }

   /**
    * Constructor.
    * 
    * @param numSlices
    *           Number of slices.
    * @param ignored
    *           Threshold value is ignored. Threshold is numSlices - 1.
    */
   public ReplicateInformationDispersalCodec(int numSlices, int ignored)
         throws IDAInvalidParametersException
   {
      super(numSlices, numSlices - 1);
      assert numSlices > 0;

      initialize();
   }

   /**
    * Constructor.
    * 
    * @param numSlices
    *           Number of slices.
    */
   public ReplicateInformationDispersalCodec(int numSlices) throws IDAInvalidParametersException
   {
      super(numSlices, numSlices - 1);
      assert numSlices > 0;

      initialize();
   }

   @Override
   public void initialize() throws IDAInvalidParametersException
   {
      if (this.numSlices < 1)
      {
         throw new IDAInvalidParametersException("Number of slices must be positive");
      }
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getName()
    */
   public String getName()
   {
      return NAME;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodecBase#setNumSlices(int)
    */
   @Override
   @RequiredForInitialization
   public void setNumSlices(int numSlices)
   {
      this.numSlices = numSlices;
      this.threshold = numSlices - 1;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodecBase#setThreshold(int)
    */
   @Override
   public void setThreshold(int ignored)
   {
   }

   @Override
   public void setChunkSize(int ignored)
   {
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalCodec#getBlowup()
    */
   public float getBlowup()
   {
      // All data is replicated in each slice
      return getNumSlices();
   }

   public long getDispersedSize(long inputSize)
   {
      return inputSize * this.numSlices;
   }

   @Override
   public InformationDispersalDecoder getDecoder()
   {
      ReplicateInformationDispersalDecoder decoder =
            new ReplicateInformationDispersalDecoder(this.numSlices);
      try
      {
         decoder.initialize();
      }
      catch (IDAInvalidParametersException e)
      {
         throw new RuntimeException("Could not initialize ReplicateInformationDispersalDecoder", e);
      }
      return decoder;
   }

   @Override
   public InformationDispersalEncoder getEncoder()
   {
      ReplicateInformationDispersalEncoder encoder =
            new ReplicateInformationDispersalEncoder(this.numSlices);
      try
      {
         encoder.initialize();
      }
      catch (IDAInvalidParametersException e)
      {
         throw new RuntimeException("Could not initialize ReplicateInformationDispersalEncoder", e);
      }
      return encoder;
   }

   @Override
   protected InformationDispersalDecoder getNewDecoder()
   {
      // This method should not be called directly
      return this.getDecoder();
   }

   @Override
   protected InformationDispersalEncoder getNewEncoder()
   {
      // This method should not be called directly
      return this.getEncoder();
   }
   
}
