//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.ida.replicate;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;

import org.cleversafe.ida.InformationDispersalDecoder;
import org.cleversafe.ida.exceptions.IDADecodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDAInvalidSliceCountException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;

/**
 * Replicate data {@link InformationDispersalDecoder} implementation.
 * 
 * @author Manish Motwani
 */
public class ReplicateInformationDispersalDecoder implements InformationDispersalDecoder
{
   private int numSlices;

   private ByteArrayOutputStream data = null;
   private boolean initialized = false;

   public ReplicateInformationDispersalDecoder()
   {
   }

   public ReplicateInformationDispersalDecoder(final int numSlices)
   {
      assert numSlices > 0;
      this.numSlices = numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#init()
    */
   public void initialize() throws IDAInvalidParametersException
   {
      if (this.numSlices < 1)
      {
         throw new IDAInvalidParametersException("Number of slices must be positive");
      }

      if (this.data != null)
      {
         this.data.reset();
      }
      this.initialized = true;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#update(java.util.List)
    */
   public byte[] update(final List<byte[]> encodedBuffers) throws IDADecodeException,
         IDANotInitializedException
   {
      assert encodedBuffers != null;
      assert encodedBuffers.size() > 1;

      if (!this.initialized)
      {
         throw new IDANotInitializedException("Decoder not initialized. Call init() first");
      }
      // Find buffer with data
      int nonNullIndex = 0;
      while (nonNullIndex < encodedBuffers.size())
      {
         if (encodedBuffers.get(nonNullIndex) != null)
         {
            break;
         }
         nonNullIndex++;
      }
      if (nonNullIndex == encodedBuffers.size())
      {
         throw new IDAInvalidSliceCountException("No data to decode");
      }
      if (this.data == null)
      {
         this.data = new ByteArrayOutputStream(encodedBuffers.get(nonNullIndex).length);
      }

      try
      {
         this.data.write(encodedBuffers.get(nonNullIndex));
         this.data.flush();
      }
      catch (IOException ioe)
      {
         throw new RuntimeException("I/O error", ioe);
      }

      return this.data.toByteArray();
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#doFinal(java.util.List)
    */
   public byte[] finish(final List<byte[]> encodedBuffers) throws IDADecodeException,
         IDANotInitializedException
   {
      final byte[] retVal = update(encodedBuffers);
      this.initialized = false;
      try
      {
         initialize();
      }
      catch (IDAInvalidParametersException e)
      {
         throw new IDANotInitializedException("Could not re-initialize IDA", e);
      }
      return retVal;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#getNumSlices()
    */
   public int getNumSlices()
   {
      return this.numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#getThreshold()
    */
   public int getThreshold()
   {
      return this.numSlices - 1;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#setNumSlices(int)
    */
   public void setNumSlices(int numSlices)
   {
      assert numSlices > 0;
      this.numSlices = numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalDecoder#setThreshold(int)
    */
   public void setThreshold(int ignored)
   {
   }

   public int getChunkSize()
   {
      // TODO Auto-generated method stub
      return 0;
   }

   public void setChunkSize(int ignored)
   {
      // TODO Auto-generated method stub
   }

}
