//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: mmotwani
//
// Date: May 30, 2007
//---------------------

package org.cleversafe.ida.replicate;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.cleversafe.ida.InformationDispersalEncoder;
import org.cleversafe.ida.exceptions.IDAEncodeException;
import org.cleversafe.ida.exceptions.IDAInvalidParametersException;
import org.cleversafe.ida.exceptions.IDANotInitializedException;

/**
 * Replicate data {@link InformationDispersalEncoder} implementation.
 * 
 * @author Manish Motwani
 */
public class ReplicateInformationDispersalEncoder implements InformationDispersalEncoder
{
   ByteArrayOutputStream[] dataBuffers = null;

   boolean initialized = false;

   private int numSlices;

   public ReplicateInformationDispersalEncoder()
   {
   }

   /**
    * Constructor.
    * 
    * @param numSlices
    *           Number of slices.
    */
   public ReplicateInformationDispersalEncoder(final int numSlices)
   {
      assert numSlices > 0;
      this.numSlices = numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#finish(byte[])
    */
   public List<byte[]> finish(final byte[] buffer) throws IDAEncodeException,
         IDANotInitializedException
   {
      final List<byte[]> byteArrayList = update(buffer);
      this.initialized = false;
      try
      {
         initialize();
      }
      catch (IDAInvalidParametersException e)
      {
         throw new IDANotInitializedException("Could not re-initialize IDA", e);
      }
      return byteArrayList;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#getNumSlices()
    */
   public int getNumSlices()
   {
      return this.numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#getThreshold()
    */
   public int getThreshold()
   {
      return this.numSlices - 1;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#init()
    */
   public void initialize() throws IDAInvalidParametersException
   {
      if (this.numSlices < 1)
      {
         throw new IDAInvalidParametersException("Number of slices must be positive");
      }

      if (this.dataBuffers == null)
      {
         this.dataBuffers = new ByteArrayOutputStream[this.numSlices];
      }
      else
      {
         for (ByteArrayOutputStream bos : this.dataBuffers)
         {
            if (bos != null)
            {
               bos.reset();
            }
         }
      }

      this.initialized = true;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#update(byte[])
    */
   public List<byte[]> update(final byte[] buffer) throws IDAEncodeException,
         IDANotInitializedException
   {
      if (!this.initialized || this.dataBuffers == null)
      {
         throw new IDANotInitializedException("Data buffers not initialized. Call init() first");
      }

      for (int i = 0; i < this.dataBuffers.length; i++)
      {
         // Create new buffer
         if (this.dataBuffers[i] == null)
         {
            this.dataBuffers[i] = new ByteArrayOutputStream(buffer.length);
         }

         try
         {
            this.dataBuffers[i].write(buffer);
            this.dataBuffers[i].flush();
         }
         catch (IOException ioe)
         {
            throw new RuntimeException("I/O error", ioe);
         }
      }

      return getByteArrays();
   }

   private List<byte[]> getByteArrays()
   {
      final List<byte[]> byteArrays = new ArrayList<byte[]>();
      for (final ByteArrayOutputStream dataBuffer : this.dataBuffers)
      {
         byteArrays.add(dataBuffer.toByteArray());
      }
      return byteArrays;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#setNumSlices(int)
    */
   public void setNumSlices(int numSlices)
   {
      assert numSlices > 0;
      this.numSlices = numSlices;
   }

   /*
    * (non-Javadoc)
    * 
    * @see org.cleversafe.ida.InformationDispersalEncoder#setThreshold(int)
    */
   public void setThreshold(int ignored)
   {
   }

   public int getChunkSize()
   {
      // TODO Auto-generated method stub
      return 0;
   }

   public void setChunkSize(int ignored)
   {
      // TODO Auto-generated method stub
   }
}
