//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// Author: John Quigley <jquigley@cleversafe.com>
//
// Date: Aug 8, 2007
//---------------------

package org.cleversafe.layer.communication;

import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.communication.exceptions.*;
import org.cleversafe.serialization.ProtocolMessageFactory;

/**
 * Connects and communicates to a remote end-point.
 * 
 * @author John Quigley <jquigley@cleversafe.com>
 * @version $Id: Connector.java 3893 2008-04-03 00:34:17Z gdhuse $
 */
public interface Connector
{
   /**
    * Attempts to open the connection (including retries if necessary). Do not call this method in a
    * loop as all looping is done internally.
    * 
    * @return true if the connection was newly opened.
    * @throws CommunicationIOException
    * @throws CommunicationConnectionException
    */
   boolean ensureConnected() throws CommunicationIOException, CommunicationConnectionException;

   /**
    * Attempt to open the connection. Precondition is that the connection is not already open. In
    * general call ensureConnect as that will use the connection manager to try and open the
    * connection if necessary.
    * 
    * @throws CommunicationIOException
    * @throws CommunicationConnectionException
    */
   void connect() throws CommunicationIOException, CommunicationConnectionException;

   /**
    * Close connection, if open. No precondition because connection could be lost
    * 
    * @throws CommunicationInterruptedException
    * @throws CommunicationIOException
    */
   void disconnect() throws CommunicationIOException, CommunicationInterruptedException;

   /**
    * Checks whether connection is properly established.
    * 
    * @return A boolean value indicating connection connectivity.
    */
   boolean isConnected();

   /**
    * Send a ping to the remote side and wait for a response
    * 
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    * @throws CommunicationIOException
    * @throws NotConnectedException
    */
   public void sendPing() throws CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, CommunicationIOException, NotConnectedException;

   /**
    * Gets the address of remote end-point.
    * 
    * @return A string representation of the address.
    */
   String getRemoteAddress();

   /**
    * Get this host's end-point address.
    * 
    * @return A string representation of the address.
    */
   String getLocalAddress();

   /**
    * Get a meaningful identifier for this connector. Used for logging and errors.
    * 
    * @return The identifier for this connector
    */
   String getIdentification();

   /**
    * Blocking communication method that sends a request and returns the subsequent response.
    * 
    * @param request
    *           The request to be sent to the connected remote end-point.
    * @return A Response object.
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    * @throws COmmunicationIOException
    * @throws NotConnectedException
    * @see Request
    * @see Response
    */
   Response exchange(Request request) throws CommunicationIOException, NotConnectedException,
         CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException;

   /**
    * Sends a request in real-time, throwing a CommunicationResponseException should a response not
    * be received by the specified time out.
    * 
    * @param request
    *           A Request object to be sent to the remote end-point.
    * @param timeout
    *           The duration of time to wait for a response.
    * @return A Response object.
    * @throws CommunicationIOException
    * @throws NotConnectedException
    * @throws CommunicationInterruptedException
    * @throws CommunicationResponseException
    * @throws CommunicationTransmissionException
    */
   Response exchange(Request request, int timeout) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException;
   
   /**
    * Returns the number of currently outstanding calls to exchange
    * 
    * @return Number of currently outstanding calls to exchange
    */
   int getNumOutstandingExchanges();
   
   /**
    * Called by the Cleversafe configuration framework.
    * 
    * @see Configuration
    */
   void initialize();

   ProtocolMessageFactory getProtocolMessageFactory();
}
