//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: dhendrickson
//
// Date: Nov 13, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import java.util.concurrent.atomic.AtomicInteger;

import org.apache.log4j.Logger;
import org.apache.mina.common.IdleStatus;
import org.apache.mina.common.IoHandler;
import org.apache.mina.common.IoSession;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.server.ClientSession;
import org.cleversafe.server.ServerApplication;

public class MinaAcceptorHandler implements IoHandler
{
   private static Logger _logger = Logger.getLogger(MinaAcceptorHandler.class);

   public final String sessionProperty = "ClientSession";

   private ServerApplication serverApplication;
   private int connectionTimeout;
   private boolean doIdleDisconnect;
   private int maxConnections;
   private final AtomicInteger connectionCount;

   public MinaAcceptorHandler()
   {
      this.serverApplication = null;
      this.connectionTimeout = 3600;
      this.doIdleDisconnect = false;
      this.maxConnections = 500;
      this.connectionCount = new AtomicInteger(0);
   }

   public ServerApplication getServerApplication()
   {
      return this.serverApplication;
   }

   public void setServerApplication(ServerApplication serverApplication)
   {
      this.serverApplication = serverApplication;
   }

   public int getConnectionTimeout()
   {
      return this.connectionTimeout;
   }

   public void setConnectionTimeout(int connectionTimeout)
   {
      this.connectionTimeout = connectionTimeout;
   }

   public boolean getIdleDisconnect()
   {
      return this.doIdleDisconnect;
   }

   public void setIdleDisconnect(boolean doIdleDisconnect)
   {
      this.doIdleDisconnect = doIdleDisconnect;
   }

   public int getMaxConnections()
   {
      return this.maxConnections;
   }

   public void setMaxConnections(int maxConnections)
   {
      this.maxConnections = maxConnections;
   }

   public void sessionCreated(IoSession session) throws Exception
   {
      this.connectionCount.incrementAndGet();

      if (this.connectionCount.get() > this.maxConnections)
      {
         _logger.error("Maximum connections exceeded, rejecting connection with "
               + session.getRemoteAddress());
         session.close();
      }
      else
      {
         _logger.info("Connection established with " + session.getRemoteAddress());
      }

      _logger.debug("Open connections: " + this.connectionCount);
   }

   public void sessionOpened(IoSession session)
   {
      session.setIdleTime(IdleStatus.BOTH_IDLE, this.connectionTimeout);

      ClientSession clientSession = new ClientSession();
      session.setAttribute(this.sessionProperty, clientSession);
   }

   public void messageReceived(IoSession session, Object message)
   {
      _logger.trace("Network message received");

      ClientSession clientSession = (ClientSession) session.getAttribute(this.sessionProperty);

      if (clientSession == null)
      {
         // This should not happen
         throw new RuntimeException("ClientSession for " + this.sessionProperty + " doesn't exist");
      }

      Request request = (Request) message;

      // Performance timing instrumentation
      long checkpoint0 = System.currentTimeMillis();
      int seqNumber =
            (message instanceof SequencedProtocolMessage)
                  ? ((SequencedProtocolMessage) message).getSequenceNumber()
                  : 0;

      if (_logger.isTraceEnabled())
         _logger.trace(String.format("Handler.process(%s, %d): started",
               request.getClass().getName(), seqNumber));

      Response response = this.serverApplication.service(request, clientSession);

      long checkpoint1 = System.currentTimeMillis();

      // NOTE: We intentionally do not join here -- the result of this write is not useful
      // and this could cause a deadlock in a single-process server/client
      session.write(response);

      long checkpoint2 = System.currentTimeMillis();
      if (_logger.isTraceEnabled())
         _logger.trace(String.format(
               "Handler.process(%s, %d): successful - %dms (proc:%dms, send:%dms)",
               request.getClass().getName(), seqNumber, checkpoint2 - checkpoint0, checkpoint1
                     - checkpoint0, checkpoint2 - checkpoint1));
   }

   public void messageSent(IoSession session, Object message) throws Exception
   {
   }

   public void exceptionCaught(IoSession session, Throwable cause)
   {
      _logger.warn("Network exception occurred", cause);

      session.close();
   }

   public void sessionIdle(IoSession session, IdleStatus status)
   {
      if (this.doIdleDisconnect)
      {
         session.close();
      }
   }

   public void sessionClosed(IoSession session)
   {
      ClientSession clientSession = (ClientSession) session.getAttribute(this.sessionProperty);
      if (clientSession == null)
      {
         // This should not happen
         throw new RuntimeException("ClientSession for " + this.sessionProperty + " doesn't exist");
      }
      // TODO: do we need synch on session to protect from double delete? just in case
      synchronized (clientSession)
      {
         SliceStore sliceStore = (SliceStore) clientSession.get(ClientSession.SLICE_STORE);
         if (sliceStore != null)
         {
            try
            {
               sliceStore.endSession();
            }
            catch (Exception ex)
            {
               _logger.warn("Failed to end abandoned session " + clientSession);
            }
         }
      }
      this.connectionCount.decrementAndGet();

      _logger.info("Connection terminated with " + session.getRemoteAddress());
      _logger.debug("Open connections: " + this.connectionCount);
   }
}
