//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER

//Cleversafe open-source code header - Version 1.1 - December 1, 2006

//Cleversafe Dispersed Storage(TM) is software for secure, private and
//reliable storage of the world's data using information dispersal.

//Copyright (C) 2005-2007 Cleversafe, Inc.

//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.

//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.

//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
//USA.

//Contact Information: Cleversafe, 10 W. 35th Street, 16th Floor #84,
//Chicago IL 60616
//email licensing@cleversafe.org

//END-OF-HEADER
//-----------------------
//Author: jquigley

//Date: Aug 9, 2007
//---------------------

package org.cleversafe.layer.communication.network.mina;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import org.apache.log4j.Logger;
import org.apache.mina.common.ByteBuffer;
import org.apache.mina.common.ConnectFuture;
import org.apache.mina.common.DefaultIoFilterChainBuilder;
import org.apache.mina.common.IdleStatus;
import org.apache.mina.common.IoHandler;
import org.apache.mina.common.IoSession;
import org.apache.mina.common.IoSessionConfig;
import org.apache.mina.common.PooledByteBufferAllocator;
import org.apache.mina.common.ThreadModel;
import org.apache.mina.common.WriteFuture;
import org.apache.mina.filter.LoggingFilter;
import org.apache.mina.filter.SSLFilter;
import org.apache.mina.filter.codec.ProtocolCodecFilter;
import org.apache.mina.transport.socket.nio.SocketConnector;
import org.apache.mina.transport.socket.nio.SocketConnectorConfig;
import org.apache.mina.transport.socket.nio.SocketSessionConfig;
import org.cleversafe.config.ConfigurationFactory;
import org.cleversafe.config.PropertiesProvider;
import org.cleversafe.config.exceptions.ConfigurationException;
import org.cleversafe.config.exceptions.ConfigurationItemNotDefinedException;
import org.cleversafe.exceptions.InitializationException;
import org.cleversafe.exceptions.NotImplementedException;
import org.cleversafe.layer.communication.ConnectorManager;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.network.IPNetworkConnector;
import org.cleversafe.layer.communication.network.mina.ssl.SSLContextFactory;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.protocol.SequencedProtocolMessage;
import org.cleversafe.serialization.ProtocolMessageFactory;
import org.cleversafe.util.NamedThreadFactory;

/*
 * TODO:
 * 
 * 1. Notification message queue
 * 2. Online SSL tear-down
 * 3. Evaluate default connect timeout semantics
 * 4. Implement connect timeout mechanism
 * 
 */

/**
 * Implementation of IPNetworkConnector using Apache's MINA framework.
 * 
 * @author John Quigley <jquigley@cleversafe.com>
 * @see org.apache.mina
 */
public class MinaConnector extends IPNetworkConnector implements IoHandler
{
   private static Logger _logger = Logger.getLogger(MinaConnector.class);
   private static final String DISCONNECTED_SESSION_ATTR = "_disconnected_";

   // ///////////////////////////////////////////////////////////////////////////
   // Configurable members.

   // TODO: should this be set to a higher value (like '30')?
   private int workerTimeout = 1;
   private boolean sslEnabled = false;
   private int maxPendingResponse = 512;
   private boolean networkLoggingEnabled = false;
   private final int writeTimeout = 1000 * 60; // Time to wait for a request to be written in ms

   // Suggested TCP window size in bytes.  To fully utilize, the complimentary client socket 
   // should set a window at least as large as this
   static private int TCP_BUFFER_SIZE = 512 * 1024;
   static private boolean NAGEL_PARAMETER = true;

   // TODO: make these configurable
   private int numServiceThreads = Runtime.getRuntime().availableProcessors() + 1;

   // ///////////////////////////////////////////////////////////////////////////
   // Internal members.

   private boolean isInitialized = false;
   private volatile boolean isConnectedState = false;
   private SocketConnector connector = null;
   private IoSession connectionSession = null;
   private ExecutorService connectionExecutor = null;
   private SocketConnectorConfig connectorConfig = null;
   private final AtomicInteger currentRequestNumber = new AtomicInteger();
   private final AtomicInteger numOutstandingExchanges = new AtomicInteger();
   private ConcurrentHashMap<Integer, CountDownLatch> responseNotify = null;
   private final ReentrantReadWriteLock connectionLock = new ReentrantReadWriteLock();
   private ConcurrentHashMap<Integer, Response> receivedResponses = null;
   private ConcurrentLinkedQueue<ProtocolMessage> unsolicitedMessageQueue = null;

   private ProtocolMessageFactory protocolMessageFactory = null;

   static
   {
      ByteBuffer.setUseDirectBuffers(false);
      ByteBuffer.setAllocator(new PooledByteBufferAllocator());

      try
      {
         PropertiesProvider propProvider =
               ConfigurationFactory.getPropertiesProvider(ConfigurationFactory.XML_CONFIG_TYPE);

         String[] tcpBufferSizeElement = {
               "org", "cleversafe", "layer", "communication", "network", "tcp-buffer-size"
         };
         try
         {
            TCP_BUFFER_SIZE = propProvider.getIntValue(tcpBufferSizeElement) * 1024;
            _logger.info("Initializing MinaConnector: TCP BUFFER SIZE is " + TCP_BUFFER_SIZE/1024 + "KB");
         }
         catch (ConfigurationItemNotDefinedException ex)
         {
            _logger.warn("Initializing MinaConnector: TCP BUFFER SIZE is not defined, use default "
                  + TCP_BUFFER_SIZE/1024 + "KB");
         }

         String[] nagelElement = {
               "org", "cleversafe", "layer", "communication", "network", "nagel"
         };
         try
         {
            NAGEL_PARAMETER = propProvider.getBooleanValue(nagelElement);
            _logger.warn("Initializing MinaConnector: Nagel parameter is " + NAGEL_PARAMETER);
         }
         catch (ConfigurationItemNotDefinedException ex)
         {
            _logger.warn("Initializing MinaConnector: Nagel parameter is not defined, use default "
                  + NAGEL_PARAMETER);
         }
      }
      catch (ConfigurationException ex)
      {
         _logger.error("Failed to parse properties file");
      }
   }

   /**
    * Returns the configured TCP buffer size
    * @return
    */
   public static int getTcpBufferSize()
   {
      return TCP_BUFFER_SIZE;
   }

   /**
    * Returns true if nagel's algorithm is disabled
    * @return
    */
   public static boolean isTcpNoDelay()
   {
      return NAGEL_PARAMETER;
   }

   // ///////////////////////////////////////////////////////////////////////////
   // Constructors.

   /**
    * Constructs a new MinaApplicationConnection.
    * 
    * <p>
    * To be used in the context of the configuration framework.
    */
   public MinaConnector()
   {
      super();
      try
      {
         this.protocolMessageFactory =
               ConfigurationFactory.getBindingsProvider(ConfigurationFactory.XML_CONFIG_TYPE).getDefaultImplementation(
                     ProtocolMessageFactory.class);
      }
      catch (ConfigurationException e)
      {
         throw new InitializationException(
               "Could not instantiate ProtocolMessageFactory from configuration", e);
      }
   }

   /**
    * Constructs a new MinaApplicationConnection.
    * 
    * <p>
    * To be used in normal operating circumstances.
    * 
    * @param host
    * @param port
    */
   public MinaConnector(
         final String host,
         final int port,
         final ConnectorManager manager,
         ProtocolMessageFactory pmFactory)
   {
      super(host, port, manager);
      this.protocolMessageFactory = pmFactory;
      initialize();
   }

   // ///////////////////////////////////////////////////////////////////////////
   // Class configuration.

   /**
    * Sets the enableSSL option.
    * 
    * @param enable
    */
   public void setEnableSSL(final boolean enable)
   {
      this.sslEnabled = enable;
   }

   /**
    * Sets the networkLoggingEnabled option.
    * 
    * @param enable
    */
   public void setNetworkLogging(final boolean enable)
   {
      this.networkLoggingEnabled = enable;
   }

   /**
    * Sets the maxPendingResponse option, enabling a bounded queue of outstanding message responses.
    * 
    * @param num
    */
   public void setMaxPendingResponse(final int num)
   {
      this.maxPendingResponse = num;
   }

   /**
    * Sets the workerTimeout, which affects the Mina Connector IO worker timeout.
    * 
    * @param num
    */
   public void setWorkerTimeout(final int num)
   {
      this.workerTimeout = num;
   }

   /**
    * Sets the numServiceThreads, which affects the number of threads in the connection IO handling
    * pool.
    * 
    * @param num
    */
   public void setNumServiceThreads(final int num)
   {
      this.numServiceThreads = num;
   }

   @Override
   public synchronized void initialize()
   {
      if (this.isInitialized)
      {
         throw new InitializationException("connection has already been initialized.");
      }

      super.initialize();
      this.isInitialized = true;
      getManager().onConnectorInit();

      _logger.info("Connection successfully initialized " + this);
   }

   // ///////////////////////////////////////////////////////////////////////////
   // Getters.

   public boolean isInitialized()
   {
      return this.isInitialized;
   }

   @Override
   public boolean isConnected()
   {
      // isConnectedState is volatile and only modified while holding the write lock, 
      // so no synchronization is necessary here
      return this.isConnectedState;
   }

   // ///////////////////////////////////////////////////////////////////////////
   // IPNetworkApplication implementation.

   /**
    * Attempt to establish a connection
    * 
    * Contract: - If no exception is thrown, connection was successful. - If connection was
    * successful, disconnect() MUST be called before exiting - If connection was NOT successful, it
    * is NOT necessary to call disconnect()
    */
   @Override
   public void connect() throws CommunicationIOException, CommunicationConnectionException
   {
      _connect(getConnectionTimeout());
   }

   /**
    * Close connection if established
    * 
    * Contract: - If disconnect() is called, caller can safely exit (all non-daemon threads are
    * stopped)
    */
   public void disconnect() throws CommunicationIOException, CommunicationInterruptedException
   {
      // It's worth calling isConnected() here to avoid a potentially long wait for the write lock
      if (!isConnected())
      {
         _logger.trace("Connection already disconnected");
         return;
      }

      // Shut down session.  Actual disconnect occurs in the sessionClosed() callback
      _logger.debug("Disconnect called on " + this);

      // We synchronize on connection modifying operations using a read/write lock.  
      // Here we are modifying the connection's state so we must acquire the write lock.
      this.connectionLock.writeLock().lock();
      try
      {
         if (isConnected())
         {
            // When sessionClosed() is called, indicate that _disconnect has already been called
            this.connectionSession.setAttribute(DISCONNECTED_SESSION_ATTR, true);
            this.connectionSession.close();
            _disconnect();
         }
      }
      finally
      {
         assert !isConnected();
         this.connectionLock.writeLock().unlock();
      }
   }

   private void _disconnect()
   {
      _logger.debug("disconnect was called for connector: " + this);

      // We synchronize on connection modifying operations using a read/write lock.  
      // Here we are modifying the connection's state so we must acquire the write lock.
      this.connectionLock.writeLock().lock();
      try
      {
         if (isConnected())
         {
            _logger.debug("Connection initiating termination " + this);

            this.connectionSession = null;

            // Notify all threads waiting for a response
            for (CountDownLatch notification : this.responseNotify.values())
            {
               notification.countDown();
            }

            this.receivedResponses.clear();

            // Clear and notify threads waiting for unsolicited messages
            this.unsolicitedMessageQueue.clear();
            synchronized (this.unsolicitedMessageQueue)
            {
               this.unsolicitedMessageQueue.notify();
            }

            // Tear down subsystem last
            _teardownSubsystem();
         }
      }
      finally
      {
         this.isConnectedState = false;
         this.connectionLock.writeLock().unlock();
      }
   }

   public Response exchange(Request request) throws CommunicationInterruptedException,
         CommunicationResponseException, CommunicationTransmissionException
   {
      return _exchange(request, 0);
   }

   public Response exchange(Request request, int timeout) throws CommunicationInterruptedException,
         CommunicationResponseException, CommunicationTransmissionException
   {
      return _exchange(request, timeout);
   }

   public String getLocalAddress()
   {
      throw new NotImplementedException("not yet implemented");
   }

   // ///////////////////////////////////////////////////////////////////////////
   // IOHandler implementation.

   /**
    * Invoked when the session is created. Initialize default socket parameters and user-defined
    * attributes here.
    */
   public void sessionCreated(IoSession session) throws Exception
   {
      // nothing to do
      _logger.debug("connection " + this + " created IO session");
   }

   /**
    * Invoked when the connection is opened. This method is not invoked if the transport type is
    * UDP.
    */
   public void sessionOpened(IoSession session) throws Exception
   {
      _logger.info("connection " + this + " opened IO session");
   }

   /**
    * Invoked when the connection is closed. This method is not invoked if the transport type is
    * UDP.
    */
   public void sessionClosed(IoSession session) throws Exception
   {
      _logger.info("connection " + this + " closed IO session");

      // If DISCONNECTED_SESSION_ATTR is set, _disconnect has already been called
      if (session.getAttribute(DISCONNECTED_SESSION_ATTR) == null)
      {
         _logger.debug("session disconnected attribute was not set, disconnecting");
         // We synchronize on connection modifying operations using a read/write lock.  
         // Here we are modifying the connection's state so we must acquire the write lock.
         this.connectionLock.writeLock().lock();
         try
         {
            _logger.debug("sessionClosed calling _disconnect");
            _disconnect();
         }
         finally
         {
            assert !isConnected();
            this.connectionLock.writeLock().unlock();
            _logger.debug("Connection lock was successfully released");
         }
      }
      else
      {
         _logger.debug("Session state indicated that a disconnect was unnecessary");
      }
   }

   /**
    * Invoked when the connection is idle. Refer to {@link IdleStatus}. This method is not invoked
    * if the transport type is UDP.
    */
   public void sessionIdle(IoSession session, IdleStatus status) throws Exception
   {
      // nothing to do
      _logger.info("connection " + this + " idled IO session");
   }

   /**
    * Invoked when any exception is thrown by user {@link IoHandler} implementation or by MINA. If
    * <code>cause</code> is instanceof {@link IOException}, MINA will close the connection
    * automatically.
    */
   public void exceptionCaught(IoSession session, Throwable cause) throws Exception
   {
      _logger.info("connection " + this + " caught an exception: " + cause.toString());
      getManager().onError();

      if (cause instanceof IOException)
      {
         disconnect();
      }
      else
      {
         throw new CommunicationIOException("Connection caught an unhandled exception", cause);
      }
   }

   /**
    * Invoked when protocol message is received. Implement your protocol flow here.
    */
   public void messageReceived(IoSession session, Object message) throws Exception
   {
      if (_logger.isTraceEnabled())
      {
         _logger.trace("connection " + this + " received message " + message);
      }

      Response response = (Response) message;

      assert response instanceof SequencedProtocolMessage : "Response isn't a SequencedProtocolMessage";

      SequencedProtocolMessage sequenceResponse = (SequencedProtocolMessage) response;

      CountDownLatch notification = this.responseNotify.get(sequenceResponse.getSequenceNumber());

      // first, check if message had a corresponding request message id
      if (notification != null)
      {
         this.receivedResponses.put(sequenceResponse.getSequenceNumber(), response);
         notification.countDown();
      }
      // otherwise, check to see if message has the unsolicited flag set
      else if (sequenceResponse.isUnsolicited())
      {
         this.unsolicitedMessageQueue.add(sequenceResponse);
         this.unsolicitedMessageQueue.notify();
      }
      // if non of the above conditions, this is an unknown message
      else
      {
         // This can happen in many legitimate circumstances including when a write times out
         _logger.debug("unexpected message received: " + sequenceResponse.getSequenceNumber());
      }
   }

   /**
    * Invoked when protocol message that user requested by {@link IoSession#write(Object)} is sent
    * out actually.
    */
   public void messageSent(IoSession session, Object message) throws Exception
   {
      if (_logger.isTraceEnabled())
         _logger.trace("connection " + this + " sent message " + message);
   }

   // ///////////////////////////////////////////////////////////////////////////
   // Utility methods.

   private void initializeSubsystem() throws CommunicationConnectionException
   {
      this.connectionExecutor =
            Executors.newCachedThreadPool(new NamedThreadFactory("Mina <" + getRemoteAddress()
                  + ">"));

      initializeConnector();
      initializeFilter();

      this.responseNotify = new ConcurrentHashMap<Integer, CountDownLatch>();
      this.receivedResponses = new ConcurrentHashMap<Integer, Response>();
      this.unsolicitedMessageQueue = new ConcurrentLinkedQueue<ProtocolMessage>();

      _logger.debug("Connection subsystem initialized for " + this);
   }

   private void initializeConnector()
   {
      this.connector = new SocketConnector(this.numServiceThreads, this.connectionExecutor);

      this.connector.setWorkerTimeout(this.workerTimeout);

      this.connectorConfig = this.connector.getDefaultConfig();
      this.connectorConfig.setThreadModel(ThreadModel.MANUAL);

      // This setting is critical for throughput over high latency connections
      this.connectorConfig.getSessionConfig().setSendBufferSize(MinaConnector.TCP_BUFFER_SIZE);
      this.connectorConfig.getSessionConfig().setReceiveBufferSize(MinaConnector.TCP_BUFFER_SIZE);
   }

   private void initializeFilter() throws InitializationException
   {
      DefaultIoFilterChainBuilder filterChain = this.connectorConfig.getFilterChain();

      // logging
      if (this.networkLoggingEnabled)
      {
         filterChain.addLast("logger", new LoggingFilter());
      }

      this.connectorConfig.getSessionConfig().setTcpNoDelay(NAGEL_PARAMETER);

      // ssl
      try
      {
         if (this.sslEnabled)
         {
            SSLFilter sslFilter = new SSLFilter(SSLContextFactory.getInstance(false, null, null));
            sslFilter.setUseClientMode(true);
            filterChain.addLast("sslFilter", sslFilter);
         }
      }
      catch (GeneralSecurityException e)
      {
         throw new InitializationException("Connection failed initializing SSL engine.", e);
      }

      // protocol
      ProtocolMessageFactory messageFactory = getProtocolMessageFactory();

      ProtocolCodecFilter codecFilter =
            new ProtocolCodecFilter(new ProtocolMessageCodecFactory(
                  messageFactory.getEncoderClass(), messageFactory.getDecoderClass()));
      this.connectorConfig.getFilterChain().addLast("codec", codecFilter);
   }

   private void _teardownSubsystem()
   {
      this.connectorConfig = null;
      this.connector = null;

      this.responseNotify = null;
      this.receivedResponses = null;
      this.unsolicitedMessageQueue = null;

      this.connectionExecutor.shutdown();
      this.connectionExecutor = null;

      _logger.debug("Connection subsystem torn down for " + this);
   }

   public void _connect(int timeout) throws CommunicationIOException,
         CommunicationConnectionException
   {
      // We synchronize on connection modifying operations using a read/write lock.  
      // Here we are modifying the connection's state so we must acquire the write lock.
      this.connectionLock.writeLock().lock();
      try
      {
         if (!isInitialized())
         {
            throw new InitializationException("connection must be initialized");
         }

         // Initialize subsystem for connection attempt
         initializeSubsystem();

         // Set Mina connect timeout in seconds
         this.connectorConfig.setConnectTimeout(timeout / 1000);

         // Begin asynchronous connection attempt
         ConnectFuture future =
               this.connector.connect(new InetSocketAddress(getHost(), getPort()), this,
                     this.connectorConfig);

         // Wait until connection is successful or the above timeout occurs
         future.join();

         if (!future.isConnected())
         {
            // Tear down subsystem and clean-up
            _teardownSubsystem();
            throw new CommunicationIOException("Connection couldn't be established");
         }

         this.connectionSession = future.getSession();

         _initializeSession();
         this.isConnectedState = true;
      }
      finally
      {
         this.connectionLock.writeLock().unlock();
      }
   }

   private Response _exchange(Request request, int timeout)
         throws CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      // We synchronize on connection modifying operations using a read/write lock.  
      // Here we are only reading the connection's state so we must acquire the read lock.
      Lock readLock = this.connectionLock.readLock();
      boolean exchangeSucceeded = false;
      readLock.lock();
      this.numOutstandingExchanges.incrementAndGet();
      try
      {
         if (_logger.isTraceEnabled())
            _logger.trace("connection " + this + " preparing for request exchange");

         // If we are connected, our read lock ensures that the session will continue to exist
         if (!isConnected())
         {
            throw new CommunicationTransmissionException(this + ": Not connected");
         }

         Response response = null;

         // FIXME: This does not handle non-sequenced protocol messages gracefully
         if (request instanceof SequencedProtocolMessage)
         {
            if (this.responseNotify.size() > this.maxPendingResponse)
            {
               _logger.debug("Connection " + this
                     + " has exceeded its maximum message queue bounds");
               throw new CommunicationTransmissionException(
                     "Connection has exceeded its maximum message queue bounds");
            }

            // Set sequence number
            int requestNumber = this.currentRequestNumber.incrementAndGet();
            assert requestNumber > 0 : "Request number is not in the valid range: " + requestNumber;
            ((SequencedProtocolMessage) request).setSequenceNumber(requestNumber);

            long checkpoint0 = System.currentTimeMillis();
            long checkpoint1 = 0;
            long checkpoint2 = 0;
            long checkpoint3 = 0;
            if (_logger.isTraceEnabled())
               _logger.trace(String.format("%s - MinaConnector.exchange(%s, %d): started",
                     this.toString(), request.getClass().getName(), requestNumber));

            // Latch will be decremented to zero when a response is received
            CountDownLatch notification = new CountDownLatch(1);
            this.responseNotify.put(requestNumber, notification);

            // Send message and check that the write was successful.  We must release our read lock
            // before waiting because Mina may need to make a callback that requires a write lock
            // to service this request
            if (_logger.isTraceEnabled())
               _logger.trace("Writing message: " + request);
            WriteFuture writeFuture = this.connectionSession.write(request);
            readLock.unlock();

            // FIXME: There appears to be a rare race condition that may be Mina's fault where
            // a writeFuture will never get notified if the connection is closed.  A timeout is used
            // here as a last resort
            if (!writeFuture.join(this.writeTimeout))
            {
               _logger.warn(this + ": Request write timed out in " + this.writeTimeout + "ms");
            }
            //writeFuture.join();

            checkpoint1 = System.currentTimeMillis();

            // We must re-check that we are connected before continuing, since the connection
            // session may have been destroyed
            readLock.lock();
            if (!isConnected())
            {
               throw new CommunicationTransmissionException(this + ": Connection lost during send");
            }

            if (writeFuture.isWritten())
            {
               // Successful write, wait for a response
               try
               {
                  // We must release our read lock before waiting
                  readLock.unlock();

                  checkpoint2 = System.currentTimeMillis();

                  if (timeout > 0)
                  {
                     notification.await(timeout, TimeUnit.MILLISECONDS);
                  }
                  else
                  {
                     // Wait indefinitely
                     notification.await();
                  }

                  checkpoint3 = System.currentTimeMillis();

                  // We must re-check that we are connected before continuing, since the connection
                  // session may have been destroyed
                  readLock.lock();
                  if (!isConnected())
                  {
                     throw new CommunicationTransmissionException(this
                           + ": Connection lost during wait for response");
                  }
               }
               catch (InterruptedException e)
               {
                  // onError() may require a write lock
                  readLock.unlock();
                  getManager().onError();
                  readLock.lock();

                  _logger.info("Communication interrupted while waiting on message response");
                  throw new CommunicationInterruptedException(
                        "Communication interrupted while waiting on message response", e);
               }
            }
            else
            {
               // Write failed
               String msg = this + "Send messsage failed: " + request;
               _logger.debug(msg);
               this.responseNotify.remove(requestNumber);
               throw new CommunicationTransmissionException(msg);
            }

            response = this.receivedResponses.remove(requestNumber);
            this.responseNotify.remove(requestNumber);

            if (response == null)
            {
               _logger.info("No response for: " + requestNumber);

               // onError() may require a write lock
               readLock.unlock();
               getManager().onError();
               readLock.lock();

               throw new CommunicationResponseException("waiting for response timed out: "
                     + timeout);
            }
            else
            {
               if (_logger.isTraceEnabled())
                  _logger.trace("Connection received response " + response);
               exchangeSucceeded = true;

               long checkpoint4 = System.currentTimeMillis();
               if (_logger.isTraceEnabled())
                  _logger.trace(String.format(
                        "%s - MinaConnector.exchange(%s, %d): successful - %dms (1:%dms, 2:%dms, 3:%dms, 4:%dms)",
                        this.toString(), request.getClass().getName(), requestNumber, checkpoint4
                              - checkpoint0, checkpoint1 - checkpoint0, checkpoint2 - checkpoint1,
                        checkpoint3 - checkpoint2, checkpoint4 - checkpoint3));
            }
         }

         if (response == null)
         {
            throw new CommunicationResponseException("did not receive a response");
         }

         return response;
      }
      finally
      {
         this.numOutstandingExchanges.decrementAndGet();
         if (exchangeSucceeded)
         {
            getManager().onSuccess();
         }
         readLock.unlock();
      }
   }

   public int getNumOutstandingExchanges()
   {
      return this.numOutstandingExchanges.get();
   }

   private void _initializeSession()
   {
   }

   /**
    * Log some useful stats about this connection
    */
   public void printStats()
   {
      StringBuilder stats = new StringBuilder();

      stats.append("pending: ");
      ConcurrentHashMap<Integer, CountDownLatch> pending = this.responseNotify;
      if (pending != null)
      {
         stats.append(pending.size());
      }
      else
      {
         stats.append("<unknown>");
      }

      IoSession ioSession = this.connectionSession;
      if (ioSession != null)
      {
         IoSessionConfig ioSessionConfig = ioSession.getConfig();
         if (ioSessionConfig instanceof SocketSessionConfig)
         {
            SocketSessionConfig cfg = (SocketSessionConfig) ioSessionConfig;
            stats.append(", rcv buffer: ");
            stats.append(cfg.getReceiveBufferSize() / 1024);
            stats.append(", snd buffer: ");
            stats.append(cfg.getSendBufferSize() / 1024);
         }
      }

      _logger.info(stats.toString());
   }

   @Override
   public ProtocolMessage getNotification() throws CommunicationInterruptedException
   {
      try
      {
         this.unsolicitedMessageQueue.wait();
      }
      catch (final InterruptedException e)
      {
         throw new CommunicationInterruptedException(
               "interrupted while waiting for unsolicited message", e);
      }
      return this.unsolicitedMessageQueue.poll();
   }

   @Override
   public ProtocolMessage getNotification(int timeout) throws CommunicationInterruptedException
   {
      try
      {
         this.unsolicitedMessageQueue.wait(timeout);
      }
      catch (final InterruptedException e)
      {
         throw new CommunicationInterruptedException(
               "interrupted while waiting for unsolicited message", e);
      }
      return this.unsolicitedMessageQueue.poll();
   }

   public ProtocolMessageFactory getProtocolMessageFactory()
   {
      return this.protocolMessageFactory;
   }

   /////////////////////////////////////////////////////////////////////////////
   /// Utility methods

   @Override
   public String toString()
   {
      return "<MinaConnector endpoint:" + getRemoteAddress() + " connected:" + isConnected()
            + " ssl:" + this.sslEnabled + ">";
   }
}
