//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER

/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */

package org.cleversafe.layer.communication.network.mina.ssl;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.Security;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;

/**
 * Factory to create a bougus SSLContext.
 */
public class SSLContextFactory
{
   /**
    * Protocol to use.
    */
   private static final String PROTOCOL = "TLS";

   private static final String KEY_MANAGER_FACTORY_ALGORITHM;

   static
   {
      String algorithm = Security.getProperty("ssl.KeyManagerFactory.algorithm");
      if (algorithm == null)
      {
         algorithm = "SunX509";
      }

      KEY_MANAGER_FACTORY_ALGORITHM = algorithm;
   }

   /**
    * Bougus Server certificate keystore file name.
    */
   //private static String KEYSTORE = "conf/dev/slice-server/ssl.cert";

   /**
    * NOTE: Keystore generated in the following way:
    * 
    * jquigley@laptop:~$ keytool -genkey -alias cleversafe -keysize 4096 \
    *    -validity 365 -keyalg RSA \
    *    -dname "CN=cleversafe.com, OU=cleversafe.com, O=Cleversafe Inc., L=Chicago, S=Illinois, C=SE" \
    *    -keypass captainbackup -storepass password.store -keystore ssl.cert
    */
   
   /*
   private static final char[] PW = {
      'c', 'a', 'p', 't', 'a', 'i', 'n', 'b', 'a', 'c', 'k', 'u', 'p'
   };
   */

   private static SSLContext serverInstance = null;

   private static SSLContext clientInstance = null;

   /**
    * Get SSLContext singleton.
    * 
    * @return SSLContext
    * @throws java.security.GeneralSecurityException
    * 
    */
   public static SSLContext getInstance(boolean server, String sslCertificate, String sslCertificatePassword) throws GeneralSecurityException
   {
      SSLContext retInstance = null;
      if (server)
      {
         if (serverInstance == null)
         {
            synchronized (SSLContextFactory.class)
            {
               if (serverInstance == null)
               {
                  try
                  {
                     serverInstance = createBougusServerSSLContext(sslCertificate, sslCertificatePassword);
                  }
                  catch (Exception ioe)
                  {
                     throw new GeneralSecurityException("Can't create Server SSLContext:" + ioe);
                  }
               }
            }
         }
         retInstance = serverInstance;
      }
      else
      {
         if (clientInstance == null)
         {
            synchronized (SSLContextFactory.class)
            {
               if (clientInstance == null)
               {
                  clientInstance = createBougusClientSSLContext();
               }
            }
         }
         retInstance = clientInstance;
      }
      return retInstance;
   }

   private static SSLContext createBougusServerSSLContext(String sslCertificate, String sslCertificatePassword) throws GeneralSecurityException,
         IOException
   {
      // Create keystore
      KeyStore ks = KeyStore.getInstance("JKS");
      InputStream in = new FileInputStream(sslCertificate);

      try
      {
         ks.load(in, sslCertificatePassword.toCharArray());
      }
      finally
      {
         if (in != null)
         {
            try
            {
               in.close();
            }
            catch (IOException ignored)
            {
            }
         }
      }

      // Set up key manager factory to use our key store
      KeyManagerFactory kmf = KeyManagerFactory.getInstance(KEY_MANAGER_FACTORY_ALGORITHM);
      kmf.init(ks, sslCertificatePassword.toCharArray());

      // Initialize the SSLContext to work with our key managers.
      SSLContext sslContext = SSLContext.getInstance(PROTOCOL);
      sslContext.init(kmf.getKeyManagers(), TrustManagerFactory.X509_MANAGERS, null);

      return sslContext;
   }

   private static SSLContext createBougusClientSSLContext() throws GeneralSecurityException
   {
      SSLContext context = SSLContext.getInstance(PROTOCOL);
      context.init(null, TrustManagerFactory.X509_MANAGERS, null);
      return context;
   }
}
