//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: jquigley
//
// Date: Aug 28, 2007
//---------------------

package org.cleversafe.layer.communication.network.unreliable;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.layer.communication.exceptions.CommunicationConnectionException;
import org.cleversafe.layer.communication.exceptions.CommunicationException;
import org.cleversafe.layer.communication.exceptions.CommunicationIOException;
import org.cleversafe.layer.communication.exceptions.CommunicationInterruptedException;
import org.cleversafe.layer.communication.exceptions.CommunicationResponseException;
import org.cleversafe.layer.communication.exceptions.CommunicationTransmissionException;
import org.cleversafe.layer.communication.exceptions.NotConnectedException;
import org.cleversafe.layer.communication.network.IPNetworkConnector;
import org.cleversafe.layer.protocol.ProtocolMessage;
import org.cleversafe.layer.protocol.Request;
import org.cleversafe.layer.protocol.Response;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreLayerException;
import org.cleversafe.serialization.ProtocolMessageFactory;

/**
 * An Unreliable network connector wraps an IPNetworkConnector but will fail by throwing the
 * underlying exceptions with a configurable probability.
 */
public class UnreliableNetworkConnector extends IPNetworkConnector
{
   private static Logger _logger = Logger.getLogger(UnreliableNetworkConnector.class);

   private IPNetworkConnector _conn = null;

   // These values should be between 0 and 1
   double _corruptionFailureRate = 0.0;
   double _wrongTransactionFailureRate = 0.0;

   private Map<String, Map<Class<?>, Double>> methodExceptionRate =
         new HashMap<String, Map<Class<?>, Double>>();

   // //////////////////////////////////////////////////////////////////////////
   // Constructors

   /**
    * Default Constructor
    */
   public UnreliableNetworkConnector()
   {
   }

   /**
    * Constructs an UnreliableNetworkApplicationConnection with 0 probability of failure
    * 
    * @param databaseDir
    */
   public UnreliableNetworkConnector(IPNetworkConnector conn)
   {
      this.setApplicationConnection(conn);
      this.initialize();
   }

   public UnreliableNetworkConnector(
         IPNetworkConnector conn,
         double readFailureRate,
         double writeFailureRate) throws NoSuchMethodException
   {
      this._conn = conn;
      this.setMethodExceptionRate("read", CommunicationIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", CommunicationIOException.class, writeFailureRate);
      this.initialize();
   }

   public UnreliableNetworkConnector(
         IPNetworkConnector conn,
         double readFailureRate,
         double writeFailureRate,
         double corruptionFailureRate) throws NoSuchMethodException
   {
      this._conn = conn;
      this.setMethodExceptionRate("read", CommunicationIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", CommunicationIOException.class, writeFailureRate);
      this._corruptionFailureRate = corruptionFailureRate;
      this.initialize();
   }

   public UnreliableNetworkConnector(
         IPNetworkConnector conn,
         double readFailureRate,
         double writeFailureRate,
         double corruptionFailureRate,
         double wrongTransactionFailureRate) throws NoSuchMethodException
   {
      this._conn = conn;
      this.setMethodExceptionRate("read", CommunicationIOException.class, readFailureRate);
      this.setMethodExceptionRate("write", CommunicationIOException.class, writeFailureRate);
      this._corruptionFailureRate = corruptionFailureRate;
      this._wrongTransactionFailureRate = wrongTransactionFailureRate;
      this.initialize();
   }

   // //////////////////////////////////////////////////////////////////////////

   public void setApplicationConnection(IPNetworkConnector conn)
   {
      this._conn = conn;
   }

   public void setCorruptionFailureRate(double corruptionFailureRate)
   {
      this._corruptionFailureRate = corruptionFailureRate;
   }

   public void setWrongTransactionFailureRate(double wrongTransactionFailureRate)
   {
      this._wrongTransactionFailureRate = wrongTransactionFailureRate;
   }

   /**
    * Used by configuration to set the unreliability of different methods
    * 
    * @param value
    */
   public void setUnreliability(String value)
   {
      String[] parameters = value.split(",");

      if (parameters.length != 3)
      {
         _logger.warn("Unable to parse the parameters from string: " + value);
         return;
      }

      double rate = Double.parseDouble(parameters[2]);

      try
      {
         String className = parameters[1];
         Class<?> exceptionClass = Class.forName(className);
         setMethodExceptionRate(parameters[0], exceptionClass, rate);
      }
      catch (NoSuchMethodException e)
      {
         _logger.warn("No such method exception was caught", e);
      }
      catch (ClassNotFoundException e)
      {
         _logger.warn("Class not found exception was caught", e);
      }
   }

   /**
    * This method is used to set the rate at which any wrapped slice store method in will throw an
    * exception of a certain type.
    * 
    * @param methodName
    * @param exception
    * @param rate
    * @throws NoSuchMethodException
    */
   public void setMethodExceptionRate(String methodName, Class<?> exception, double rate)
         throws NoSuchMethodException
   {
      // Assert the error rate is within the proper bounds
      assert (rate >= 0 && rate <= 1) : "The rate must be between 0 and 1";

      // Ensure that the method exists for this class
      Method foundMethod = null;
      for (Method method : this.getClass().getMethods())
      {
         if (method.getName().equals(methodName))
         {
            foundMethod = method;
            break;
         }
      }
      if (foundMethod == null)
      {
         throw new NoSuchMethodException();
      }

      // Ensure the method can thrown the given exception
      List<Class<?>> exceptions = Arrays.asList(foundMethod.getExceptionTypes());
      if (!exceptions.contains(exception))
      {
         throw new IllegalArgumentException("The provided exception is not thrown by the method: "
               + methodName);
      }

      // Get the exceptions thrown map for this method
      Map<Class<?>, Double> exceptionsThrown = methodExceptionRate.get(methodName);
      if (exceptionsThrown == null)
      {
         exceptionsThrown = new HashMap<Class<?>, Double>();
         methodExceptionRate.put(methodName, exceptionsThrown);
      }

      // Save exception and rate in the exceptions thrown map
      exceptionsThrown.put(exception, rate);
   }

   /**
    * Throws an exception for a certain method given its name. These exceptions must be registered
    * using the setMethodExceptionRate method.
    * 
    * @param methodName
    * @throws SliceStoreLayerException
    */
   private void conditionalException(String methodName) throws CommunicationException
   {
      // Get the exceptions thrown map for this method
      Map<Class<?>, Double> exceptionsThrown = methodExceptionRate.get(methodName);

      if (exceptionsThrown != null)
      {
         for (Map.Entry<Class<?>, Double> entry : exceptionsThrown.entrySet())
         {
            if (Math.random() < entry.getValue()) // Throw the exception
            {
               Class<?> exceptionClass = entry.getKey();

               CommunicationException genericCommunicationException = null;
               try
               {
                  genericCommunicationException =
                        (CommunicationException) exceptionClass.getConstructor(String.class).newInstance(
                              "Generated unreliable exception");
               }
               catch (Exception e)
               {
                  throw new RuntimeException("Unable to construct SliceStoreLayerException", e);
               }

               throw genericCommunicationException;
            }
         }
      }
   }

   public IPNetworkConnector getBaseNetworkApplicationConnection()
   {
      return this._conn;
   }

   public void initialize()
   {
      assert this._conn != null : "Application connection to be wrapped has not been set";
   }

   // //////////////////////////////////////////////////////////////////////////

   @Override
   public ProtocolMessage getNotification() throws CommunicationInterruptedException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      return this._conn.getNotification();
   }

   @Override
   public ProtocolMessage getNotification(int timeout) throws CommunicationInterruptedException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      return this._conn.getNotification(timeout);
   }

   public void connect() throws CommunicationIOException, CommunicationConnectionException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationIOException)
            throw (CommunicationIOException) ex;
         else if (ex instanceof CommunicationConnectionException)
            throw (CommunicationConnectionException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      this._conn.ensureConnected();
   }

   public void disconnect() throws CommunicationIOException, CommunicationInterruptedException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationIOException)
            throw (CommunicationIOException) ex;
         else if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      this._conn.disconnect();
   }

   public Response exchange(Request request) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationIOException)
            throw (CommunicationIOException) ex;
         else if (ex instanceof NotConnectedException)
            throw (NotConnectedException) ex;
         else if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
         else if (ex instanceof CommunicationResponseException)
            throw (CommunicationResponseException) ex;
         else if (ex instanceof CommunicationTransmissionException)
            throw (CommunicationTransmissionException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      return this._conn.exchange(request);
   }

   public Response exchange(Request request, int timeout) throws CommunicationIOException,
         NotConnectedException, CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException
   {
      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationIOException)
            throw (CommunicationIOException) ex;
         else if (ex instanceof NotConnectedException)
            throw (NotConnectedException) ex;
         else if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
         else if (ex instanceof CommunicationResponseException)
            throw (CommunicationResponseException) ex;
         else if (ex instanceof CommunicationTransmissionException)
            throw (CommunicationTransmissionException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      return this._conn.exchange(request, timeout);
   }

   public String getLocalAddress()
   {
      return this._conn.getLocalAddress();
   }

   public boolean isConnected()
   {
      return this._conn.isConnected();
   }

   @Override
   public void sendPing() throws CommunicationInterruptedException, CommunicationResponseException,
         CommunicationTransmissionException, CommunicationIOException, NotConnectedException
   {

      // /////////////////////////////////////////////////////////////////////////////////////////////
      // Throw conditional exception
      String thisMethodName = (new Exception()).getStackTrace()[0].getMethodName();

      try
      {
         conditionalException(thisMethodName);
      }
      catch (CommunicationException ex)
      {
         if (ex instanceof CommunicationIOException)
            throw (CommunicationIOException) ex;
         else if (ex instanceof NotConnectedException)
            throw (NotConnectedException) ex;
         else if (ex instanceof CommunicationInterruptedException)
            throw (CommunicationInterruptedException) ex;
         else if (ex instanceof CommunicationResponseException)
            throw (CommunicationResponseException) ex;
         else if (ex instanceof CommunicationTransmissionException)
            throw (CommunicationTransmissionException) ex;
      }
      // End throw of conditional exception
      // /////////////////////////////////////////////////////////////////////////////////////////////

      this._conn.sendPing();
   }

   public ProtocolMessageFactory getProtocolMessageFactory()
   {
      return _conn.getProtocolMessageFactory();
   }

   public int getNumOutstandingExchanges()
   {
      return this._conn.getNumOutstandingExchanges();
   }
}
