//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 23, 2007
//---------------------

package org.cleversafe.layer.grid;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.TablePrinter;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.TypeAggregator;

/**
 * Helper class to organize transformation between datasources and slices for grid operations
 */
public class OperationScorecard implements Cloneable
{
   // private static Logger _logger = Logger.getLogger(DataTransformationMatrix.class);

   /**
    * Status value for each entry in the matrix
    */
   public static enum Status
   {
      EXISTING("existing"), // Data for this entry exists
      EXPECTED("expected"), // This entry is expected
      PENDING("pending"), // Data has been requested and is pending arrival
      IGNORED("ignored"), // Data for this entry is no longer needed
      ERROR("error"); // An error occurred with this entry

      // Readable name string
      private String name;

      private Status(String name)
      {
         this.name = name;
      }

      @Override
      public String toString()
      {
         return this.name;
      }
   }

   // Transformation matrix - rows are slices of a datasource, columns are slices of a slicestore
   private Entry[][] matrix;

   // Ordered list of datasources
   private List<SourceName> rows;

   // Ordered list of stores
   private List<SliceStore> cols;

   // Datasource for each row
   private Map<SourceName, Integer> rowIndex;

   // SliceStore for each column
   private Map<SliceStore, Integer> columnIndex;

   // Status histograms for all rows
   private EnumMap<Status, Integer>[] rowStatus;

   // Status histograms for all columns
   private EnumMap<Status, Integer>[] columnStatus;

   // Set of revisions for each source that are known to be invalid or impossible to restore
   private Set<Long>[] revisionBlacklist;

   /**
    * Construct a new matrix helper from known sources and stores. Guarantees that the order of
    * sources and stores will be preserved in rows and columns unless setStoreOrder() is called.
    * 
    * @param dataSources
    * @param sliceStores
    */
   @SuppressWarnings("unchecked")
   public OperationScorecard(List<SourceName> dataSources, List<SliceStore> sliceStores)
   {
      this.matrix = new Entry[dataSources.size()][sliceStores.size()];
      this.rows = dataSources;
      this.cols = sliceStores;

      // Build indices
      this.rowIndex = new HashMap<SourceName, Integer>();
      this.rowStatus = new EnumMap[dataSources.size()];
      this.revisionBlacklist = new HashSet[dataSources.size()];
      for (int i = 0; i < this.rows.size(); ++i)
      {
         this.rowIndex.put(dataSources.get(i), i);
         this.rowStatus[i] = new EnumMap<Status, Integer>(Status.class);
         this.revisionBlacklist[i] = new HashSet<Long>();
      }

      this.columnIndex = new HashMap<SliceStore, Integer>();
      this.columnStatus = new EnumMap[sliceStores.size()];
      for (int i = 0; i < this.cols.size(); ++i)
      {
         this.columnIndex.put(sliceStores.get(i), i);
         this.columnStatus[i] = new EnumMap<Status, Integer>(Status.class);
      }
   }

   /**
    * Add a new slice to the matrix
    * 
    * @param source
    * @param store
    * @param entry
    */
   public void addEntry(SourceName source, SliceStore store, Entry entry)
   {
      int row = this.getRow(source);
      int col = this.getColumn(store);
      this.helperSetEntry(row, col, entry);

   }

   /**
    * Internal helper to add an entry to the matrix. All changes to matrix membership should occur
    * through this function
    */
   private void helperSetEntry(int row, int col, Entry entry)
   {
      assert entry != null;

      // Keep status histograms in sync
      Entry oldEntry = this.matrix[row][col];
      Status oldStatus = oldEntry != null ? oldEntry.getStatus() : null;
      Status newStatus = entry.getStatus();
      this.helperUpdateStatusHistograms(row, col, oldStatus, newStatus);

      this.matrix[row][col] = entry;
   }

   /**
    * Updates status histograms to reflect new status for an entry at {row, col}. oldStatus may be
    * passed as null if this entry was previously null
    */
   private void helperUpdateStatusHistograms(int row, int col, Status oldStatus, Status newStatus)
   {
      assert newStatus != null;

      Integer i;
      if (oldStatus != newStatus)
      {
         if (oldStatus != null)
         {
            // Decrement old status
            i = this.rowStatus[row].get(oldStatus);
            assert (i != null && i > 0) : "Row histogram out of sync";
            this.rowStatus[row].put(oldStatus, i - 1);

            i = this.columnStatus[col].get(oldStatus);
            assert (i != null && i > 0) : "Column histogram out of sync";
            this.columnStatus[col].put(oldStatus, i - 1);
         }

         // Increment new status
         i = this.rowStatus[row].get(newStatus);
         this.rowStatus[row].put(newStatus, i != null ? i + 1 : 1);

         i = this.columnStatus[col].get(newStatus);
         this.columnStatus[col].put(newStatus, i != null ? i + 1 : 1);
      }
   }

   /**
    * Get a matrix entry
    * 
    * @param source
    * @param store
    * @return
    */
   public Entry getEntry(SourceName source, SliceStore store)
   {
      int row = this.getRow(source);
      int col = this.getColumn(store);
      return this.matrix[row][col];
   }

   /**
    * Unlink any data payload from scorecard entries to allow it
    * to be garbage collected.
    */
   public void clearData()
   {
      for (int row = 0; row < this.rows.size(); ++row)
      {
         for (int col = 0; col < this.cols.size(); ++col)
         {
            Entry entry = this.matrix[row][col];
            if (entry != null)
            {
               entry.clearData();
            }
         }
      }
   }

   /**
    * Sets all entries for a store to the given status
    * 
    * @param status
    */
   public void setStoreStatus(SliceStore store, Status status)
   {
      this.setStoreStatus(store, status, null);
   }

   /**
    * Sets all entries for a store to the given status (non-ERROR)
    * 
    * @param status
    */
   public void setStoreStatus(SliceStore store, Status status, Exception ex)
   {
      int col = this.getColumn(store);
      for (int row = 0; row < this.rowIndex.size(); ++row)
      {
         Entry entry = this.matrix[row][col];
         if (entry != null)
         {
            this.helperUpdateStatusHistograms(row, col, entry.getStatus(), status);
            entry.setStatus(status);
            entry.setException(ex);
         }
      }
   }

   /**
    * Sets the status for a given entry
    */
   public void setStatus(SourceName source, SliceStore store, Status status)
   {
      this.setStatus(source, store, status, null);
   }

   /**
    * Sets the status and an exception for a given entry
    */
   public void setStatus(SourceName source, SliceStore store, Status status, Exception ex)
   {
      int row = this.getRow(source);
      int col = this.getColumn(store);

      Entry entry = this.matrix[row][col];
      assert entry != null : "Attempt to set status for a nonexistant entry";
      this.helperUpdateStatusHistograms(row, col, entry.getStatus(), status);
      entry.setStatus(status);
      entry.setException(ex);
   }

   /**
    * Adds a list of placed slice names to the matrix with the given status
    * 
    * @param sliceNames
    * @param status
    */
   public void addPlacedSliceNames(List<PlacedSliceName> sliceNames, Status status)
   {
      for (PlacedSliceName sliceName : sliceNames)
      {
         Entry entry = new Entry(status, sliceName.getSliceIndex());
         this.addEntry(sliceName.getSourceName(), sliceName.getStore(), entry);
      }
   }

   /**
    * Add a StoreBundle as a column in the matrix
    * 
    * @param bundle
    */
   public void addStoreSliceSet(StoreSliceSet bundle, Status status)
   {
      int col = this.getColumn(bundle.getSliceStore());

      for (DataSlice slice : bundle.getSlices())
      {
         SliceName name = slice.getSliceName();

         SourceName source = name.getSourceName();
         int row = this.getRow(source);

         Entry entry =
               new Entry(status, name.getSliceIndex(), slice.getTransactionId(), slice.getData());
         this.helperSetEntry(row, col, entry);
      }
   }

   /**
    * Add a SourceBundle as a row in the matrix
    * 
    * @param bundle
    */
   public void addSourceSliceSet(PlacedSourceSliceSet bundle, Status status)
   {
      int row = this.getRow(bundle.getName());

      for (PlacedDataSlice slice : bundle.getSlices())
      {
         SliceName name = slice.getSliceName();
         int col = this.getColumn(slice.getStore());

         Entry entry =
               new Entry(status, name.getSliceIndex(), slice.getTransactionId(), slice.getData());
         this.helperSetEntry(row, col, entry);
      }
   }

   /**
    * Get a StoreBundle of EXISTING slices for a given store in an arbitrary order
    * 
    * @param store
    * @return
    */
   public StoreSliceSet getStoreSliceSet(SliceStore store)
   {
      int col = this.getColumn(store);
      StoreSliceSet bundle = new StoreSliceSet(store);

      for (Map.Entry<SourceName, Integer> indexEntry : this.rowIndex.entrySet())
      {
         Entry entry = this.matrix[indexEntry.getValue()][col];

         if (entry != null && entry.getStatus() == Status.EXISTING)
         {
            SliceName name = new SliceName(indexEntry.getKey(), entry.getSliceIndex());
            bundle.addSlice(new DataSlice(name, entry.getTransactionId(), entry.getData()));
         }
      }

      return bundle;
   }

   /**
    * Adds the given revision to a source's blacklist
    * 
    * @param source
    * @param revision
    */
   public void blacklistRevision(SourceName source, long revision)
   {
      this.revisionBlacklist[this.getRow(source)].add(revision);
   }

   /**
    * Get a PlacedSourceSliceSet of EXISTING slices for a given source in an arbitrary order
    * 
    * @param source
    * @return
    */
   public PlacedSourceSliceSet getSourceSliceSet(SourceName source)
   {
      int row = this.getRow(source);
      PlacedSourceSliceSet bundle = new PlacedSourceSliceSet(source);

      for (Map.Entry<SliceStore, Integer> indexEntry : this.columnIndex.entrySet())
      {
         Entry entry = this.matrix[row][indexEntry.getValue()];

         if (entry != null && entry.getStatus() == Status.EXISTING)
         {
            PlacedSliceName name =
                  new PlacedSliceName(source, indexEntry.getKey(), entry.getSliceIndex());
            bundle.addSlice(new PlacedDataSlice(name, entry.getTransactionId(), entry.getData()));
         }
      }

      return bundle;
   }

   /**
    * Gets the slices of the highest non-blacklisted revision for a source. May
    * return null if no slices of any revision have been found or if all
    * known revisions have been blacklisted
    * 
    * @param source
    * @return Revision number and slices, or null if no slices found
    */
   public Tuple2<Long, PlacedSourceSliceSet> getHighRevision(SourceName source)
   {
      // FIXME: If this is a bottleneck it can be optimized
      Tuple2<Long, PlacedSourceSliceSet> slices = null;
      Set<Long> sourceBlacklist = this.revisionBlacklist[this.getRow(source)];

      Map<Long, PlacedSourceSliceSet> map = this.getRevisions(source);
      for (Map.Entry<Long, PlacedSourceSliceSet> tx : map.entrySet())
      {
         if (((slices == null) || (tx.getKey() > slices.getFirst()))
               && !sourceBlacklist.contains(tx.getKey()))
         {
            slices = new Tuple2<Long, PlacedSourceSliceSet>(tx.getKey(), tx.getValue());
         }
      }

      return slices;
   }

   /**
    * Helper method to return a map of {revision} => PlacedSourceSliceSet for all distinct
    * revisions represented in the matrix for the given source. This can be used to decide what
    * revision of a datasource to reconstruct
    * 
    * @param source
    * @return
    */
   public Map<Long, PlacedSourceSliceSet> getRevisions(SourceName source)
   {
      // Transaction id => Set of slices for source with this id
      Map<Long, PlacedSourceSliceSet> map = new HashMap<Long, PlacedSourceSliceSet>();

      PlacedSourceSliceSet set = this.getSourceSliceSet(source);
      for (PlacedDataSlice slice : set.getSlices())
      {
         PlacedSourceSliceSet txSet = map.get(slice.getTransactionId());
         if (txSet == null)
         {
            txSet = new PlacedSourceSliceSet(source);
            map.put(slice.getTransactionId(), txSet);
         }

         txSet.addSlice(slice);
      }

      return map;
   }

   /**
    * Returns the SliceNames for a given store and status in any order
    * 
    * @param store
    * @param status
    * @return
    */
   public List<SliceName> getStoreSliceNames(SliceStore store, Status status)
   {
      int col = this.getColumn(store);
      List<SliceName> sliceNames = new ArrayList<SliceName>(this.rows.size());

      for (Map.Entry<SourceName, Integer> indexEntry : this.rowIndex.entrySet())
      {
         Entry entry = this.matrix[indexEntry.getValue()][col];
         if (entry != null && entry.getStatus() == status)
         {
            sliceNames.add(new SliceName(indexEntry.getKey(), entry.getSliceIndex()));
         }
      }

      return sliceNames;
   }

   /**
    * Returns an aggregation of the errors for a given source
    * 
    * @param source
    * @return
    */
   public TypeAggregator<Exception> aggregateSourceErrors(SourceName source)
   {
      int row = this.getRow(source);
      TypeAggregator<Exception> aggregator = new TypeAggregator<Exception>();

      for (int col = 0; col < this.columnIndex.size(); ++col)
      {
         Entry entry = this.matrix[row][col];
         if (entry != null && entry.getStatus() == Status.ERROR)
         {
            aggregator.add(entry.getException());
         }
      }

      return aggregator;
   }

   /**
    * Returns the number of entries for a source with the given status
    * 
    * @param source
    * @param status
    * @return
    */
   public int getSourceStatusCount(SourceName source, Status status)
   {
      int row = this.getRow(source);
      Integer i = this.rowStatus[row].get(status);
      return i != null ? i : 0;
   }

   /**
    * Returns the number of entries for a store with the given status
    * 
    * @param store
    * @param status
    * @return
    */
   public int getStoreStatusCount(SliceStore store, Status status)
   {
      int col = this.getColumn(store);
      Integer i = this.columnStatus[col].get(status);
      return i != null ? i : 0;
   }

   /**
    * Return the names sources managed by this matrix in the order they were presented
    * 
    * @return
    */
   public List<SourceName> getSources()
   {
      return this.rows;
   }

   /**
    * Returns a list of sources with entries for the given store and status, in original order
    * 
    * @param store
    * @param status
    * @return
    */
   public List<SourceName> getSources(SliceStore store, Status status)
   {
      int col = this.getColumn(store);
      List<SourceName> sources = new ArrayList<SourceName>(this.rows.size());

      for (SourceName source : this.rows)
      {
         int row = this.getRow(source);
         Entry entry = this.matrix[row][col];
         if (entry != null && entry.getStatus() == status)
         {
            sources.add(source);
         }
      }

      return sources;
   }

   /**
    * Returns the stores managed by this matrix in the order they were presented
    * 
    * @return
    */
   public List<SliceStore> getStores()
   {
      return this.cols;
   }

   /**
    * Returns a list of stores for the given source and status, in original order
    * 
    * @param source
    * @param status
    * @return
    */
   public List<SliceStore> getStores(SourceName source, Status status)
   {
      int row = this.getRow(source);
      List<SliceStore> stores = new ArrayList<SliceStore>(this.cols.size());

      for (SliceStore store : this.cols)
      {
         int col = this.getColumn(store);
         Entry entry = this.matrix[row][col];
         if (entry != null && entry.getStatus() == status)
         {
            stores.add(store);
         }
      }

      return stores;
   }

   /**
    * Sets a new order for SliceStores in the matrix.  The matrix itself is not affected, but
    * future calls to getStores() and iterative operations will use the new ordering.
    * 
    * @param newStoreOrder
    */
   public void setStoreOrder(List<SliceStore> newStoreOrder)
   {
      // We could do more thorough & expensive checking here if desired
      assert newStoreOrder.size() == this.cols.size();
      this.cols = newStoreOrder;
   }

   /**
    * Helper method to get a the matrix row of a source
    * 
    * @param sliceName
    * @return
    */
   private int getRow(SourceName source)
   {
      Integer row = this.rowIndex.get(source);
      if (row == null)
      {
         throw new IllegalArgumentException("Source not found: " + source);
      }
      return row;
   }

   /**
    * Helper method to get the matrix column of a store
    * 
    * @param store
    * @return
    */
   private int getColumn(SliceStore store)
   {
      Integer col = this.columnIndex.get(store);
      if (col == null)
      {
         throw new IllegalArgumentException("Store not found: " + store);
      }
      return col;
   }

   /**
    * Returns a tabular view of the current state of this scorecard.  This is a relatively 
    * expensive operation
    */
   @Override
   public String toString()
   {
      // Column headers
      List<SliceStore> stores = this.getStores();
      String[] columnHeaders = new String[stores.size() + 1];
      columnHeaders[0] = ""; // First column is row headers
      for (int i = 0; i < stores.size(); ++i)
      {
         columnHeaders[i + 1] = stores.get(i).getIdentification();
      }

      TablePrinter table = new TablePrinter(columnHeaders);

      // Rows
      for (SourceName source : this.getSources())
      {
         String[] row = new String[stores.size() + 1];

         // Row header
         row[0] = source.getName();

         // Slices
         for (int i = 0; i < stores.size(); ++i)
         {
            String cell = "";
            Entry entry = this.getEntry(source, stores.get(i));
            if (entry != null)
            {
               cell = entry.toString();
            }
            row[i + 1] = cell;
         }

         table.add(row);
      }

      ByteArrayOutputStream out = new ByteArrayOutputStream();
      table.print(new PrintStream(out));
      return out.toString();
   }

   /**
    * Create a deep copy of this object
    */
   @Override
   public Object clone()
   {
      OperationScorecard clone = new OperationScorecard(this.rows, this.cols);

      for (int row = 0; row < this.rowIndex.size(); ++row)
      {
         for (int col = 0; col < this.columnIndex.size(); ++col)
         {
            // We don't currently clone entries
            Entry entry = this.matrix[row][col];
            clone.helperSetEntry(row, col, entry);
         }

         clone.revisionBlacklist[row].addAll(this.revisionBlacklist[row]);
      }

      return clone;
   }

   /**
    * Helper container to hold a slice entry in the transformation matrix
    */
   public static class Entry
   {
      // Status of this entry
      private Status status;

      // Exception, if the status is ERROR
      private Exception exception;

      // Index of slice (1-width)
      private int sliceIndex;

      // Transaction Id
      private long transactionId;

      // Slice data
      private byte[] data;

      /**
       * Constructor primarily useful for EXPECTED and IGNORED data
       * 
       * @param status
       */
      public Entry(Status status, int sliceIndex)
      {
         this(status, sliceIndex, -1, null);
      }

      /**
       * Constructor primarily useful for EXISTING
       * 
       * @param status
       * @param sliceIndex
       * @param transactionId
       * @param data
       */
      public Entry(Status status, int sliceIndex, long transactionId, byte[] data)
      {
         this.status = status;
         this.sliceIndex = sliceIndex;
         this.transactionId = transactionId;
         this.data = data;

         this.exception = null;
      }

      /**
       * Unlink any data payload to allow it to be garbage collected
       */
      public void clearData()
      {
         this.data = null;
      }

      /**
       * Set status
       * @param status
       */
      public void setStatus(Status status)
      {
         this.status = status;
      }

      public Status getStatus()
      {
         return this.status;
      }

      public int getSliceIndex()
      {
         return this.sliceIndex;
      }

      public long getTransactionId()
      {
         return this.transactionId;
      }

      public byte[] getData()
      {
         return this.data;
      }

      public void setException(Exception ex)
      {
         this.exception = ex;
      }

      public Exception getException()
      {
         return this.exception;
      }

      @Override
      public String toString()
      {
         StringBuilder sb = new StringBuilder();
         sb.append(this.sliceIndex);
         sb.append(": ");
         sb.append(this.status);

         if (this.exception != null)
         {
            sb.append(" (");
            sb.append(this.exception);
            sb.append(")");
         }

         if (this.getTransactionId() > -1)
         {
            sb.append(", txid=");
            sb.append(this.getTransactionId());
         }

         return sb.toString();
      }
   }
}
