//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Aug 18, 2007
//---------------------

package org.cleversafe.layer.grid.smartcontroller;

import java.util.Map;

import org.apache.log4j.Logger;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.ControllerTransactionException;
import org.cleversafe.layer.grid.exceptions.GridLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.layer.slicestore.exceptions.IllegalSourceNameException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreIOException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreNotFoundException;
import org.cleversafe.layer.slicestore.exceptions.SliceStoreTransactionException;
import org.cleversafe.layer.slicestore.exceptions.WriteTransactionDeadlockException;
import org.cleversafe.util.Tuple2;
import org.cleversafe.util.TypeAggregator;
import org.cleversafe.util.UnimodStateMachine;
import org.cleversafe.vault.Vault;
import org.cleversafe.vault.exceptions.DataTransformationException;

import com.evelopers.common.exception.SystemException;
import com.evelopers.unimod.runtime.context.StateMachineContext;

public abstract class GridControllerStateMachine extends UnimodStateMachine
{
   private static Logger _logger = Logger.getLogger(GridControllerStateMachine.class);

   /************************************************************************************************
    * Events
    ***********************************************************************************************/

   /** @unimod.event.descr Slice store task successful */
   public static final String EV_SLICE_STORE_TASK_SUCCESS = "slice_store_task_success";

   /** @unimod.event.descr Slice store task exception */
   public static final String EV_SLICE_STORE_TASK_EXCEPTION = "slice_store_task_exception";

   /************************************************************************************************
    * World state
    ***********************************************************************************************/

   // Vault for this operation
   protected Vault vault;

   // Data transformation matrix
   protected OperationScorecard matrix;

   protected GridControllerStateMachine()
   {
      // Initial world state
      this.vault = null;
   }

   /**
    * Set this operation's vault (mandatory)
    * 
    * @param vault
    */
   public void setVault(Vault vault)
   {
      this.vault = vault;
   }

   /**
    * Set a matrix of slice data to use to manage this operation
    * 
    * @param matrix
    */
   public void setMatrix(OperationScorecard matrix)
   {
      this.matrix = matrix;
   }

   protected static class SliceStoreTaskSuccessEvent extends UnimodEvent
   {
      private static final long serialVersionUID = -6411262645666091354L;
      SliceStore store;

      public SliceStoreTaskSuccessEvent(SliceStore store)
      {
         super(EV_SLICE_STORE_TASK_SUCCESS);
         this.store = store;
      }

      public SliceStore getSliceStore()
      {
         return this.store;
      }
   }

   protected static class SliceStoreTaskExceptionEvent extends ExceptionEvent
   {
      private static final long serialVersionUID = -4382774993212832806L;
      SliceStore store;

      public SliceStoreTaskExceptionEvent(SliceStore store, Exception exception)
      {
         super(EV_SLICE_STORE_TASK_EXCEPTION, exception);
         this.store = store;
      }

      public SliceStore getSliceStore()
      {
         return this.store;
      }
   }

   /************************************************************************************************
    * Permissible world operations
    ***********************************************************************************************/

   /**
    * @unimod.action.descr Task completed with an exception
    */
   public void sliceStoreTaskException(StateMachineContext context)
   {
      UnimodStateMachineContext unimodContext = (UnimodStateMachineContext) context;
      SliceStoreTaskExceptionEvent event =
            (SliceStoreTaskExceptionEvent) unimodContext.getCurrentEvent();

      // Set error status for all of this store's entries in the matrix
      // FIXME: For multiple writes per transaction, this should only error out the entries
      //        for sources affected by this particular error
      this.matrix.setStoreStatus(event.getSliceStore(), OperationScorecard.Status.ERROR,
            event.getException());
   }

   /**
    * @unimod.action.descr Is the current concurrent operation finished?
    */
   public void checkOperationCompleted(StateMachineContext context)
   {
      // Check all stores in the matrix for pending operations
      for (SliceStore store : this.matrix.getStores())
      {
         if (this.matrix.getStoreStatusCount(store, OperationScorecard.Status.PENDING) > 0)
         {
            // At least one store has pending operations
            return;
         }
      }

      // Fire immediately
      this.fireEvent(new UnimodEvent(EV_OPERATION_COMPLETED), true);
   }

   /**
    * @unimod.action.descr Is the matrix in an acceptable state?
    */
   public void checkOperationStatusOK(StateMachineContext context)
   {
      InformationDispersalCodec ida = this.vault.getInformationDispersalCodec();
      int allowedErrors = ida.getNumSlices() - ida.getThreshold();

      for (SourceName source : this.matrix.getSources())
      {
         if (this.matrix.getSourceStatusCount(source, OperationScorecard.Status.ERROR) > allowedErrors)
         {
            // Intractable, send immediate response
            this.fireEvent(new UnimodEvent(EV_NO), true);
            return;
         }
      }

      // Outage status is still technically acceptable
      this.fireEvent(new UnimodEvent(EV_YES), true);
   }

   /**
    * Helper to determine what exception to throw based on the grid state and error threshold
    */
   protected void determineExceptionHelper(String operationName, int numFatalErrors)
         throws GridLayerException
   {
      // Find the first datasource with a fatal number of errors. This source will serve as a
      // reasonable indicator of the general grid state
      SourceName canonicalSource = null;
      for (SourceName source : this.matrix.getSources())
      {
         if (this.matrix.getSourceStatusCount(source, OperationScorecard.Status.ERROR) > numFatalErrors)
         {
            canonicalSource = source;
            break;
         }
      }

      // Grid state cannot be determined, probably due to conflicting slice revisions
      if (canonicalSource == null)
      {
         String errorString =
               String.format("%s: Inconclusive grid error state (likely no revision plurality)",
                     operationName);

         if (_logger.isDebugEnabled())
            _logger.debug(errorString + "\nOperation status:\n" + this.matrix.toString());

         throw new ControllerGridStateUnknownException(errorString);
      }

      // Find the most common exception
      TypeAggregator<Exception> exceptions = this.matrix.aggregateSourceErrors(canonicalSource);
      Map.Entry<Class<? extends Exception>, Tuple2<Integer, Exception>> max =
            exceptions.getMaximum();

      // See if the most common exception is the clear cause of failure
      if (max.getValue().getFirst() > numFatalErrors)
      {
         // An exception can clearly be determined
         Class<? extends Exception> c = max.getKey();
         Exception example = max.getValue().getSecond();

         // Marshal to an expected grid exception
         if (c == IllegalSourceNameException.class)
         {
            throw new ControllerIllegalSourceNameException(operationName + ": Illegal source name",
                  example);
         }
         else if (c == SliceStoreIOException.class)
         {
            throw new ControllerIOException(operationName + ": IO exception", example);
         }
         else if (c == SliceStoreTransactionException.class
               || c == WriteTransactionDeadlockException.class)
         {
            throw new ControllerTransactionException(operationName + ": Transaction error", example);
         }
         else if (c == SliceStoreNotFoundException.class)
         {
            throw new ControllerStoresNotFoundException(
                  operationName
                        + ": More than an allowable number of stores could not be found.  Check & repair this vault.",
                  example);
         }
         else if (c == DataTransformationException.class)
         {
            throw new ControllerDataTransformationException(operationName
                  + ": Transformation/integrity error", example);
         }
         else
         {
            throw new RuntimeException(operationName + ": Unexpected exception type", example);
         }
      }
      else
      {
         // The SliceStores do not agree, probably due in part to conflicting slice revisions
         String errorString =
               String.format("%s: Inconclusive grid error state (multiple errors)", operationName);

         if (_logger.isDebugEnabled())
            _logger.debug(errorString + "\nOperation status:\n" + this.matrix.toString());

         throw new ControllerGridStateUnknownException(errorString);
      }
   }

   /**
    * Run the state machine and marshal its exceptions
    */
   public Object runStateMachine(UnimodStateMachineContext context) throws GridLayerException
   {
      try
      {
         return super.runStateMachine(context);
      }
      catch (final GridLayerException ex)
      {
         throw ex;
      }
      catch (Throwable ex)
      {
         if (ex instanceof SystemException)
         {
            ex = ((SystemException) ex).getParentException();
         }

         _logger.error("Only GridLayerExceptions are expected", ex);
         throw new RuntimeException("Only GridLayerExceptions are expected", ex);
      }
   }
}
