//
// Cleversafe open-source code header - Version 1.2 - February 15, 2008
//
// Cleversafe Dispersed Storage(TM) is software for secure, private and
// reliable storage of the world's data using information dispersal.
//
// Copyright (C) 2005-2008 Cleversafe, Inc.
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
// USA.
//
// Contact Information: Cleversafe, 224 North Desplaines Street, Suite 500 
// Chicago IL 60661
// email licensing@cleversafe.org
//
// END-OF-HEADER
//-----------------------
// @author: gdhuse
//
// Date: Sep 11, 2007
//---------------------

package org.cleversafe.layer.grid.smartcontroller;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.cleversafe.ida.InformationDispersalCodec;
import org.cleversafe.layer.grid.DataSlice;
import org.cleversafe.layer.grid.GridTransaction;
import org.cleversafe.layer.grid.OperationScorecard;
import org.cleversafe.layer.grid.PlacedSourceSliceSet;
import org.cleversafe.layer.grid.SliceName;
import org.cleversafe.layer.grid.SourceName;
import org.cleversafe.layer.grid.exceptions.ControllerDataTransformationException;
import org.cleversafe.layer.grid.exceptions.ControllerGridStateUnknownException;
import org.cleversafe.layer.grid.exceptions.ControllerIOException;
import org.cleversafe.layer.grid.exceptions.ControllerIllegalSourceNameException;
import org.cleversafe.layer.grid.exceptions.ControllerStoresNotFoundException;
import org.cleversafe.layer.grid.exceptions.GridLayerException;
import org.cleversafe.layer.slicestore.SliceStore;
import org.cleversafe.util.Tuple2;
import org.cleversafe.vault.exceptions.DataTransformationException;

import com.evelopers.unimod.runtime.context.StateMachineContext;

public class ReadStateMachine extends GridControllerStateMachine
{
   private static Logger _logger = Logger.getLogger(ReadStateMachine.class);

   // Read policy:
   // POLICY_PENDING_READ_FACTOR * (readPolicy - largestTxSize) + POLICY_PENDING_READ_ADDEND
   public static int POLICY_PENDING_READ_FACTOR = 1;
   public static int POLICY_PENDING_READ_ADDEND = 0;

   // A possibly ordered list of SliceStores
   protected List<SliceStore> sliceStores;

   /************************************************************************************************
    * Events
    ***********************************************************************************************/

   /** @unimod.event.descr Read successful */
   public static final String EV_READ_SUCCESS = "read_success";

   /** @unimod.event.descr Policy is impossible to satisfied */
   public static final String EV_IMPOSSIBLE_TO_SATISFY = "impossible_to_satisfy";

   protected static class ReadSuccessEvent extends UnimodEvent
   {
      private static final long serialVersionUID = -2498653469696000394L;

      private SliceStore store;
      private List<DataSlice> slices;

      public ReadSuccessEvent(SliceStore store, List<DataSlice> slices)
      {
         super(EV_READ_SUCCESS);
         this.store = store;
         this.slices = slices;
      }

      public SliceStore getSliceStore()
      {
         return this.store;
      }

      public List<DataSlice> getSlices()
      {
         return this.slices;
      }
   }

   /************************************************************************************************
    * World state
    ***********************************************************************************************/

   public ReadStateMachine()
   {
      super();
      this.sliceStores = new LinkedList<SliceStore>();
   }

   /**
    * Sets the stores for this operation. The provided collection will not be altered
    * 
    * @param stores
    */
   public void setSliceStores(final Collection<SliceStore> stores)
   {
      // We maintain our own container so we can alter it
      for (SliceStore store : stores)
      {
         this.sliceStores.add(store);
      }
   }

   /************************************************************************************************
    * Permissible world operations
    ***********************************************************************************************/

   /**
    * Async read task
    */
   private class ReadTask extends ConcurrentFunction
   {
      SliceStore store;
      List<SliceName> sliceNames;

      public ReadTask(SliceStore store, List<SliceName> sliceNames)
      {
         this.store = store;
         this.sliceNames = sliceNames;
      }

      public void run()
      {
         try
         {
            List<DataSlice> slices = this.store.read(this.sliceNames);

            // Read was successful
            this.fireEvent(new ReadSuccessEvent(store, slices));
         }
         catch (final Exception ex)
         {
            if (_logger.isDebugEnabled())
               _logger.debug("Read error '" + ex + "' from store: " + store);

            // Fire exception event
            this.fireEvent(new SliceStoreTaskExceptionEvent(store, ex));
         }
      }
   }

   /**
    * @unimod.action.descr Begins zero or more read operations according to policy
    */
   public void beginReadOperations(StateMachineContext context)
   {
      // Attempt to keep total of |Pending| = c1*(max(Threshold, ReadPolicy)-|Existing|) + c2
      // outstanding pending read operations per source. The actual number may be limited
      // by the number of slices remaining to be read from the grid.
      int readPolicy = Math.max(this.vault.getThreshold(), this.vault.getReadThreshold());

      // Initiate reads to satisfy pending criteria for each source in a greedy manner
      // TODO: Consider an optimized way of choosing stores that satisfy multiple source
      // requirements in fewer reads
      for (SourceName source : this.matrix.getSources())
      {
         Tuple2<Long, PlacedSourceSliceSet> highRevision = this.matrix.getHighRevision(source);
         int largestTxSize = highRevision != null ? highRevision.getSecond().size() : 0;

         int currentlyPending =
               this.matrix.getSourceStatusCount(source, OperationScorecard.Status.PENDING);
         int desiredPending =
               POLICY_PENDING_READ_FACTOR * (readPolicy - largestTxSize)
                     + POLICY_PENDING_READ_ADDEND;

         if (currentlyPending < desiredPending)
         {
            List<SliceStore> stores =
                  this.matrix.getStores(source, OperationScorecard.Status.EXPECTED);

            while (currentlyPending < desiredPending && !stores.isEmpty())
            {
               SliceStore nextStore = stores.remove(0);
               List<SliceName> storeSlices =
                     this.matrix.getStoreSliceNames(nextStore, OperationScorecard.Status.EXPECTED);
               assert !storeSlices.isEmpty() : "No slices to write?";

               // Mark all affected slices pending
               for (SliceName sliceName : storeSlices)
               {
                  this.matrix.setStatus(sliceName.getSourceName(), nextStore,
                        OperationScorecard.Status.PENDING);
               }

               // Begin read
               this.runTask(new ReadTask(nextStore, storeSlices));
               ++currentlyPending;
            }
         }
      }

      this.fireEvent(new UnimodEvent(EV_OK), true);
   }

   /**
    * @unimod.action.descr Process new slices and attempt to assemble datasources
    */
   public void decodeSlices(StateMachineContext context)
   {
      UnimodStateMachineContext unimodContext = (UnimodStateMachineContext) context;
      ReadSuccessEvent event = (ReadSuccessEvent) unimodContext.getCurrentEvent();

      SliceStore store = event.getSliceStore();
      for (DataSlice slice : event.getSlices())
      {
         SliceName sliceName = slice.getSliceName();
         SourceName sourceName = sliceName.getSourceName();

         if (slice.getTransactionId() != GridTransaction.NOT_FOUND_TRANSACTION_ID)
         {
            // This slice was found on the store
            try
            {
               // Decode
               slice = this.vault.decodeSlice(slice);

               // Slice decoded successfully, update the matrix
               OperationScorecard.Entry entry =
                     new OperationScorecard.Entry(OperationScorecard.Status.EXISTING,
                           sliceName.getSliceIndex(), slice.getTransactionId(), slice.getData());
               this.matrix.addEntry(sourceName, store, entry);
            }
            catch (DataTransformationException ex)
            {
               // Note an error for this slice
               _logger.debug("Decode failure, corrupt slice: " + slice.toString());
               this.matrix.setStatus(sourceName, store, OperationScorecard.Status.ERROR, ex);
            }
         }
         else
         {
            // This source was not found on the store, add to the matrix with null data
            OperationScorecard.Entry entry =
                  new OperationScorecard.Entry(OperationScorecard.Status.EXISTING,
                        sliceName.getSliceIndex(), slice.getTransactionId(), null);
            this.matrix.addEntry(sourceName, store, entry);
         }
      }

      this.fireEvent(new UnimodEvent(EV_OK), true);
   }

   /**
    * @unimod.action.descr Checks whether the read policy is satisfied and fires EV_YES, EV_NO,
    *                      or EV_IMPOSSIBLE_TO_SATISFY
    */
   public void checkReadPolicySatisfied(StateMachineContext context)
   {
      int readPolicy = Math.max(this.vault.getThreshold(), this.vault.getReadThreshold());
      InformationDispersalCodec ida = this.vault.getInformationDispersalCodec();

      // Each source in the read is evaluated independently
      //
      // DEVELOPER'S NOTE: This logic can be tightened regarding corrupt slices.  Corrupt slices
      //    should count towards the read policy and should not count as an unknown revision 
      //    when determining whether revision fallback is legal.  The result is that in the 
      //    rare case when corrupt slices are present some reads will fail that could possibly
      //    succeed.  See US1283 for more information.
      for (SourceName source : this.matrix.getSources())
      {
         int numErrors = this.matrix.getSourceStatusCount(source, OperationScorecard.Status.ERROR);
         int numSuccessful =
               this.matrix.getSourceStatusCount(source, OperationScorecard.Status.EXISTING);
         int numRemaining = ida.getNumSlices() - numSuccessful - numErrors;

         // Has the read policy been met?
         if (numSuccessful < readPolicy)
         {
            // Not yet, but is it still possible?
            if (numRemaining < readPolicy - numSuccessful)
            {
               // Impossible to satisfy read policy for this source 
               if (_logger.isDebugEnabled())
                  _logger.debug(String.format("Read policy cannot be met for source '%s'",
                        source.toString()));

               this.fireEvent(new UnimodEvent(EV_IMPOSSIBLE_TO_SATISFY), true);
               return;
            }
            else
            {
               // Still possible to satisfy read policy, but not ready yet (Case 2)
               this.fireEvent(new UnimodEvent(EV_NO), true);
               return;
            }
         }

         // Read policy has been satisfied, see if any revision is ready to restore from high to low
         while (true)
         {
            // Find status of the current highest read revision
            Tuple2<Long, PlacedSourceSliceSet> highRevision = this.matrix.getHighRevision(source);
            int highRevisionSize = (highRevision != null) ? highRevision.getSecond().size() : 0;

            // Is this revision ready to restore?
            if (highRevisionSize >= ida.getThreshold())
            {
               // This source is ready to restore
               break;
            }
            // Not yet, but is it still possible with the current vault state?
            else if (highRevisionSize + numRemaining >= ida.getThreshold())
            {
               // Still possible to restore this revision, but not ready yet
               this.fireEvent(new UnimodEvent(EV_NO), true);
               return;
            }
            // This revision cannot be restored with the current vault state.  Will it ever be
            // possible to restore this revision?
            else if (highRevisionSize + numRemaining + numErrors >= ida.getThreshold())
            {
               // This revision cannot be restored now, but it is possible that it can be
               // restored in the future.  Temporary failure.
               if (_logger.isDebugEnabled())
                  _logger.debug(String.format("Temporary read failure for source '%s'",
                        source.toString()));

               this.fireEvent(new UnimodEvent(EV_IMPOSSIBLE_TO_SATISFY), true);
               return;
            }
            // This revision can never be restored.  Is it possible to fall back to a past
            // revision?
            else if (highRevision != null)
            {
               // This revision can never be restored, but there may be a lower revision with more
               // slices.  Blacklist this revision and try the next lowest
               if (_logger.isDebugEnabled())
                  _logger.debug(String.format(
                        "Revision %d of source '%s' cannot be restored, falling back to next lowest revision",
                        highRevision.getFirst(), source.toString()));

               this.matrix.blacklistRevision(source, highRevision.getFirst());
               continue;
            }
            // No revisions of this source can ever be restored.  This source is corrupt.
            else
            {
               // This source is corrupt no revision can never be restored
               _logger.warn(String.format("Source corrupt: '%s'", source.toString()));

               this.fireEvent(new UnimodEvent(EV_IMPOSSIBLE_TO_SATISFY), true);
               return;
            }
         }
      }

      // All sources satisfy policy
      this.fireEvent(new UnimodEvent(EV_YES), true);
   }

   /**
    * @unimod.action.descr Determine what exception to throw based on the grid state and error threshold
    */
   public void determineException(StateMachineContext context) throws GridLayerException
   {
      InformationDispersalCodec ida = this.vault.getInformationDispersalCodec();
      int maxErrors =
            ida.getNumSlices() - Math.max(this.vault.getReadThreshold(), ida.getThreshold());
      super.determineExceptionHelper("Read", maxErrors);
   }

   /************************************************************************************************
    * External interface
    ***********************************************************************************************/

   /**
    * Multi-datasource read
    */
   public void run(OperationScorecard data) throws ControllerIOException,
         ControllerIllegalSourceNameException, ControllerGridStateUnknownException,
         ControllerDataTransformationException, ControllerStoresNotFoundException
   {
      try
      {
         this.setMatrix(data);
         super.runStateMachine(new UnimodStateMachineContext());
      }
      catch (final GridLayerException ex)
      {
         // Marshal to known exceptions
         if (ex instanceof ControllerIOException)
         {
            throw (ControllerIOException) ex;
         }
         else if (ex instanceof ControllerIllegalSourceNameException)
         {
            throw (ControllerIllegalSourceNameException) ex;
         }
         else if (ex instanceof ControllerGridStateUnknownException)
         {
            throw (ControllerGridStateUnknownException) ex;
         }
         else if (ex instanceof ControllerDataTransformationException)
         {
            throw (ControllerDataTransformationException) ex;
         }
         else if (ex instanceof ControllerStoresNotFoundException)
         {
            throw (ControllerStoresNotFoundException) ex;
         }
         else
         {
            throw new RuntimeException("Unexpected exception type", ex);
         }
      }
   }
}
